#include <ft2build.h>
#include FT_FREETYPE_H
#define STB_IMAGE_WRITE_IMPLEMENTATION
#include "stb_image_write.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define FIRST_CHAR 32
#define LAST_CHAR 127
#define COLS 11
#define ROWS ((LAST_CHAR - FIRST_CHAR + COLS) / COLS)

void save_png(int width, int height, unsigned char *image, const char *filename) {
    stbi_write_png(filename, width, height, 4, image, width * 4);
    printf("Saved %s (%dx%d)\n", filename, width, height);
}

void save_tga(int width, int height, unsigned char *image, const char *filename) {
    FILE *file = fopen(filename, "wb");
    if (!file) {
        printf("Error: Failed to create TGA file.\n");
        return;
    }

    // TGA header
    unsigned char header[18] = {0};
    header[2] = 10; // RLE compressed true-color image
    header[12] = width & 0xFF;
    header[13] = (width >> 8) & 0xFF;
    header[14] = height & 0xFF;
    header[15] = (height >> 8) & 0xFF;
    header[16] = 32; // Bits per pixel
    header[17] = 0x00; // Bottom-left origin

    fwrite(header, sizeof(header), 1, file);

    // Write image data with RLE compression
    for (int y = height - 1; y >= 0; y--) {
        int row_start = y * width * 4;
        int x = 0;
        while (x < width) {
            int i = row_start + x * 4;

            // Check if we can use RLE (run of identical pixels)
            int run_length = 1;
            while (x + run_length < width && run_length < 128) {
                int j = row_start + (x + run_length) * 4;
                if (memcmp(&image[i], &image[j], 4) != 0)
                    break;
                run_length++;
            }

            if (run_length > 1) {
                unsigned char packet_header = 0x80 | (run_length - 1);
                fwrite(&packet_header, 1, 1, file);
                fwrite(&image[i], 4, 1, file);
                x += run_length;
            } else {
                int raw_start = x;
                run_length = 1;
                while (x + run_length < width && run_length < 128) {
                    int a = row_start + (x + run_length - 1) * 4;
                    int b = row_start + (x + run_length) * 4;
                    if (memcmp(&image[a], &image[b], 4) == 0)
                        break;
                    run_length++;
                }

                unsigned char packet_header = run_length - 1;
                fwrite(&packet_header, 1, 1, file);
                fwrite(&image[row_start + raw_start * 4], run_length * 4, 1, file);
                x += run_length;
            }
        }
    }

    fclose(file);
    printf("Saved %s (RLE compressed, bottom-left origin, %dx%d)\n", filename, width, height);
}

// Function for input handling (handles empty input cases)
void get_input(const char *prompt, char *buffer, size_t size, const char *default_value) {
    printf("%s", prompt);
    fgets(buffer, size, stdin);

    // Remove newline character if present
    buffer[strcspn(buffer, "\n")] = 0;

    // If input is empty, use default value
    if (strlen(buffer) == 0) {
        strcpy(buffer, default_value);
    }

    // **Ensure input buffer is cleared** before the next prompt
    fflush(stdin);
}

int main() {
    char fontFile[256], outputFormat[4];
    int scale_factor, r, g, b, letter_size, r_shadow, g_shadow, b_shadow;

    // Get font filename
    get_input("Enter font filename (default: FreeMono.ttf): ", fontFile, sizeof(fontFile), "FreeMono.ttf");

    FT_Library ft;
    if (FT_Init_FreeType(&ft)) {
        printf("Error: Could not initialize FreeType.\n");
        return 1;
    }

    FT_Face face;
    if (FT_New_Face(ft, fontFile, 0, &face)) {
        printf("Error: Could not load font file: %s\n", fontFile);
        return 1;
    }

    // Get output format
    get_input("Select output format (default: png) [png/tga]: ", outputFormat, sizeof(outputFormat), "png");

    // Validate format input
    if (!(strcmp(outputFormat, "png") == 0 || strcmp(outputFormat, "tga") == 0)) {
        printf("Error: Invalid format. Please enter 'png' or 'tga'.\n");
        return 1;
    }

    // Explain scale factor options
    printf("\nScale Factor Guide:\n");
    printf("  1 = 256x256 resolution\n");
    printf("  2 = 512x512 resolution\n");
    printf("  3 = 768x768 resolution\n");
    printf("  4 = 1024x1024 resolution\n");

    char scaleInput[10];
    get_input("Enter scale factor (default: 1, max: 4): ", scaleInput, sizeof(scaleInput), "1");
    scale_factor = atoi(scaleInput);
    if (scale_factor < 1) scale_factor = 1;
    if (scale_factor > 4) scale_factor = 4; // Ensure max is 4

    // Get letter size while keeping the fixed box (23x23)
    char letterSizeInput[10];
    get_input("Enter letter size (default: 20, min: 10, max: 23): ", letterSizeInput, sizeof(letterSizeInput), "20");
    letter_size = atoi(letterSizeInput);
    if (letter_size < 10) letter_size = 10; // Minimum size
    if (letter_size > 23) letter_size = 23; // Maximum size

    // Get text color
    char colorInput[20];
    get_input("Enter text color RGB (default: 255 255 255): ", colorInput, sizeof(colorInput), "255 255 255");
    sscanf(colorInput, "%d %d %d", &r, &g, &b);

    printf("\nShadow Options:\n");
    printf("  0 = No shadow\n");
    printf("  1 = Simple shadow (offset +1, +1)\n");
    printf("  2 = Full shadow (outline around glyph), Kingpin use this one for default (best to use high scale).\n");

    char shadowInput[10];
    get_input("Select shadow option (default: 0): ", shadowInput, sizeof(shadowInput), "0");
    int shadow_mode = atoi(shadowInput);
    if (shadow_mode < 0 || shadow_mode > 2) shadow_mode = 0; // Ensure valid input

    if (shadow_mode)
    {
        // Get Shadow text color
        char ShadowcolorInput[20];
        get_input("Enter text shadow color RGB (default: 0 0 0): ", ShadowcolorInput, sizeof(ShadowcolorInput), "0 0 0");
        sscanf(ShadowcolorInput, "%d %d %d", &r_shadow, &g_shadow, &b_shadow);
    }

    // Fixed filename
    char outputFile[256];
    if (strcmp(outputFormat, "png") == 0) {
        strcpy(outputFile, "conchars_high_11x11.png");
    } else {
        strcpy(outputFile, "conchars_high_11x11.tga");
    }

    // Image dimensions based on scale factor
    int TILE_W = 23 * scale_factor;
    int TILE_H = 23 * scale_factor;
    int IMAGE_WIDTH = 256 * scale_factor;
    int IMAGE_HEIGHT = 256 * scale_factor;

    unsigned char *image = malloc(IMAGE_WIDTH * IMAGE_HEIGHT * 4);
    memset(image, 0, IMAGE_WIDTH * IMAGE_HEIGHT * 4);

    FT_Set_Pixel_Sizes(face, 0, letter_size * scale_factor); // Scale letter while keeping the box fixed

    for (int ch = FIRST_CHAR; ch <= LAST_CHAR; ch++) {
        int actual_char = ch;
        if (ch == 126) actual_char = 34; //FREDZ Kingpin Replace '~' with '"'
        if (ch == 127) actual_char = 60; //FREDZ Kingpin Replace '' with '<'

        if (FT_Load_Char(face, actual_char, FT_LOAD_RENDER | FT_LOAD_TARGET_LIGHT))
            continue;

        FT_Bitmap *bmp = &face->glyph->bitmap;
        int x = (ch - FIRST_CHAR) % COLS;
        int y = (ch - FIRST_CHAR) / COLS;
        int tile_x = x * TILE_W;
        int tile_y = y * TILE_H;

        int x_offset = face->glyph->bitmap_left + (scale_factor * 2);
//      int y_offset = TILE_H - face->glyph->bitmap_top - (scale_factor * 2) -4;//FREDZ - 4 abit offset
        int y_offset = TILE_H - face->glyph->bitmap_top - (scale_factor * 2) - (scale_factor * 4);//FREDZ -offset 4 *scale


        for (int row = 0; row < bmp->rows; row++) {
            for (int col = 0; col < bmp->width; col++) {
                int px = tile_x + x_offset + col;
                int py = tile_y + y_offset + row;
                if (px >= 0 && px < IMAGE_WIDTH && py >= 0 && py < IMAGE_HEIGHT)
                {
                    unsigned char alpha = bmp->buffer[row * bmp->pitch + col];

                    if (shadow_mode == 1) {
                        // Simple shadow (offset slightly)
                        int shadow_x = px + 1;
                        int shadow_y = py + 1;
                        if (shadow_x >= 0 && shadow_x < IMAGE_WIDTH && shadow_y >= 0 && shadow_y < IMAGE_HEIGHT) {
                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 0] = b_shadow;
                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 1] = g_shadow;
                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 2] = r_shadow;
                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3] = alpha * 0.8; // Soft shadow
                        }
                    }
                    else if (shadow_mode == 2) {
                        for (int dy = -2; dy <= 2; dy++) {
                            for (int dx = -2; dx <= 2; dx++) {
                                if (dx == 0 && dy == 0) continue;
                                int shadow_x = px + dx;
                                int shadow_y = py + dy;
                                if (shadow_x >= 0 && shadow_x < IMAGE_WIDTH && shadow_y >= 0 && shadow_y < IMAGE_HEIGHT) {
                                    float distance = dx * dx + dy * dy;
                                    float strength = 1.0f - (distance / 8.0f); // distance 1: strong, 8: weak
                                    if (strength > 0) {
                                        unsigned char existing_alpha = image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3];
                                        if (existing_alpha < alpha * strength) {
                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 0] = r_shadow;
                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 1] = g_shadow;
                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 2] = b_shadow;
                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3] = alpha * strength;
                                        }
                                    }
                                }
                            }
                        }
                    }

                    image[(py * IMAGE_WIDTH + px) * 4 + 0] = b; // Blue
                    image[(py * IMAGE_WIDTH + px) * 4 + 1] = g; // Green
                    image[(py * IMAGE_WIDTH + px) * 4 + 2] = r; // Red
                    image[(py * IMAGE_WIDTH + px) * 4 + 3] = alpha; // Alpha
                }
            }
        }
    }

    // Save based on output format
    if (strcmp(outputFormat, "png") == 0) {
        save_png(IMAGE_WIDTH, IMAGE_HEIGHT, image, outputFile);
    } else {
        save_tga(IMAGE_WIDTH, IMAGE_HEIGHT, image, outputFile);
    }

    free(image);
    FT_Done_Face(face);
    FT_Done_FreeType(ft);
    return 0;
}
