
#include "bsp5.h"

#define __STDC__ 1 	// Kludge
#include "io.h"
#include "setjmp.h"

#define BSPLIMIT 4088	// For unknown reasons (4096 - 8)

void	 BuildName (char Name[], char Orig[], char Ext[]);
void	 DoLightGraph (int Light[], qboolean FixedScale);
void	 DoLightStat (qboolean Graph, qboolean FixedScale);
void	 DoPrt (char FullPath[]);
void	 MakeNode (int nodenum);
qboolean DoCutVisLight (void);
qboolean DoEnts (char FullPath[], qboolean CopyEnts);
qboolean DoLimit (qboolean Limit);
qboolean DoTex (qboolean Fixup);
qboolean FLimit (char Obj1[], char Obj2[], int ObjNo, float *PValue, qboolean Limit, qboolean *PErrFound);
qboolean SLimit (char Obj1[], char Obj2[], int ObjNo, short *PValue, qboolean Limit, qboolean *PErrFound);

qboolean NewLine; // Kludge
qboolean BlockExit;
jmp_buf  AbortBSP;

void logvprintf (char *fmt, va_list argptr)
{
        vprintf (fmt, argptr);
        fflush(stdout);
}

void logprintf (char *fmt, ...)
{
	va_list argptr;

	va_start (argptr, fmt);
	logvprintf (fmt,argptr);
	va_end (argptr);
}

void ErrorExit (void)
{
	if (BlockExit)
		longjmp (AbortBSP, 1); // Continue with next bsp instead of exiting
	
	exit (1);
}

/*
==============
PrintOptions
==============
*/
void PrintOptions (void)
{
	logprintf ("\nBspInfo lists or modifies the contents of Quake .BSP files\n\n");
	logprintf ("bspinfo [options] bspfile\n\n");
	logprintf ("Options:\n");
	logprintf ("   -chk            Enable stronger BSP validation\n");
	logprintf ("   -copyents       Copy all entities to file\n");
	logprintf ("   -cutvislight    Delete vis and light data\n");
	logprintf ("   -extents        Extract non-brush entities to file\n");
	logprintf ("   -lightstat      Display light distribution statistics in summary\n");
	logprintf ("   -lightstat2/3   Display light distribution statistics graphically,\n");
	logprintf ("                   2 = auto scale, 3 = fixed scale\n");
	logprintf ("   -limit          Limit bspdata to +/-4096 in all planes\n");
	logprintf ("   -nowatervis     Disable portal information for transparent water\n");
	logprintf ("   -prt            Generate PRT file\n");
	logprintf ("   -quiet          Disable BSP summary\n");
	logprintf ("   -texname        Replace missing texture names with 'nonameN'\n");
	logprintf ("   bspfile         .BSP file to process\n");

	exit (1);
}

/*
==================
ChkLimit
==================
*/
void ChkLimit (char Object[], int Value, int Limit)
{
	if (Value > Limit)
	{
		printf ("\nWARNING: %s %d exceed normal engine max %d", Object, Value, Limit);
		NewLine = true;
	}
}

int main (int argc, char **argv)
{
	struct _finddata_t FindInfo;
	long		   Handle;
	int		   i, Result, TotErrors = 0;
	char		   source[1024], FullPath[1024], Drv[_MAX_DRIVE], Dir[_MAX_DIR], *Option;
	qboolean	   ExtEnts = false, CutVisLight = false, Limit = false, TexName = false, BSPChanged;
	qboolean	   LightStat = false, Graph = false, FixedScale = false, GenPrt = false, Quiet = false;
	qboolean	   CopyEnts = false, ChkBSP = false;

	printf ("---- BspInfo 1.27 ---- Modified by Bengt Jardrup\n");

	// Default options
	memset (&options, 0, sizeof(options_t));
	options.watervis = true;

	for (i = 1; i < argc; ++i)
	{
		Option = argv[i];

		if (Option[0] != '-')
			break;

		++Option;

		if (!stricmp (Option, "chk"))
			ChkBSP = true;
		else if (!stricmp (Option, "copyents"))
			CopyEnts = true;
		else if (!stricmp (Option, "cutvislight"))
			CutVisLight = true;
		else if (!stricmp (Option, "extents"))
			ExtEnts = true;
		else if (!stricmp (Option, "lightstat"))
			LightStat = true;
		else if (!stricmp (Option, "lightstat2"))
			LightStat = Graph = true;
		else if (!stricmp (Option, "lightstat3"))
			LightStat = Graph = FixedScale = true;
		else if (!stricmp (Option, "limit"))
			Limit = true;
		else if (!stricmp (Option, "nowatervis"))
			options.watervis = false;
		else if (!stricmp (Option, "prt"))
			GenPrt = true;
		else if (!stricmp (Option, "quiet"))
			Quiet = true;
		else if (!stricmp (Option, "texname"))
			TexName = true;
		else if (!stricmp (Option, "?") || !stricmp (Option, "help"))
			PrintOptions ();
	}

	if (i >= argc)
		PrintOptions ();

	if (CopyEnts && ExtEnts)
	{
		// Only allow one entity operation
		ExtEnts = false;
		printf ("\nWARNING: CopyEnts overrides ExtEnts option\n");
	}

	for (; i<argc ; i++)
	{
		strcpy (source, argv[i]);
		DefaultExtension (source, ".bsp");

		_splitpath (source, Drv, Dir, NULL, NULL);

		Handle = _findfirst (source, &FindInfo);

		if (Handle == -1L)
			printf ("\nWARNING: No match for %s\n", source);
		else
		{
			Result = 0;

			while (Result == 0)
			{
				_makepath (FullPath, Drv, Dir, FindInfo.name, NULL);

				printf ("\n%s\n", FullPath);

				if (!setjmp (AbortBSP))
				{
					// Prevent program exit during BSP load
					BlockExit = true;
					LoadBSPFile (FullPath, ChkBSP);
					BlockExit = false;

					// No detected errors in BSP
					BSPChanged = NewLine = false;

					if (!Quiet)
						PrintBSPFileSizes ();

					if (DoTex (TexName))
						BSPChanged = true;

					if (LightStat)
    						DoLightStat (Graph, FixedScale);

					if (CutVisLight)
					{
    						if (DoCutVisLight ())
							BSPChanged = true;
			    		}

					if (CopyEnts || ExtEnts)
					{
    						if (DoEnts (FullPath, CopyEnts))
							BSPChanged = true;
			    		}

					if (DoLimit (Limit))
					{
						++TotErrors;

						if (Limit)
							BSPChanged = true;
						else
						{
							printf ("\nWARNING: BSP limit of +/-4096 exceeded");
							NewLine = true;
						}
					}

					// Check various engine limits
					ChkLimit ("Vis leafs", dmodels[0].visleafs, 8192); // normal engine bug
					ChkLimit ("Models", nummodels, MAX_MAP_MODELS);
					ChkLimit ("Planes", numplanes, MAX_MAP_PLANES);
					ChkLimit ("Nodes", numnodes, MAX_MAP_NODES);
					ChkLimit ("Clipnodes", numclipnodes, MAX_MAP_CLIPNODES);
					ChkLimit ("Leafs", numleafs, MAX_MAP_LEAFS);
					ChkLimit ("Vertexes", numvertexes, MAX_MAP_VERTS);
					ChkLimit ("Faces", numfaces, 32767); // normal engine bug
					ChkLimit ("Marksurfaces", nummarksurfaces, 32767); // normal engine bug
					ChkLimit ("Texinfo", numtexinfo, MAX_MAP_TEXINFO);

    					if (GenPrt)
						DoPrt (FullPath);

					if (BSPChanged)
					{
						WriteBSPFile (FullPath);
						NewLine = true;
					}

					if (NewLine)
						printf ("\n");
				}

				Result = _findnext (Handle, &FindInfo);
			}

			_findclose (Handle);
		}
	}

	return TotErrors != 0 && !Limit ? 1 : 0;
}

qboolean DoLimit (qboolean Limit)
{
	vec3_t	 v1, v2;
	float	 v3[3], Max;
	int	 i, j;
	qboolean ErrFound = false;

/*
	// Check planes for invalid normals
	for (i = 0; i < numplanes; ++i)
	{
		if (dplanes[i].normal[0] == 0 && dplanes[i].normal[1] == 0 && dplanes[i].normal[2] == 0)
		{
			printf ("\n%Plane[%5d].normal is invalid", i);

			if (!Limit)
				return true;

			// Force valid normal (0 0 1)
			dplanes[i].normal[0] = 0;
			dplanes[i].normal[1] = 0;
			dplanes[i].normal[2] = 1;
		}
	}
*/

/*
	// Limit planes to within (-BSPLIMIT BSPLIMIT)
	for (i = 0; i < numplanes; ++i)
	{
		// Get dist components
		for (j = 0; j < 3; ++j)
			v1[j] = v3[j] = dplanes[i].normal[j] * dplanes[i].dist;

		// Find max component and put in first position,
		// zero other components
		for (j = 1; j < 3; ++j)
		{
			if (fabs (v3[j]) > fabs (v3[0]))
				v3[0] = v3[j];

			v3[j] = 0;
		}

		Max = v3[0];

		if (FLimit ("Plane", "dist", i, v3, Limit, &ErrFound))
			return true;

		// Changed ?
		if (fabs (v3[0] - Max) > EQUAL_EPSILON)
		{
			VectorScale (v1, (vec_t)fabs((v3[0] / Max)), v2);
			dplanes[i].dist = VectorLength (v2); // Calculate new dist
		}
	}
*/

	// Vertices
	for (i = 0; i < numvertexes; ++i)
	{
		if (FLimit ("Vertex", "point", i, dvertexes[i].point, Limit, &ErrFound))
			return true;
	}

	// Nodes
	for (i = 0; i < numnodes; ++i)
	{
		if (SLimit ("Node", "mins", i, dnodes[i].mins, Limit, &ErrFound))
			return true;

		if (SLimit ("Node", "maxs", i, dnodes[i].maxs, Limit, &ErrFound))
			return true;
	}

/*
	// Leafs
	for (i = 0; i < numleafs; ++i)
	{
		if (SLimit ("Leaf", "mins", i, dleafs[i].mins, Limit, &ErrFound))
			return true ;

		if (SLimit ("Leaf", "maxs", i, dleafs[i].maxs, Limit, &ErrFound))
			return true ;
	}

	// Models (world only)
	for (i = 0; i < 1; ++i)
	{
		if (FLimit ("Model", "mins", i, dmodels[i].mins, Limit, &ErrFound))
			return true;

		if (FLimit ("Model", "maxs", i, dmodels[i].maxs, Limit, &ErrFound))
			return true;
	}
*/
	return ErrFound;
}

qboolean FLimit (char Obj1[], char Obj2[], int ObjNo, float *PValue, qboolean Limit, qboolean *PErrFound)
{
	int i;

	for (i = 0; i < 3; ++i)
	{
		if (fabs(*PValue) > BSPLIMIT)
		{
			if (!Limit)
				return true;

			*PErrFound = true;

			printf ("\n%s[%5d].%s[%d] = %5.0f", Obj1, ObjNo, Obj2, i, *PValue);

			*PValue = *PValue < 0 ? -BSPLIMIT : BSPLIMIT;
		}

		++PValue;
    	}

	return false;
}

qboolean SLimit (char Obj1[], char Obj2[], int ObjNo, short *PValue, qboolean Limit, qboolean *PErrFound)
{
	int i;

	for (i = 0; i < 3; ++i)
	{
		if (abs (*PValue) > BSPLIMIT)
		{
			if (!Limit)
				return true;

			*PErrFound = true;

			printf ("\n%s[%5d].%s[%d] = %5d", Obj1, ObjNo, Obj2, i, *PValue);

			*PValue = *PValue < 0 ? -BSPLIMIT : BSPLIMIT;
		}

		++PValue;
    	}

	return false;
}

qboolean DoCutVisLight (void)
{
	dleaf_t	*leaf;
	dface_t *face;
	int     i, j;

	// No vis/light data ?
	if (visdatasize == 0 && lightdatasize == 0)
		return false;

	// Get rid of vis/light data lumps
	if (dvisdata != NULL)
	{
		free (dvisdata);
		dvisdata = NULL;
	}

	if (dlightdata != NULL)
	{
		free (dlightdata);
		dlightdata = NULL;
	}

	visdatasize = lightdatasize = 0;

	// Delete vis info from leafs
	// Don't touch first leaf, it's a common solid with no faces
	for (i = 1; i < numleafs; ++i)
	{
		leaf = &dleafs[i];

		leaf->visofs = -1;

		for (j = 0; j < NUM_AMBIENTS; ++j)
			leaf->ambient_level[j] = 0;
	}

	// Delete light info from faces
	for (i = 0; i < numfaces; ++i)
	{
		face = &dfaces[i];

		face->lightofs = -1;

		for (j = 0; j < MAXLIGHTMAPS; ++j)
			face->styles[j] = 255;
	}

	printf ("\nDeleted: lightdata and visdata");

	return true;
}

qboolean DoEnts (char FullPath[], qboolean CopyEnts)
{
	int      i, j, k, EStart, ESize, TotEnts, ExtEnts;
	char     *KeepEnt, *ExtEnt, FileName[1024];
	qboolean InsideEntity, ModelPending, WorldPending, Model, World;

	// entdatasize should be sufficient
	KeepEnt = malloc (entdatasize);
	ExtEnt = malloc (entdatasize);

	// Copy all entities OR
	// extract non-brush entities and write to file

	TotEnts = ExtEnts = 0;
	InsideEntity = false;

	for (i = j = k = 0; i < entdatasize; ++i)
	{
		if (dentdata[i] == '{')
		{
			EStart = i;
			InsideEntity = true;
			ModelPending = WorldPending = Model = World = false;
		}
		else if (InsideEntity)
		{
			if (dentdata[i] == '}')
			{
				++TotEnts;
				InsideEntity = false;

				ESize = i - EStart + 1;

				if (!CopyEnts && (World || Model))
				{
					memcpy (&KeepEnt[j], &dentdata[EStart], ESize);
					j += ESize;
					KeepEnt[j++] = '\x0A'; // Add LF
				}
				else
				{
					// Entity to file
					++ExtEnts;
					memcpy (&ExtEnt[k], &dentdata[EStart], ESize);
					k += ESize;
					ExtEnt[k++] = '\x0A'; // Add LF
				}
			}
			else if (dentdata[i] == '"')
			{
				if (ModelPending)
				{
					if (dentdata[i + 1] == '*')
						Model = true;

					ModelPending = false;
				}
				else if (WorldPending)
				{
					if (!strncmp (&dentdata[i + 1], "worldspawn\"", 11))
						World = true;

					WorldPending = false;
				}
				else if (!strncmp (&dentdata[i + 1], "model\"", 6))
				{
					ModelPending = true;
					i += 6;
				}
				else if (!strncmp (&dentdata[i + 1], "classname\"", 10))
				{
					WorldPending = true;
					i += 10;
				}
			}
		}
	}

	// Only process if there are any non-brush entitites
	if (ExtEnts == 0)
	{
		printf ("\nNo non-brush entities found");
		NewLine = true;
	}
	else
	{
		if (!CopyEnts)
		{
			KeepEnt[j++] = '\0'; // Add final NUL

			// Move back kept entities (world/brush)
			entdatasize = j;
			memcpy (dentdata, KeepEnt, entdatasize);
		}

		// Generate ent file with all other entitites
		BuildName (FileName, FullPath, ".ent");

		SaveFile (FileName, ExtEnt, k);

		printf ("\n");

		if (CopyEnts)
		{
			printf ("Copied: %d entities", ExtEnts);
			NewLine = true;
		}
		else
		    	printf ("Extracted: %d non-brush entities out of %d", ExtEnts, TotEnts);

		printf (" to file %s", FileName);
	}

	free (KeepEnt);
	free (ExtEnt);

	return !CopyEnts && ExtEnts != 0;
}

qboolean DoTex (qboolean TexName)
{
	dmiptexlump_t *mtl;
	miptex_t      *mt;
	int	      i, MissTex = 0, MissNames = 0;
	char	      NoName[16];

	// No textures ?
	if (texdatasize == 0)
		return false;

	mtl = (dmiptexlump_t *)dtexdata;

	for (i = 0; i < mtl->nummiptex; ++i)
	{
		if (mtl->dataofs[i] == -1)
			++MissTex;
		else
		{
			mt = (miptex_t *)(dtexdata + mtl->dataofs[i]);

			if (!strlen (mt->name))
			{
				++MissNames;

				if (TexName)
				{
					memset (mt->name, 0, 16);
					sprintf (NoName, "noname%d", MissNames);
					strcpy (mt->name, NoName);
				}
			}
		}
	}

	if (MissTex > 0)
		printf ("\n%d missing texture%s", MissTex, MissTex > 1 ? "s" : "");

	if (MissNames > 0)
		printf ("\n%d missing texture name%s%s", MissNames, MissNames > 1 ? "s" : "", TexName ? " fixed" : "");

	if (MissTex > 0 || MissNames > 0)
		NewLine = true;

	return MissNames > 0 && TexName;
}

void DoLightStat (qboolean Graph, qboolean FixedScale)
{
	float Mean, SDev, FBPercent;
	int   i, j, k, Light[256], PeakLight[3], MeanVals, Fullbrights;

	// No light data ?
	if (lightdatasize == 0)
	{
		printf ("\nNo light data");
		NewLine = true;
		return;
	}

	// Check frequency of light data

	memset (Light, 0, sizeof(Light));

	for (i = 0; i < lightdatasize; ++i)
		++Light[dlightdata[i]];

	if (Graph)
		DoLightGraph (Light, FixedScale);

	Light[0] = 0; // Ignore 0 entry (probably noise)

	// Calculate mean, standard deviation and # fullbrights (> 255)
	for (i = Mean = MeanVals = SDev = Fullbrights = 0; i < 256; ++i)
	{
		if (Light[i] == 0)
			continue;

		Mean += Light[i] * i * 2;
		SDev += (float)Light[i] * i * 2 * i * 2;
		MeanVals += Light[i];

		if (i >= 128)
			Fullbrights += Light[i];
	}

	FBPercent = Fullbrights * 100;

	if (MeanVals == 1)
		SDev = 0;
	else if (MeanVals > 1)
	{
		Mean /= MeanVals;
		SDev = sqrt (fabs(SDev - MeanVals * Mean * Mean) / (MeanVals - 1));
		FBPercent /= MeanVals;
	}

	// Find three highest peak values, in descending order

	memset (PeakLight, 0, sizeof(PeakLight));

	for (i = 0; i < 3; ++i)
	{
		for (j = 1; j < 256; ++j)
		{
			// Make sure entry not already recorded in higher peak
/*			for (k = 1; i - k >= 0; ++k)
			{
				if (abs(j - PeakLight[i - k]) < 10) // Peaks must be separated
					break;
			}*/

			for (k = 1; i - k >= 0; ++k)
			{
				if (j == PeakLight[i - k])
					break;
			}

			if (i - k >= 0)
				continue;

			if (Light[j] > Light[PeakLight[i]])
				PeakLight[i] = j; // Current highest peak for position i
		}
	}

	printf ("\nLight peaks :");

	// Print in order of light intensity
	for (i = 0; i < 256; ++i)
	{
		for (j = 0; j < 3; ++j)
		{
			if (i != 0 && i == PeakLight[j])
			{
				printf (" %d (%d)", i * 2, Light[i]);
				break;
			}
		}
	}

	if (PeakLight[0] == 0)
		printf (" None");

	printf (", Mean = %g, SDev = %g\nFullbrights : %d (%.1f%%)\n", Q_rint(Mean), Q_rint(SDev), Fullbrights, FBPercent);
}

void DoLightGraph (int Light[], qboolean FixedScale)
{
	float Mean, Scale, MaxVal;
	int   i, j, MeanVals, Overflow, Cols;

	if (FixedScale)
	{
		// Fixed scale in about 60 columns
		MaxVal = (numfaces / 1000 + 1) * 1000;

		if (MaxVal < 3000)
			MaxVal = 3000;
		else if (MaxVal > 20000)
			MaxVal = 20000;

		Cols = 60;
	}
	else
	{
		// Calculate mean light amount (ignore level 0)
		Mean = MeanVals = 0;

		for (i = 1; i < 256; ++i)
		{
			if (Light[i] == 0)
				continue;

			Mean += Light[i];
			++MeanVals;
		}

		if (MeanVals > 0)
			Mean /= MeanVals;

		MaxVal = (int)Mean / 100 * 100;

		// Adjust MaxVal until 3 or less bars overflow
		do
		{
			MaxVal += 100;
			Overflow = 0;

			for (i = 1; i < 256; ++i)
			{
				if (Light[i] > MaxVal * 0.9)
				{
					if (++Overflow > 3)
						break;
				}
			}
		}
		while (Overflow > 3);

		// Attempt to fit most of graph in about 60 columns
		Cols = 60;
	}

	Scale = MaxVal / Cols;

	if (NewLine)
		printf ("\n");

	printf ("\nLevel Amount %s Scale: %d cols = %g", FixedScale ? "Fixed" : "Auto", Cols, Q_rint(MaxVal));

	for (i = 0; i < 256; ++i)
	{
		if (Light[i] == 0)
			continue;

		printf ("\n %3d %7d ", i * 2, Light[i]);

		for (j = 0; j < Light[i] / Scale; ++j)
		{
			if (j == Cols)
			{
				printf (">"); // Indicates bar overflowing graph
				break;
			}

			printf ("*");
		}
	}

	printf ("\n");
}

node_t *nodes, *node_p;

void DoPrt (char FullPath[])
{
	brushset_t bs;
	int        i, j;

	// Too many nodes ?
	if (numnodes > 32767 || numleafs > 32767)
	{
		printf ("\nToo many nodes or leafs for prt");
		NewLine = true;
		return;
	}

	BuildName (portfilename, FullPath, ".prt");

	planes = malloc (sizeof(plane_t) * (numplanes + 6)); // Add 6 for portals

	// Build planes array
	for (i = 0; i < numplanes; ++i)
	{
		for (j = 0; j < 3; ++j)
			planes[i].normal[j] = dplanes[i].normal[j];

		planes[i].dist = dplanes[i].dist;
		planes[i].type = dplanes[i].type;
	}

	numbrushplanes = numplanes;

	// Memory nodes can max be 3 * numnodes, usually less than 2 * numnodes
	node_p = nodes = malloc (sizeof(node_t) * numnodes * 3);
	memset (nodes, 0, sizeof(node_t) * numnodes * 3);

	// Build nodes tree
	MakeNode (0);

	// Initialize brushset boundaries
	brushset = &bs;
	memset (brushset, 0, sizeof(brushset_t));

	for (i = 0; i < 3; ++i)
	{
		brushset->mins[i] = dnodes[0].mins[i];
		brushset->maxs[i] = dnodes[0].maxs[i];
	}

	printf ("\n");

	// make the real portals for vis tracing
	PortalizeWorld (nodes, false);

	// save portal file for vis tracing
	WritePortalfile (nodes);
	FreeAllPortals (nodes);

	free (nodes);
	free (planes);
	NewLine = false;
}

void MakeNode (int nodenum)
{
	node_t	*n;
	dnode_t *dn;
	int	i;

	n = node_p++;

	if (nodenum < 0)
	{
		n->contents = dleafs[-nodenum - 1].contents;
		return;
	}

	dn = dnodes + nodenum;

	n->planenum = dn->planenum;

	for (i = 0; i < 2; ++i)
	{
		n->children[i] = node_p;
		MakeNode (dn->children[i]);
	}
}
