/*
Copyright 1998, 1998 Alexander Malmberg

Distributed under the GNU General Public License.
See legal.txt for more information.
*/

#include <std.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <math.h>

#include "d2q.h"

#include "error.h"
#include "level.h"
#include "things.h"
#include "vec.h"
#include "wad.h"

//#define DRAW

#ifdef DRAW
#include "vga.h"
#include <bios.h>
#endif

int brushes;
int entities;

FILE *map;

typedef struct
{
   int l;
   int side;
} ls_t;

typedef struct
{
   int len;
   int side;
   vec2_t vecs[512];
   ls_t ls[512];
} vchain_t;

#ifdef DRAW
void DrawChain(vchain_t *c,int col)
{
   int i;

   for (i=0;i<c->len-1;i++)
   {
      LineVec(c->vecs[i],c->vecs[i+1],col);
   }
   LineVec(c->vecs[i],c->vecs[0],col);
}
#endif

vchain_t *FindChain(int sd_start)
{
   int sec;
   linedef_t *ld=NULL;
   sidedef_t *sd;
   int i;
   int cld;
   int v;
   int nsd;

   int side=0;

   ls_t lst;

   vchain_t *temp;

   temp=Q_malloc(sizeof(vchain_t));
   memset(temp,0,sizeof(vchain_t));

   sd_flag[sd_start]=1;
   sd=&sidedefs[sd_start];

   sec=sd->sector;

   for (i=0;i<n_linedefs;i++)
   {
      ld=&linedefs[i];

      if (ld->sidedef[0]==sd_start)
      {
         side=0;
         break;
      }

      if (ld->sidedef[1]==sd_start)
      {
         side=1;
         break;
      }
   }

   if (i==n_linedefs)
      Error("Can't find linedef for sidedef!");

   cld=i;
   if (side)
      v=ld->v[1];
   else
      v=ld->v[0];

   VecCopy(temp->vecs[0],verts[v]);

   if (ld->sidedef[!side]!=-1)
   {
      lst.l=cld;
      lst.side=!side;
   }
   else
      lst.l=-1;

   temp->len=1;

   while (1)
   {
      nsd=0;
      for (i=0;i<n_linedefs;i++)
      {
         ld=&linedefs[i];

         if ((ld->v[0]!=v) && (ld->v[1]!=v))
            continue;

         nsd=ld->sidedef[0];
         sd=&sidedefs[nsd];
         if (!sd_flag[nsd] && sd->sector==sec && ld->v[0]!=v)
         {
            side=0;
            break;
         }

         nsd=ld->sidedef[1];
         if (nsd!=-1)
         {
            sd=&sidedefs[nsd];
            if (!sd_flag[nsd] && sd->sector==sec && ld->v[1]!=v)
            {
               side=1;
               break;
            }
         }
      }
      if (i==n_linedefs)
         break;

      sd_flag[nsd]=1;

      if (side)
         v=ld->v[1];
      else
         v=ld->v[0];

      VecCopy(temp->vecs[temp->len],verts[v]);

      if (ld->sidedef[!side]!=-1)
      {
         temp->ls[temp->len-1].l=i;
         temp->ls[temp->len-1].side=!side;
      }
      else
         temp->ls[temp->len-1].l=-1;

      temp->len++;
   }

   temp->ls[temp->len-1].l=lst.l;
   temp->ls[temp->len-1].side=lst.side;

   return temp;
}

int IntersectChain(vec2_t v1,vec2_t v2,vchain_t *ch)
{
   int i;

   for (i=0;i<ch->len-1;i++)
   {
      if (Intersect(v1,v2,ch->vecs[i],ch->vecs[i+1]))
         return 1;
   }

   if (Intersect(v1,v2,ch->vecs[i],ch->vecs[0]))
      return 1;

   return 0;
}

void WriteFlat(vchain_t *orig,int h1,int h2,char *stex,int roof)
{
   int i,j,k,l;
   vchain_t *c;
   char tex[128];

   sidedef_t *sd;

//   strcpy(tex,stex);
//   printf("WriteFlat(): len=%i h=%i-%i tex=%s\n",orig->len,h1,h2,tex);

   c=Q_malloc(sizeof(vchain_t));
   memcpy(c,orig,sizeof(vchain_t));

/*#ifdef DRAW
   ClrVGA();
   DrawChain(orig,1);
#endif*/
   for (i=0;i<c->len;i++)
   {
/*#ifdef DRAW
      vec2_t v2;
      if (i==c->len-1)
      {
         VecCopy(v2,c->vecs[0]);
      }
      else
      {
         VecCopy(v2,c->vecs[i+1]);
      }

      LineVec(c->vecs[i],v2,1);

      PutString(0,0,15,"%i: %i %i",i,c->ls[i].l,c->ls[i].side);
      UpdateScreen();
      PutString(0,0, 0,"%i: %i %i",i,c->ls[i].l,c->ls[i].side);

      bioskey(0);
#endif*/

/*      printf(" %3i: (%4g %4g) %i",
         i,c->vecs[i][0],c->vecs[i][1],
         c->ls[i].l);*/

      if (c->ls[i].l!=-1)
      {
         linedef_t *ld=&linedefs[c->ls[i].l];
         int side=c->ls[i].side;
         sidedef_t *sd;

         sd=&sidedefs[ld->sidedef[side]];

/*         printf(" %i (%i) %s\n",
            side,ld->sidedef[side],sd->lower);*/

         if (roof)
         {
            if (sectors[sd->sector].ch>h2) h2=sectors[sd->sector].ch;
         }
         else
         {
            if (sectors[sd->sector].fh<h1) h1=sectors[sd->sector].fh;
         }
      }
/*      else
         printf("\n");*/
   }

#define AddPlane(p00,p01,p02,p10,p11,p12,p20,p21,p22) \
   fprintf(map,"( %g %g %g ) " \
               "( %g %g %g ) " \
               "( %g %g %g ) " \
               "%s 0 0 0 1 1\n", \
      (float)p00,(float)p01,(float)p02, \
      (float)p10,(float)p11,(float)p12, \
      (float)p20,(float)p21,(float)p22, \
      tex)

   while (c->len>=3)
   {
/*#ifdef DRAW
      ClrVGA();
      DrawChain(c,15);
      UpdateScreen();
      bioskey(0);
#endif*/

      for (i=0;i<c->len;i++)
      {
         j=i+1;
         if (j>=c->len) j-=c->len;

         k=i+2;
         if (k>=c->len) k-=c->len;

         if (Angle2(c->vecs[i],c->vecs[j],c->vecs[k])>=180)
            continue;

         if (IntersectChain(c->vecs[i],c->vecs[k],c))
            continue;

         l=i-1;
         if (l<0) l+=c->len;
         if (Angle2(c->vecs[k],c->vecs[i],c->vecs[j])>
             Angle2(c->vecs[l],c->vecs[i],c->vecs[j]))
            continue;

         l=k+1;
         if (l>=c->len) l-=c->len;
         if (Angle2(c->vecs[j],c->vecs[k],c->vecs[i])>
             Angle2(c->vecs[j],c->vecs[k],c->vecs[l]))
            continue;

         break;
      }

      if (i==c->len)
      {
/*#ifdef DRAW
         ClrVGA();
         DrawChain(c,15);
         UpdateScreen();
         bioskey(0);
#endif*/

         printf("Unable to divide into convex subspaces!\n");
         return;
      }

/*#ifdef DRAW
      LineVec(c->vecs[i],c->vecs[j],14);
      LineVec(c->vecs[j],c->vecs[k],14);
      LineVec(c->vecs[k],c->vecs[i],14);
      UpdateScreen();
      bioskey(0);
#endif*/

      brushes++;
      fprintf(map,"{\n");

      strcpy(tex,stex);
      AddPlane(0,0,h2,
               0,1,h2,
               1,0,h2
               );
      AddPlane(0,0,h1,
               0,1,h1,
              -1,0,h1
               );

      if (c->ls[i].l!=-1)
      {
         sd=&sidedefs[linedefs[c->ls[i].l].sidedef[c->ls[i].side]];

         if (roof)
         {
            if (strcmp(sd->upper,"-"))
            {
               strcpy(tex,"doom/");
               strncat(tex,sd->upper,8);
            }
         }
         else
         {
            if (strcmp(sd->lower,"-"))
            {
               strcpy(tex,"doom/");
               strncat(tex,sd->lower,8);
            }
         }
         strlwr(tex);
      }

      AddPlane(c->vecs[i][0],c->vecs[i][1],h2,
               c->vecs[j][0],c->vecs[j][1],h2,
               c->vecs[i][0],c->vecs[i][1],h1
               );

      strcpy(tex,stex);

      if (c->ls[j].l!=-1)
      {
         sd=&sidedefs[linedefs[c->ls[j].l].sidedef[c->ls[j].side]];

         if (roof)
         {
            if (strcmp(sd->upper,"-"))
            {
               strcpy(tex,"doom/");
               strncat(tex,sd->upper,8);
            }
         }
         else
         {
            if (strcmp(sd->lower,"-"))
            {
               strcpy(tex,"doom/");
               strncat(tex,sd->lower,8);
            }
         }
         strlwr(tex);
      }

      AddPlane(c->vecs[j][0],c->vecs[j][1],h2,
               c->vecs[k][0],c->vecs[k][1],h2,
               c->vecs[j][0],c->vecs[j][1],h1
               );

      strcpy(tex,stex);
      if (c->len==3)
      {
         if (c->ls[k].l!=-1)
         {
            sd=&sidedefs[linedefs[c->ls[k].l].sidedef[c->ls[k].side]];
   
            if (roof)
            {
               if (strcmp(sd->upper,"-"))
               {
                  strcpy(tex,"doom/");
                  strncat(tex,sd->upper,8);
               }
            }
            else
            {
               if (strcmp(sd->lower,"-"))
               {
                  strcpy(tex,"doom/");
                  strncat(tex,sd->lower,8);
               }
            }
            strlwr(tex);
         }
      }

      AddPlane(c->vecs[k][0],c->vecs[k][1],h2,
               c->vecs[i][0],c->vecs[i][1],h2,
               c->vecs[k][0],c->vecs[k][1],h1
               );

      fprintf(map,"}\n");

      c->len--;
      c->ls[i].l=-1;
      for (i=j;i<c->len;i++)
      {
         VecCopy(c->vecs[i],c->vecs[i+1]);
         c->ls[i]=c->ls[i+1];
      }
   }

#undef AddPlane

   free(c);
}

void AddSector(int nsec)
{
   sector_t *sec;
   int i,j,k,l,m;
   int c;
   float a;
   int odd;

   char ftex[32],ctex[32],temp[32];

   vchain_t *chain[128];
   vchain_t *ch,*ch1;
   vchain_t *nc;
   int nchains;

   int nc1,nc2;
   int v1,v2;


   sec=&sectors[nsec];

   strncpy(temp,sec->ftex,8);
   temp[8]=0;
   strcpy(ftex,"doom/");
   strcat(ftex,temp);
   strlwr(ftex);

   strncpy(temp,sec->ctex,8);
   temp[8]=0;
   strcpy(ctex,"doom/");
   strcat(ctex,temp);
   strlwr(ctex);

   nchains=0;
   for (i=0;i<n_sidedefs;i++)
   {
      if (sd_flag[i])
         continue;

      if (sidedefs[i].sector==nsec)
      {
         chain[nchains]=FindChain(i);
         nchains++;
      }
   }

   odd=0;
   for (i=0;i<nchains;i++)
   {
      ch=chain[i];

      a=0;

      for (j=0;j<ch->len;j++)
      {
         k=j+1;
         if (k>=ch->len) k-=ch->len;

         l=j+2;
         if (l>=ch->len) l-=ch->len;

         a+=Angle2(ch->vecs[j],ch->vecs[k],ch->vecs[l]);
      }

      if (a<ch->len*180)
         ch->side=0;
      else
         ch->side=1;

      c=ch->side+1;

      if ((fabs(a-(ch->len-2)*180)>1) && (fabs(a-(ch->len+2)*180)>1))
      {
         printf("Warning: Odd angle!\n"
            "sec=%3i chain=%3i len=%2i sum=%4g isum=%4i osum=%4i side=%i\n",
            nsec,i,ch->len,a,(ch->len-2)*180,(ch->len+2)*180,ch->side);

         c=15;
         odd=1;
      }

/*#ifdef DRAW
      for (j=0;j<ch->len-1;j++)
      {
         LineVec(ch->vecs[j],ch->vecs[j+1],c);
         UpdateScreen();
         bioskey(0);
      }
      LineVec(ch->vecs[j],ch->vecs[0],c);

      UpdateScreen();
      bioskey(0);
#endif*/
   }

   if (odd)
   {
      printf("Skipping sector %i\n",nsec);
      return;
   }

restart_combine:
   for (i=0;i<nchains;i++)
   {
      if (!chain[i]->side)
         continue;

      ch=chain[i];

      for (j=0;j<nchains;j++)
      {
         if (chain[j]->side)
            continue;

         ch1=chain[j];

         for (k=0;k<ch->len;k++)
         {
            for (l=0;l<ch1->len;l++)
            {
               for (m=0;m<nchains;m++)
                  if (IntersectChain(ch->vecs[k],ch1->vecs[l],chain[m]))
                     break;

               if (m==nchains)
               {
                  c=15;

/*#ifdef DRAW
               ClrVGA();

               DrawChain(ch,2);
               DrawChain(ch1,1);

               LineVec(ch->vecs[k],ch1->vecs[l],c);

               UpdateScreen();
               bioskey(0);
#endif*/
               }
               else
                  c=14;

               if (c==15)
                  break;
            }
            if (l!=ch1->len)
               break;
         }
         if (k!=ch->len)
            break;
      }

      if (j==nchains)
         continue;

      nc1=i;
      nc2=j;

      v1=k;
      v2=l;

      nc=Q_malloc(sizeof(vchain_t));

      nc->side=0;

      nc->len=0;

#define AddVec(v,cl,cside) \
   { \
      VecCopy(nc->vecs[nc->len],v); \
      nc->ls[nc->len].l=cl; \
      nc->ls[nc->len].side=cside; \
      nc->len++; \
   }

      AddVec(ch1->vecs[v2],ch1->ls[v2].l,ch1->ls[v2].side);
      for (j=v2+1;(j!=v2) && (j!=v2+ch1->len);j++)
      {
         if (j>=ch1->len) j-=ch1->len;

         AddVec(ch1->vecs[j],ch1->ls[j].l,ch1->ls[j].side);
      }
      AddVec(ch1->vecs[v2],-1,0);

      AddVec(ch->vecs[v1],ch->ls[v1].l,ch->ls[v1].side);
      for (j=v1+1;(j!=v1) && (j!=v1+ch->len);j++)
      {
         if (j>=ch->len) j-=ch->len;

         AddVec(ch->vecs[j],ch->ls[j].l,ch->ls[j].side);
      }
      AddVec(ch->vecs[v1],-1,0);

/*      printf("sec=%i chain %i (%i) and %i (%i) combined, len=%i\n",
         nsec,nc1,ch->len,nc2,ch1->len,nc->len);*/

/*      DrawChain(nc,15);
      UpdateScreen();
      bioskey(0);*/

      free(ch);
      free(ch1);

      chain[nc1]=nc;

      nchains--;
      for (j=nc2;j<nchains;j++)
         chain[j]=chain[j+1];

#undef AddVec
      goto restart_combine;
   }

   for (i=0;i<nchains;i++)
   {
      ch=chain[i];

/*#ifdef DRAW
      c=1;
      ClrVGA();
      for (j=0;j<ch->len-1;j++)
      {
         LineVec(ch->vecs[j],ch->vecs[j+1],c++);
         UpdateScreen();
         bioskey(0);
      }
      LineVec(ch->vecs[j],ch->vecs[0],c);

      UpdateScreen();
      bioskey(0);
#endif*/

      if (ch->side)
      {
         printf("sec=%i chain=%i len=%i : Unable to combine inner chain!\n",
            nsec,i,chain[i]->len);
         continue;
      }

//      printf("sec=%3i chain=%3i\n",nsec,i);

      WriteFlat(ch,sec->fh-16,sec->fh,ftex,0);

      WriteFlat(ch,sec->ch,sec->ch+16,ctex,1);

      free(ch);
   }
}

void AddSectors(void)
{
   int i;

   printf("AddSectors()\n");

   memset(sd_flag,0,sizeof(int)*n_sidedefs);
   for (i=0;i<n_linedefs;i++)
   {
      if (linedefs[i].sidedef[1]==-1)
         continue;

      if (sidedefs[linedefs[i].sidedef[0]].sector==
          sidedefs[linedefs[i].sidedef[1]].sector)
      {
         sd_flag[linedefs[i].sidedef[0]]=1;
         sd_flag[linedefs[i].sidedef[1]]=1;
      }
   }

   for (i=0;i<n_sectors;i++)
   {
      AddSector(i);
   }
}

void AddWall(float x1,float y1,float a1,
             float x2,float y2,float a2,
             float z1,float z2,char *tex)
{
   char texname[16];
   float xn,yn;

   float xn1,xn2,yn1,yn2;

   float rx,ry;

   yn=x2-x1;
   xn=y1-y2;

   if (fabs(yn)>fabs(xn))
   {
      if (yn<0)
         yn=-8;
      else
         yn=8;
      xn=0;
   }
   else
   {
      if (xn<0)
         xn=-8;
      else
         xn=8;
      yn=0;
   }

   xn1=x1+xn;
   xn2=x2+xn;
   yn1=y1+yn;
   yn2=y2+yn;

   xn+=(x1+x2)/2;
   yn+=(y1+y2)/2;

   strncpy(texname,tex,8);
   texname[8]=0;
   strlwr(texname);

   brushes++;
   fprintf(map,"{\n");

#define AddPlane(p00,p01,p02,p10,p11,p12,p20,p21,p22) \
   fprintf(map,"( %g %g %g ) " \
               "( %g %g %g ) " \
               "( %g %g %g ) " \
               "doom/%s 0 0 0 1 1\n", \
      p00,p01,p02,p10,p11,p12,p20,p21,p22,texname)
         
   AddPlane( x1, y1,z2,
             x2, y2,z2,
             x1, y1,z1
            );
   AddPlane(xn1,yn1,z2,
            xn1,yn1,z1,
            xn2,yn2,z2
            );

   AddPlane(xn1,yn1,z2,
            xn2,yn2,z2,
             x1, y1,z2
            );
   AddPlane(xn1,yn1,z1,
             x1, y1,z1,
            xn2,yn2,z1
            );

   AddPlane(xn1,yn1,z2,
             x1, y1,z2,
            xn1,yn1,z1
            );
   if (a1!=-1)
   {
      rx=x1+100*cos(a1*PI/180);
      ry=y1+100*sin(a1*PI/180);

      AddPlane(x1,y1,z1,
               rx,ry,z1,
               x1,y1,z2
               );
   }

   AddPlane(xn2,yn2,z2,
            xn2,yn2,z1,
            x2, y2,z2
            );
   if (a2!=-1)
   {
      rx=x2+100*cos(a2*PI/180);
      ry=y2+100*sin(a2*PI/180);

      AddPlane(x2,y2,z1,
               x2,y2,z2,
               rx,ry,z1
               );
   }


/*   AddPlane( xn, yn,z2,
             x1, y1,z2,
             xn, yn,z1
            );
   AddPlane( xn, yn,z2,
             xn, yn,z1,
             x2, y2,z2
            );*/

   fprintf(map,"}\n");

#undef AddPlane
}

void AddWalls(void)
{
   int i,j;
   sidedef_t *r[2];
   linedef_t *l;
   sector_t *s[2];

   float a,a1,a2;

   vec2_t v1,v2,v3;

   printf("AddWalls()\n");

   for (i=0;i<n_linedefs;i++)
   {
      l=&linedefs[i];
      r[0]=&sidedefs[l->sidedef[0]];
      if (l->sidedef[1]==-1)
         r[1]=NULL;
      else
         r[1]=&sidedefs[l->sidedef[1]];

      s[0]=&sectors[r[0]->sector];
      if (r[1])
         s[1]=&sectors[r[1]->sector];
      else
         s[1]=NULL;

      if (r[1])
         continue;

      if (!strcmp(r[0]->middle,"-") || (s[0]->fh==s[0]->ch))
         continue;

      a1=a2=360;
      for (j=0;j<n_linedefs;j++)
      {
         if (j==i)
            continue;

         if (linedefs[j].v[0]==l->v[0])
         {
            VecCopy(v1,verts[linedefs[j].v[1]]);
            VecCopy(v2,verts[l->v[0]]);
            VecCopy(v3,verts[l->v[1]]);
            a=Angle2(v1,v2,v3);

            if (a<a1)
               a1=a;
         }

         if (linedefs[j].v[1]==l->v[0])
         {
            VecCopy(v1,verts[linedefs[j].v[0]]);
            VecCopy(v2,verts[l->v[0]]);
            VecCopy(v3,verts[l->v[1]]);
            a=Angle2(v1,v2,v3);
            if (a<a1)
               a1=a;
         }

         if (linedefs[j].v[0]==l->v[1])
         {
            VecCopy(v1,verts[l->v[0]]);
            VecCopy(v2,verts[l->v[1]]);
            VecCopy(v3,verts[linedefs[j].v[1]]);
            a=Angle2(v1,v2,v3);

            if (a<a2)
               a2=a;
         }

         if (linedefs[j].v[1]==l->v[1])
         {
            VecCopy(v1,verts[l->v[0]]);
            VecCopy(v2,verts[l->v[1]]);
            VecCopy(v3,verts[linedefs[j].v[0]]);
            a=Angle2(v1,v2,v3);
            if (a<a2)
               a2=a;
         }
      }

#ifdef DRAW
      LineVec(verts[l->v[0]],verts[l->v[1]],14);
      UpdateScreen();
      bioskey(0);
#endif

//      printf("a1=%g a2=%g\n",a1,a2);

      if (a1>=180)
         a1=-1;
      else
      {
         VecCopy(v1,verts[l->v[1]]);
         VecCopy(v2,verts[l->v[0]]);
         VecCopy(v3,verts[l->v[0]]);
         v3[0]+=100;
         a1=a1/2+Angle2(v1,v2,v3);

         while (a1>=360) a1-=360;
      }

      if (a2>=180)
         a2=-1;
      else
      {
         VecCopy(v1,verts[l->v[0]]);
         VecCopy(v2,verts[l->v[1]]);
         VecCopy(v3,verts[l->v[1]]);
         v3[0]+=100;
         a2=Angle2(v1,v2,v3)-a2/2;
         while (a2<0) a2+=360;
      }

      if ((a1!=-1) || (a2!=-1))
      {
/*         printf("call values: a1=%g a2=%g\n",a1,a2);
         printf("a1=(%i %i) a2=(%i %i)\n",
            verts[l->v[0]][0],verts[l->v[0]][1],
            verts[l->v[1]][0],verts[l->v[1]][1]);*/
      }

      AddWall(verts[l->v[0]][0],verts[l->v[0]][1],a1,
              verts[l->v[1]][0],verts[l->v[1]][1],a2,
              s[0]->fh,s[0]->ch,
              r[0]->middle);
   }
}

int main(int argc, char **argv)
{
   char mapname[128];

   printf(
      "d2q version 1.0 (" __DATE__ ")\n"
      "-----------------------------------\n"
      " copyright 1998, 1998\n"
      "   Alexander Malmberg <alexander@malmberg.org>\n"
      "\n"
      "Distributed under the GNU General Public License.\n"
      "See legal.txt for more information.\n"
      "\n"
      );

   if (argc!=3)
   {
      printf(
         "Syntax:\n"
         "  d2q wadfile.wad lump\n"
         "\n"
         "  'wadfile.wad' is the name of the wad file with extension.\n"
         "  'lump' is the name of the base map lump in the wad file.\n"
         "  lump.map will be written\n"
         "\n"
         "Example:\n"
         "  d2q doom.wad e1m1\n"
         "    Will extract e1m1 from doom.wad and write e1m1.map.\n"
         "\n"
         );
      exit(1);
   }

   InitWad(argv[1]);

   LoadLevel(argv[2]);

   Recenter();

   InitBSP();

   strcpy(mapname,argv[2]);
   strcat(mapname,".map");
   map=fopen(mapname,"wt");
//   map=fopen("/dev/nul","wt");
   if (!map)
      Error("Couldn't create '%s'!",mapname);

   fprintf(map,
      "{\n"
      "\"classname\" \"worldspawn\"\n"
      );

   brushes=0;
   entities=1;

#ifdef DRAW
   InitVGA();
   SetVGA();
   ClrVGA();
#endif

   AddSectors();

   AddWalls();

   fprintf(map,
      "}\n"
      );

   AddThings();

#ifdef DRAW
   SetText();
#endif

   fclose(map);

   printf("%s written\n",mapname);
   printf("%5i brushes\n",brushes);
   printf("%5i entities\n",entities);

   return 0;
}

