/*
clip.c file of the MergeT3D Source Code

Copyright 1997, 1998 Alexander Malmberg
Copyright Gyro Gearloose
Copyright 1996, Trey Harrison and Chris Carollo

This program is distributed under the GNU General Public License.
See legal.txt for more information.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "types.h"

#include "clip.h"

#include "error.h"
#include "geom.h"
#include "map.h"
#include "memory.h"
#include "misc.h"


void AddPoint(face_my_t *F, vec3_t pt)
/* 
   Adds a point to the face
*/
{
	F->pts[F->numpts] = pt;
	F->numpts++;
}

int RemoveFace(brush_my_t *B, int pos)
/* 
   Removes the face from the brush
*/
{
	int i;

	if (B->numfaces==0) return 0;
	for (i=pos;i<B->numfaces-1;i++)
	{
		B->faces[i] = B->faces[i+1];
	}
	B->numfaces--;
	return 1;
}

int DivideFaceByPlane(face_my_t F,plane_my_t P,face_my_t *Front,face_my_t *Back, int KeepBack)
/* 
  
  Returns:     0 - Face was clipped successfully
	       -1 - Face was completely behind the plane
	       -2 - Face was completely in front of the plane
	       -3 - Error clipping / culling face 
	       -4 - Face was completely ON the plane
*/
{        
	face_my_t FrontFace;
	face_my_t BackFace;
	vec3_t p1,p2,p3;
	int i,j,tfpts,tbpts;
	double dot;
	int backtest=0;
	int fronttest=0;
	int ontest=0;
	int splitnum=0;
	int sides[MAX_FACE_POINTS];
	double dists[MAX_FACE_POINTS];
	
	if (F.numpts>MAX_FACE_POINTS) return -3;
	tfpts = 0;
	tbpts = 0;

	// determine sides and distances for each point
	for (i=0;i<F.numpts;i++)
	{
		dot = DotProd(F.pts[i], P.normal);
		dot -= P.dist;
		dists[i] = dot;
		if (dot > 0.01) { sides[i] = 1; fronttest=1; }
		else if (dot < -0.01) { sides[i] = -1; backtest=1; }
		else { dists[i]=0; sides[i] = 0; ontest=1; }
	}
	
	if ((ontest==1)&&(backtest==0)&&(fronttest==0))
	{
		return -4;
	}

	if ( ((backtest==0)&&(fronttest==1)) || ((backtest==1)&&(fronttest==0)) )
	{
		//if all in front
		if (fronttest)
		{
			*Front = F;
			return -2;
		}
		//if all in back
		if (backtest)
		{
			if (KeepBack)
			{
				*Back = F;
			}
			return -1;
		}
	}
	
	FrontFace.numpts = 0;
	
	if (KeepBack)
	{
		BackFace.numpts = 0;
	}

	for (i=0;i<F.numpts;i++)
	{
		p1 = F.pts[i];
		
		if (i==F.numpts-1)
         j=0;
      else
         j=i+1;
		
		if (sides[i] == 0)
		{
			tfpts++; if (tfpts>MAX_FACE_POINTS) return -3;
			AddPoint(&FrontFace,p1);
			if (KeepBack)
			{
				tbpts++;
				if (tbpts>MAX_FACE_POINTS) return -3;
				AddPoint(&BackFace,p1);
			}
			if ((splitnum==2)||(sides[j]==0))
			{
				return -3;
			}
			splitnum++;
			continue;
		}
	
		if (sides[i] == 1)
		{
			tfpts++;
			if (tfpts>MAX_FACE_POINTS) return -3;
			AddPoint(&FrontFace,p1);
		}
		if (sides[i] == -1)
		{
			if (KeepBack)
			{
				tbpts++;
				if (tbpts>MAX_FACE_POINTS) return -3;
				AddPoint(&BackFace,p1);
			}
		}
		
		if ((sides[j]==sides[i])||(sides[j]==0)) continue;

		// generate a split point
		p2 = F.pts[j];
		
		dot = dists[i] / (dists[i]-dists[j]);
		
		if (P.normal.x == 1)		   p3.x = P.dist;
		else if (P.normal.x == -1) p3.x = -P.dist;
		else                       p3.x = p1.x + dot*(p2.x-p1.x);
		
		if (P.normal.y == 1)       p3.y = P.dist;
		else if (P.normal.y == -1) p3.y = -P.dist;
		else p3.y =                p1.y + dot*(p2.y-p1.y);
		
		if (P.normal.z == 1)       p3.z = P.dist;
		else if (P.normal.z == -1) p3.z = -P.dist;
		else                       p3.z = p1.z + dot*(p2.z-p1.z);
		
		tfpts++;
		if (tfpts>MAX_FACE_POINTS) return -3;
		AddPoint(&FrontFace,p3);
		if (KeepBack)
		{
			tbpts++;
			if (tbpts>MAX_FACE_POINTS) return -3;
			AddPoint(&BackFace,p3);
		}
		if ((splitnum==2)||(sides[j]==0))
		{
			return -3;
		}
		splitnum++;
	}
		
	if (splitnum!=2)
	{
		return -3;
	}
	Front->numpts = FrontFace.numpts;
	for (i=0;i<Front->numpts;i++) Front->pts[i] = FrontFace.pts[i];
	Front->normal = F.normal;
	Front->dist = F.dist;
   Front->tex=F.tex;

	if (KeepBack)
	{
		Back->numpts = BackFace.numpts;
		for (i=0;i<Back->numpts;i++) Back->pts[i] = BackFace.pts[i];
		Back->normal = F.normal;
		Back->dist = F.dist;
      Back->tex=F.tex;
	}

	return 0;
}

int CreateBrushFaces(brush_my_t *B);

int BuildBrush(brush_my_t *B, brush_t *b)
{
	edge_t          tmpedge;
	int             i,j,k,l;
	int            flag;
	vec3_t *oldvert=NULL;
	edge_t *oldedge;

	if (!CreateBrushFaces(B))
	{
		return 0;
	}

	b->plane = (plane_t *)Q_malloc(sizeof(plane_t) * (B->numfaces));
	if (b->plane==NULL)
	{
		HandleError("BuildBrush","Could not allocate brush planes");
		Q_free(B->faces);
		return 0;
	}
	b->num_planes = B->numfaces;
	
	b->edges = NULL;
	b->num_edges = 0;
	b->verts = NULL;
	b->num_verts = 0;
	b->tverts = NULL;
	
	for (i=0;i<B->numfaces;i++)
	{
      b->plane[i].tex=B->faces[i].tex;
		
		b->plane[i].normal = B->faces[i].normal;
		
		b->plane[i].verts = (int *)Q_malloc(sizeof(int) * (B->faces[i].numpts));
		if (b->plane[i].verts==NULL)
		{
			Q_free(B->faces);
			HandleError("BuildBrush","Could not allocate brush plane vertices");
			return NULL;
		}
		
		b->plane[i].num_verts = B->faces[i].numpts;
		
		for (j=0;j<B->faces[i].numpts;j++)
		{
			flag=0;
			for (k=0;k<b->num_verts;k++)
			{
				if (VertsAreEqual(b->verts[k],B->faces[i].pts[j]))
				{
					flag=1;
					break;
				}
			}
			if (flag==1)
         {
			   b->plane[i].verts[j] = k;
         }
			else
			{
				oldvert = b->verts;
				b->verts = (vec3_t *)Q_realloc(b->verts,sizeof(vec3_t) * (b->num_verts+1));
				if (b->verts==NULL)
				{
					b->verts=oldvert;
					Q_free(B->faces);
					HandleError("BuildBrush","Could not allocate brush vertices");
					return NULL;
				}
				oldvert=b->verts;
				b->verts[b->num_verts] = B->faces[i].pts[j];
				b->plane[i].verts[j] = b->num_verts;
				b->num_verts++;
			}
		}
		
		for (j=0;j<b->plane[i].num_verts;j++) {
			if (j==b->plane[i].num_verts-1) l=0;
				else l=j+1;
			flag=0;
			for (k=0;k<b->num_edges;k++) {
				tmpedge.startvertex = b->plane[i].verts[j];
				tmpedge.endvertex   = b->plane[i].verts[l];
				if  (EdgesAreEqual(b->edges[k],tmpedge))
				{
					flag=1;
					break;
				}
			}
			if (flag==1)
			{
			} // do nothing, the edge already has been made
			else
			{
				oldedge = b->edges;
				b->edges = (edge_t *)Q_realloc(b->edges,sizeof(edge_t) * (b->num_edges+1));
				if (b->edges==NULL)
				{
					b->edges=oldedge;
					b->verts=oldvert;
					Q_free(B->faces);
					HandleError("BuildBrush","Could not allocate brush edges");
					return NULL;
				}
				oldedge=b->edges;
				
				b->edges[b->num_edges].startvertex = b->plane[i].verts[j];
				b->edges[b->num_edges].endvertex   = b->plane[i].verts[l];
				b->num_edges++;
			}
		}
	}
	Q_free(B->faces);
	b->tverts = (vec3_t *)Q_malloc(sizeof(vec3_t) * (b->num_verts));
	if (b->tverts==NULL)
	{
		HandleError("BuildBrush","Could not allocate transformed brush verts");
		return NULL;
	}
	
	return 1;
}

typedef float qbsp_vec3_t[3];

typedef struct
{
	qbsp_vec3_t  normal;
	float   dist;
	int     type;
} qbsp_plane_t;

void VectorMA (qbsp_vec3_t va, float scale, qbsp_vec3_t vb, qbsp_vec3_t vc)
{
	vc[0] = va[0] + scale*vb[0];
	vc[1] = va[1] + scale*vb[1];
	vc[2] = va[2] + scale*vb[2];
}

void CrossProduct (qbsp_vec3_t v1, qbsp_vec3_t v2, qbsp_vec3_t cross)
{
	cross[0] = v1[1]*v2[2] - v1[2]*v2[1];
	cross[1] = v1[2]*v2[0] - v1[0]*v2[2];
	cross[2] = v1[0]*v2[1] - v1[1]*v2[0];
}

float DotProduct (qbsp_vec3_t v1, qbsp_vec3_t v2)
{
	return v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2];
}

void VectorSubtract (qbsp_vec3_t va, qbsp_vec3_t vb, qbsp_vec3_t out)
{
	out[0] = va[0]-vb[0];
	out[1] = va[1]-vb[1];
	out[2] = va[2]-vb[2];
}

void VectorAdd (qbsp_vec3_t va, qbsp_vec3_t vb, qbsp_vec3_t out)
{
	out[0] = va[0]+vb[0];
	out[1] = va[1]+vb[1];
	out[2] = va[2]+vb[2];
}


void VectorNormalize (qbsp_vec3_t v)
{
	int             i;
	double  length;
	
	length = 0;
	for (i=0 ; i< 3 ; i++)
		length += v[i]*v[i];
	length = sqrt (length);
	if (length == 0) return;
		
	for (i=0 ; i< 3 ; i++) v[i] /= length; 
}


void VectorScale (qbsp_vec3_t v, float scale, qbsp_vec3_t out)
{
	out[0] = v[0] * scale;
	out[1] = v[1] * scale;
	out[2] = v[2] * scale;
}

void MakeBoxOnPlane (face_my_t *P)
{
	int             i, x;
	float           max, v;
	qbsp_vec3_t         org, vright, vup,temppt;
	qbsp_plane_t         p;

	p.normal[0] = P->normal.x;
	p.normal[1] = P->normal.y;
	p.normal[2] = P->normal.z;
	p.dist = P->dist;
	
	// find the major axis
	max = -99999;
	x = -1;
	for (i=0 ; i<3; i++)
	{
		v = fabs(p.normal[i]);
		if (v > max) {
			x = i;
			max = v;
		}
	}
	if (x==-1) {
		printf("Could not find major axis");
		exit(-1);
	}
		
	vup[0]=0; vup[1]=0; vup[2]=0;
	switch (x) {
		case 0:
		case 1:
			vup[2] = 1;
			break;          
		case 2:
			vup[0] = 1;
			break;          
	}

	v = DotProduct (vup, p.normal);
	VectorMA (vup, -v, p.normal, vup);
	VectorNormalize (vup);
		
	VectorScale (p.normal, p.dist, org);
	
	CrossProduct (vup, p.normal, vright);
	
	VectorScale (vup, 8192, vup);
	VectorScale (vright, 8192, vright);

	// project a really big axis aligned box onto the plane
	P->numpts = 4;
	
	VectorSubtract (org, vright, temppt); 
	VectorAdd (temppt, vup, temppt);
	P->pts[0].x = temppt[0];
	P->pts[0].y = temppt[1];
	P->pts[0].z = temppt[2];
	
	VectorAdd (org, vright, temppt);
	VectorAdd (temppt, vup, temppt);
	P->pts[1].x = temppt[0];
	P->pts[1].y = temppt[1];
	P->pts[1].z = temppt[2];

	
	VectorAdd (org, vright, temppt);
	VectorSubtract (temppt, vup, temppt);
	P->pts[2].x = temppt[0];
	P->pts[2].y = temppt[1];
	P->pts[2].z = temppt[2];

	
	VectorSubtract (org, vright, temppt);
	VectorSubtract (temppt, vup, temppt);
	P->pts[3].x = temppt[0];
	P->pts[3].y = temppt[1];
	P->pts[3].z = temppt[2];

}


int CreateBrushFaces(brush_my_t *B)
{                               
	face_my_t Face1,Face2;
	face_my_t NewFace;
	plane_my_t P;
	int i,j,k;
	int flag;
	
	for (i=0; i<B->numfaces;i++) B->faces[i].misc=0;

	for (i=0 ; i<B->numfaces ; i++) {
		NewFace = B->faces[i];
		MakeBoxOnPlane(&NewFace);        

		flag=0;
		for (j=0;j<B->numfaces;j++) {
			if (j==i) continue;
			if (B->faces[j].misc==1) continue;
			
			P = B->faces[j];
			P.normal.x = -P.normal.x;
			P.normal.y = -P.normal.y;
			P.normal.z = -P.normal.z;
			P.dist = -P.dist;
			
			k = DivideFaceByPlane(NewFace,P,&Face1,&Face2,0);
			if (k==-4) {
				//printf("Duplicate plane encountered, face removed\n");
				flag=1;
				break;
			}
			if (k==-3) {
				//printf("Error splitting face, face removed\n");
				flag=1;
				break;
			}
			if (k==-1) {
				//printf("Face exceeded brush bounds, face removed\n");
				flag=1;
				break;
			}
			NewFace = Face1;
		}
		if (flag) {
			B->faces[i].misc = 1;
			continue;
		}
		
		B->faces[i] = NewFace;
		B->faces[i].misc = 0;
	}       
	i=0;
	j=B->numfaces;
	flag=0;
	do {
		if (B->faces[i].misc==1) {
			if (RemoveFace(B,i)==0) {
				flag=1;
				break;
			}
			j--;
		} else i++;
	} while (i<j);
	if ((flag)||(B->numfaces<4)) {        
		Q_free(B->faces);
		return 0;
	}
	return 1;
}

