#include <Aztec3DPCH.h>

#include <controls/MainWindow.h>

// Aztec2 includes
#include <controls/CommandButton.h>
#include <controls/FourWaySplitterMethod.h>
#include <controls/TimeControlPanel.h>
#include <controls/TimeSlider.h>
#include <controls/ToolButton.h>
#include <scripting/MScriptInterp.h>
#include <views/Aztec3DView.h>
#include <views/AztecGraphView.h>
#include <views/AztecViewManager.h>
#include <views/UVEditView.h>
#include <views/ImageView.h>
#include <views/OutlineView.h>

// AztecGUI includes
#include <gui/MApplication.h>
#include <gui/MBorderLayout.h>
#include <gui/MFlowLayout.h>
#include <gui/MGridLayout.h>
#include <gui/MButton.h>
#include <gui/MComboBox.h>
#include <gui/MTextField.h>
#include <gui/MMenuBar.h>
#include <gui/util/MMenuFactory.h>
#include <tools/MToolManager.h>

// AztecLib includes
#include <MUIManager.h>
#include <MListener.h>
#include <misc/ResourceManager.h>

#include <assert.h>

namespace AztecGUI {

  class SingleLineEditListener : public Aztec::MLogFileListener { 
  public:
    
    SingleLineEditListener(Aztec::MTextFieldPtr field) 
      : textbox(field)
    {
      clearLineForNextTime = true;
    }
    
    void writeString(Aztec::MLogFile *logfile, const std::string &text) {
      // if we have to clear the line, then just use the write line method,
      // otherwise we append to the current line.
      if (clearLineForNextTime) {
        writeLine(logfile, text);
        clearLineForNextTime = false;
      } else {
        // however, if we have a end of line character in the text,
        // then we have to only use what is after that
        int endOfLineIndex = text.rfind('\n');
        
        if (endOfLineIndex != std::string::npos && endOfLineIndex == text.length() - 1) {
          // only set the text to what appears after the end of line character.
          std::string line = text.substr(endOfLineIndex+1);
          
          // only bother doing anything if there is anything on the 
          // line worth doing
          if (line.length() > 0) {
            textbox->setValue(line);
          }
          clearLineForNextTime = true;
          
        } else {
          clearLineForNextTime = true;
          // otherwise, we append what we have to the current text.
          textbox->setValue(textbox->getValue() + text);
        }
      }
    }
    
    void writeLine(Aztec::MLogFile *logfile, const std::string &text) {

      std::string str = text;

      // first strip out the "JS: "
      if (str.find("JS: ") == 0) {
        str = str.substr(4);
      }

      while (str.length() > 0) {
        // get the last line of it.
        int endOfLineIndex = str.rfind('\n');
        std::string line = str.substr(endOfLineIndex+1);
        // trim any whitespaces.
        while (line.length() > 0 && line.rfind(' ') + 1 == line.length()) {
          line = line.substr(0, line.length() - 1);
        }

        if (line.length() > 0) {
          textbox->setValue(line);
          break;
        }

        str = str.substr(0, endOfLineIndex);
      }
      
      clearLineForNextTime = true;
    }
    
  private:
    Aztec::MTextFieldPtr textbox;
    bool clearLineForNextTime;
  };


  static MainWindow *globalInstance = NULL;

  MainWindow::MainWindow() 
    : MWindow("Aztec 2.0.g-alpha")
  {
    assert(globalInstance == NULL);
    globalInstance = this;
  }

  MainWindow::~MainWindow() {
    globalInstance = NULL;
  }

  MainWindow* MainWindow::getInstance() {
    return globalInstance;
  }

  class ModeListener : public Aztec::MListener { 
  public:

    ModeListener(const AztecGUI::MButtonPtr &object, 
                 const AztecGUI::MButtonPtr &point, 
                 const AztecGUI::MButtonPtr &face, 
                 const AztecGUI::MButtonPtr &edge) 
      : o(object), p(point), f(face), e(edge)
    {
    }

    void onListen() {
      o->setButtonDown(Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE);
      p->setButtonDown(Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE);
      f->setButtonDown(Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE);
      e->setButtonDown(Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE);
    }

    Aztec::MButtonPtr o, p, f, e;


  };

  class AxisCombo : public Aztec::MComboBox {
  public:
    static const char *WORLD;
    static const char *LOCAL;
    static const char *PARENT;
    static const char *SCREEN;
    static const char *OBJECT;

    AxisCombo() : MComboBox()
    {
    }

    bool onChanged() {
      std::string value = getSelectedItem();
      if (value == WORLD) {
        changeMode("world");
      } else if (value == LOCAL) {
        changeMode("local");
      } else if (value == PARENT) {
        changeMode("parent");
      } else if (value == SCREEN) {
        changeMode("screen");
      } else if (value == OBJECT) {
        changeMode("object");
      }

      return true;
    }

    void onCreate() {
      addItem(WORLD);
      addItem(LOCAL);
      addItem(PARENT);
      addItem(SCREEN);
      addItem(OBJECT);
      setSelectedIndex(0);
    }

  private:
    void changeMode(const std::string &mode) {
      std::string cmd = "Scene.axisModeSet('" + mode + "');";
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript(cmd.c_str());
    }
  };

  class AxisComboListener : public Aztec::MListener {
  public:
    AxisComboListener(const Aztec::MComboBoxPtr &combo) : comboBox(combo)
    {
    }

    void onListen() {
      switch (Aztec::MUIManager::getAxisMode()) {
      case Aztec::MUIManager::WORLD_AXIS:
        comboBox->setValue(AxisCombo::WORLD); break;
      case Aztec::MUIManager::LOCAL_AXIS:
        comboBox->setValue(AxisCombo::LOCAL); break;
      case Aztec::MUIManager::PARENT_AXIS:
        comboBox->setValue(AxisCombo::PARENT); break;
      case Aztec::MUIManager::SCREEN_AXIS:
        comboBox->setValue(AxisCombo::SCREEN); break;
      case Aztec::MUIManager::OBJECT_AXIS:
        comboBox->setValue(AxisCombo::OBJECT); break;
      }
    }

  private:
    Aztec::MComboBoxPtr comboBox;
  };


  const char *AxisCombo::WORLD = "World";
  const char *AxisCombo::LOCAL = "Local";
  const char *AxisCombo::PARENT = "Parent";
  const char *AxisCombo::SCREEN = "Screen";
  const char *AxisCombo::OBJECT = "Object";

  class SeparateAxisListener : public Aztec::MListener {
  public:
    SeparateAxisListener(const MButtonPtr &button) {
      but = button;
    }

    void onListen() {
      bool separate = Aztec::MUIManager::getSeparateAxisMode();

      if (separate) {
        but->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/axisSeparateOn.bmp")));
      } else {
        but->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/axisSeparateOff.bmp")));
      }
    }

  private:
    MButtonPtr but;
  };

  class CurrentToolListener : public AztecGUI::MToolChangeListener {
  public:
    CurrentToolListener(const ToolButtonPtr &button) {
      but = button;
    }

    void onToolChange(const std::string &group, AztecGUI::MToolTypePtr oldTool) {
      std::string newToolName = AztecGUI::MToolManager::getInstance()->GetTool(group)->getName();

      if (!AztecGUI::MToolManager::getInstance()->isTemporary(group) &&
          newToolName != "toolSelect" && 
          newToolName != "toolMove" && 
          newToolName != "toolRotate" && 
          newToolName != "toolScale") 
      {
        but->setTool(newToolName, group, newToolName);
      }
    }

    ToolButtonPtr but;
  };

  class ScriptListener : public Aztec::MTextField::TextListener {
  public:
    void onListen(const MTextFieldPtr &field) {
      Aztec::MSystemManager::getInstance()->logOutput(field->getValue().c_str());
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript(field->getValue().c_str());
      field->setValue("");
    }

  };

  void MainWindow::onCreate() {
    splitter = new AztecGUI::MultiSplitter();
    splitter->setSplitterMethod(new FourWaySplitter(6, 0, FourWaySplitter::FourWay));
    channelBar = new AztecGUI::ChannelBar();
    bottomToolbar = new Aztec::MContainer();

    setLayoutManager(new Aztec::MBorderLayout());
    addComponent(splitter, Aztec::MBorderLayout::CENTRE);
    addComponent(channelBar, Aztec::MBorderLayout::EAST);

    MContainerPtr bottomCont = new Aztec::MContainer();
    addComponent(bottomCont, Aztec::MBorderLayout::SOUTH);
//    addComponent(new MTimeSlider(), Aztec::MBorderLayout::SOUTH_INNER);

    bottomCont->setLayoutManager(new Aztec::MBorderLayout());
    bottomCont->addComponent(bottomToolbar, Aztec::MBorderLayout::CENTRE);
    bottomCont->addComponent(new MTimeSlider(), Aztec::MBorderLayout::NORTH_INNER);

    MMenuBarPtr bar = new MMenuBar;
    bar->createImpl();  // Temporary hack to make Qt/Linux version work (the smart pointer bites once again)

    Aztec::MMenuFactory::loadMenu(bar, "menus/mainWindow.menu");
    addComponent(bar, Aztec::MBorderLayout::NORTH);

    {
      Aztec::MContainerPtr topToolbar = new Aztec::MContainer();
      addComponent(topToolbar, Aztec::MBorderLayout::NORTH_INNER);
      topToolbar->setLayoutManager(new Aztec::MFlowLayout());

      Aztec::MButtonPtr objectBut, pointBut, faceBut, edgeBut;

      objectBut = new CommandButton("Object","Scene.componentModeSet('object')");
      pointBut = new CommandButton("Point","Scene.componentModeSet('point')");
      faceBut = new CommandButton("Face","Scene.componentModeSet('facet')");
      edgeBut = new CommandButton("Edge","Scene.componentModeSet('edge')");

      objectBut->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/modeObject_16.bmp")));
      pointBut->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/modePoint_16.bmp")));
      faceBut->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/modeFace_16.bmp")));
      edgeBut->setImage(MSystemManager::getInstance()->loadImage(Aztec::ResourceManager::locateResource("icons/modeEdge_16.bmp")));

      MUIManager::addListener(MUIManager::COMPONENT_MODE_LISTENER, new ModeListener(objectBut, pointBut, faceBut, edgeBut));



      topToolbar->addComponent(new ToolButton("Select","3D", "toolSelect"));
      topToolbar->addComponent(new ToolButton("Move","3D", "toolMove"));
      topToolbar->addComponent(new ToolButton("Rotate","3D", "toolRotate"));
      topToolbar->addComponent(new ToolButton("Scale","3D", "toolScale"));
      
      ToolButtonPtr currentToolBut = new ToolButton("Current Tool","3D", "toolSelect");

      topToolbar->addComponent(currentToolBut);

      MToolManager::getInstance()->addListener(new CurrentToolListener(currentToolBut), "3D");
      
      topToolbar->addComponent(objectBut);      
      topToolbar->addComponent(pointBut);      
      topToolbar->addComponent(faceBut);      
      topToolbar->addComponent(edgeBut);      

      Aztec::MComboBoxPtr box = new AxisCombo();
      box->setMinimumSize(Aztec::MSize2D(100,24));
      topToolbar->addComponent(box);

      Aztec::MUIManager::addListener(Aztec::MUIManager::AXIS_MODE_LISTENER, new AxisComboListener(box));

      Aztec::MButtonPtr axisToggleBut = new CommandButton("o","Scene.axisSeparateToggle()");
      Aztec::MUIManager::addListener(Aztec::MUIManager::SEPARATE_AXIS_LISTENER, new SeparateAxisListener(axisToggleBut));
      topToolbar->addComponent(axisToggleBut);



    }

    Aztec::MUIManager::setSeparateAxisMode(true);

    TimeControlPanelPtr panel = new TimeControlPanel();
    bottomCont->addComponent(panel, Aztec::MBorderLayout::EAST);


    bottomToolbar->setLayoutManager(new Aztec::MGridLayout(2,2));


    
    Aztec::MTextFieldPtr helpFeedback = new Aztec::MTextField(10);
    Aztec::MTextFieldPtr scriptCommandLine = new Aztec::MTextField(10);
    Aztec::MTextFieldPtr scriptFeedback = new Aztec::MTextField(10);

    bottomToolbar->addComponent(scriptCommandLine, Aztec::MGridLayout::getConstaint(0, 0));
    bottomToolbar->addComponent(scriptFeedback, Aztec::MGridLayout::getConstaint(1, 0));
    bottomToolbar->addComponent(helpFeedback, Aztec::MGridLayout::getConstaint(0, 1));

    helpFeedback->setEnabled(false);
    scriptFeedback->setEnabled(false);
    scriptCommandLine->setEnabled(true);
    scriptCommandLine->addAcceptListener(new ScriptListener());

    Aztec::MSystemManager::getInstance()->getLogger()->addListener(new SingleLineEditListener(scriptFeedback));


    AztecViewManager::registerViewType("perspective", new AztecGUI::Aztec3DView("Perspective", AztecGUI::Aztec3DView::Persp, Aztec3DSceneCanvas::ShadingFlat));
    AztecViewManager::registerViewType("front", new AztecGUI::Aztec3DView("Front", AztecGUI::Aztec3DView::Front));
    AztecViewManager::registerViewType("back", new AztecGUI::Aztec3DView("Back", AztecGUI::Aztec3DView::Back));
    AztecViewManager::registerViewType("top", new AztecGUI::Aztec3DView("Top", AztecGUI::Aztec3DView::Top));
    AztecViewManager::registerViewType("bottom", new AztecGUI::Aztec3DView("Bottom", AztecGUI::Aztec3DView::Bottom));
    AztecViewManager::registerViewType("left", new AztecGUI::Aztec3DView("Left", AztecGUI::Aztec3DView::Left));
    AztecViewManager::registerViewType("right", new AztecGUI::Aztec3DView("Right", AztecGUI::Aztec3DView::Right));
    AztecViewManager::registerViewType("uv", new AztecGUI::UVEditView("UV Edit"));
    AztecViewManager::registerViewType("image", new AztecGUI::ImageView());
    AztecViewManager::registerViewType("outline", new AztecGUI::OutlineView());
    AztecViewManager::registerViewType("graph", new AztecGUI::AztecGraphView());

    splitter->addComponent(AztecViewManager::createView("top")); 
    splitter->addComponent(AztecViewManager::createView("perspective"));
    splitter->addComponent(AztecViewManager::createView("front"));
    splitter->addComponent(AztecViewManager::createView("right"));

  }


}

