#include <Aztec3DPCH.h>
#include <tools/MEdgeDivideTool.h>

// Aztec2 includes
#include <views/Aztec3DView.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include <MSceneObject.h>
#include <MUIManager.h>
#include <MScene.h>
#include <MSystemManager.h>

// standard includes
#include <assert.h>

namespace AztecGUI {

  MEdgeDivideTool::MEdgeDivideTool()
  {
    EdgeMesh = NULL;
    EdgeNum = -1;
    edgeFrac = 0.5; // This is the distance along the edge that we will split at
    lastPointCreated = -1;
  }

  std::string MEdgeDivideTool::getName() {
    return "toolEdgeDivide";
  }

  void MEdgeDivideTool::initialise() {
    Aztec::MSystemManager::getInstance()->logOutput("Edge Divide Tool: Hold and drag the LEFT mouse button, and then release to divide an edge. Click the right mouse button while dragging to cancel the edge division.");
    lastPointCreated = -1;
  }

  bool MEdgeDivideTool::cancel() {
    if (selectedObject != NULL && EdgeMesh != NULL && EdgeNum != -1) {
      selectedObject = NULL;
      EdgeMesh = NULL;
      EdgeNum = -1;
      lastPointCreated = -1;
      return true;
    } else {
      return false;
    }

  }

  int MEdgeDivideTool::drawTool(bool select, const Aztec::MComponentPtr &component)
  {
    if (selectedObject != NULL && EdgeMesh != NULL && EdgeNum != -1) {
      int indexA, indexB;
      EdgeMesh->getVertsFromEdge(EdgeNum, &indexA, &indexB);
      assert(indexA != -1 && indexB != -1);
      
      Aztec::MVector3 A = EdgeMesh->getVertexPosition(indexA);
      Aztec::MVector3 B = EdgeMesh->getVertexPosition(indexB);

      Aztec::MVector3 pos = A + (B - A) * edgeFrac;

      Aztec::MMatrix4 objMatInv;
      
      Aztec::MVector3 worldPos = Aztec::MScene::getGlobalScene()->objectToWorldSpace(
        Aztec::MBaseObjectPtr(selectedObject), pos);

      // draw a point at our edge split position.

      // disable z-buffer testing, because we always want our point to show up.

      glDisable(GL_DEPTH_TEST);
      glEnable(GL_BLEND);

      glBegin(GL_POINTS);
      glColor3f(1.0, 0, 0);
      glVertex3fv((float*)&worldPos);
      glEnd();

      if (lastEdgeMesh == EdgeMesh && lastPointCreated != -1) {
        Aztec::MVector3 otherWorldPos = 
          Aztec::MScene::getGlobalScene()->objectToWorldSpace(
          Aztec::MBaseObjectPtr(selectedObject), EdgeMesh->getVertexPosition(lastPointCreated));

        glBegin(GL_LINES);
        glVertex3fv((float*)&worldPos);
        glVertex3fv((float*)&otherWorldPos);
        glEnd();

        glBegin(GL_POINTS);
        glColor3f(1.0, 0, 0);
        glVertex3fv((float*)&otherWorldPos);
        glEnd();
      }


      {
        Aztec3DSceneCanvasPtr GLWnd = AZTEC_CAST(Aztec3DSceneCanvas, AztecGLView::getGLCanvasFor(component));
        
        if (GLWnd == NULL) {
          return TOOLRESULT_DRAWNONE;
        }
        GLWnd->getFont()->setForegroundColor(0, 0, 0);
        
        // these postitions were just picked so they looked nice, they may appear
        // dodgy in some circumstances!
        char text[1024];

        AztecGLFont *font = GLWnd->getFont();
        float height = 14;
        
        float    Mat[16];
        glGetFloatv(GL_MODELVIEW_MATRIX, Mat);

        Aztec::MVector3 x;
        Aztec::MVector3 y;

        GLWnd->getScreenVectors(x, y);

        Aztec::MVector3 line = y * height;

        glPushMatrix();
        glTranslatef(worldPos.x, worldPos.y, worldPos.z);
        glTranslatef(3.0 * x.x, 3.0 * x.y, 3.0 * x.z);

        sprintf(text, "%.0f %%", 100.0 * edgeFrac);
        glTranslatef(3.0 * y.x, 3.0 * y.y, 3.0 * y.z);
        font->draw(0, 0, text);

        sprintf(text, "%.2f %.2f %.2f", worldPos.x, worldPos.y, worldPos.z);
        glTranslatef(line.x, line.y, line.z);
        font->draw(0, 0, text);

        glPopMatrix();

      }

      glEnable(GL_DEPTH_TEST);

    }

    return 1;
  }

  int MEdgeDivideTool::onMouseDown(const Aztec::MMouseEvent &event)
  {
    M3DToolType::onMouseDown(event);

    int result = TOOLRESULT_DRAWNONE;

    // first check to see if the right mouse button was prese
    if (event.getType() == Aztec::MMouseEvent::RBUTTON_DOWN) {
      // if we are creating a point, then use the right mouse to cancel.
      if (selectedObject != NULL && EdgeMesh != NULL && EdgeNum != -1) {
        cancel();
        result = TOOLRESULT_DRAWCURRENT;
      }

    } else if (event.getType() == Aztec::MMouseEvent::LBUTTON_DOWN) {

      // Check to see if the mouse buttons are down.
      Aztec::MUIManager::ComponentType OldMode;
  
      AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(event.getComponent());
  
      if (GLWnd == NULL) {
        return TOOLRESULT_DRAWNONE;
      }
  
      Aztec::MVector3 EdgeCentre;
      float edgeDistance;
  
      // check to see if all the mouse buttons are up
      if (!event.getShiftState().leftMouseDown) {
        return TOOLRESULT_DRAWNONE;
      }
  
      // select all the objects in the selection
  
      OldMode = Aztec::MUIManager::getComponentMode();
      Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::EDGE_TYPE);

      GLWnd->getSelection(event.getX(), event.getY(), m_InputList);

      Aztec::MUIManager::setComponentMode(OldMode);
  
      // Go through and select the required objects
      Aztec::MBaseObjectPtr Obj;
  
      for (int i = 0; i < m_InputList.size(); ++i) {
        Aztec::MSelectionItem *SelItem = &m_InputList[i];
   
        if (SelItem->getNumComponents() != 0) {
          Aztec::MSceneObjectPtr SelObj;
          Aztec::MEditableMeshPtr MeshObj;
      
          SelObj = AZTEC_CAST(Aztec::MSceneObject, SelItem->getObject());
      
          if (SelObj != NULL)
          {
            MeshObj = AZTEC_CAST(Aztec::MEditableMesh, SelObj->getShapeObject()->convertToMesh());
        
            if (MeshObj == NULL)
              continue;

            Aztec::MRay mouseRay;
            Aztec::MMatrix4 objMatInv;

            Aztec::MScene::getGlobalScene()->getWorldTransformMatrix(Aztec::MScene::getGlobalScene()->getObjectList()->findObject(SelObj), objMatInv);
            objMatInv.inverse();
            mouseRay = objMatInv * getDownRay();

            for (int n=0; n<SelItem->getNumComponents(); n++)
            {
              int   Num;
              Aztec::AztecFlags  Mode;
          
              int triangleIndex;
              int            TriEdge;
          
              SelItem->getComponent(n, Mode, Num);
          
              if (Mode != SELECTITEM_EDGE) {
                continue;
              }
          
              // Check to see if this is the closest one
          
              if (!MeshObj->getTriFromEdge(Num, triangleIndex, TriEdge)) {
                continue;
              }
          
              int indexA, indexB;
              Aztec::MVector3 Centre, distanceVec;
              float distance;
              double tempEdgeFrac;
          
              if (!MeshObj->getVertsFromEdge(Num, &indexA, &indexB)) {
                continue;
              }

              Aztec::MVector3 A = MeshObj->getVertexPosition(indexA);
              Aztec::MVector3 B = MeshObj->getVertexPosition(indexB);
              // calculate the distance from the edge to the cursor
              Centre = 0.5*( B + A );
              distanceVec = Centre - mouseRay.Org;
              Aztec::MMath::distanceBetween(mouseRay, Aztec::MRay( A, (B - A) ), NULL, &tempEdgeFrac);
          
              distance = distanceVec.length2();
              if ( tempEdgeFrac > 0.0 && tempEdgeFrac < 1.0 && (EdgeNum == -1 || distance < edgeDistance) )
              {
                EdgeMesh = MeshObj;
                selectedObject = SelObj;
                EdgeNum = Num;
            
                edgeDistance = distance;
                EdgeCentre = Centre;
                edgeFrac = tempEdgeFrac;
              }
          
            }
          }
      
        }
      }
  
      if (EdgeMesh != NULL && EdgeNum != -1) {
        result = TOOLRESULT_DRAWCURRENT;
      }

      m_InputList.clear();
    }
    return result;
  }

  int MEdgeDivideTool::onMouseMove(const Aztec::MMouseEvent &event) {
    M3DToolType::onMouseMove(event);

    if (selectedObject != NULL && EdgeMesh != NULL && EdgeNum != -1) {
      Aztec::MRay mouseRay;
      Aztec::MMatrix4 objMatInv;
      
      Aztec::MScene::getGlobalScene()->getWorldTransformMatrix(
        Aztec::MScene::getGlobalScene()->getObjectList()->findObject(selectedObject), objMatInv);
      objMatInv.inverse();
      mouseRay = objMatInv * getCurrentRay();

      int indexA, indexB;
      EdgeMesh->getVertsFromEdge(EdgeNum, &indexA, &indexB);
      assert(indexA != -1 && indexB != -1);
      
      Aztec::MVector3 A = EdgeMesh->getVertexPosition(indexA);
      Aztec::MVector3 B = EdgeMesh->getVertexPosition(indexB);
      
      Aztec::MMath::distanceBetween(mouseRay, Aztec::MRay( A, (B - A) ), NULL, &edgeFrac);
      
      if (edgeFrac < 0.0) edgeFrac = 0.0;
      if (edgeFrac > 1.0) edgeFrac = 1.0;

      return TOOLRESULT_DRAWCURRENT;
    }

    return TOOLRESULT_DRAWNONE;
  }

  int MEdgeDivideTool::onMouseUp(const Aztec::MMouseEvent &event) {
    M3DToolType::onMouseUp(event);

    int result = TOOLRESULT_DRAWNONE;

    if (EdgeMesh != NULL && EdgeNum != -1) {
      int newPoint = EdgeMesh->divideEdge(EdgeNum, edgeFrac);

      if (lastEdgeMesh == EdgeMesh && lastPointCreated != -1) {
        pointConnectMesh(lastEdgeMesh, lastPointCreated, newPoint);
      }

      lastPointCreated = newPoint;
      lastEdgeMesh = EdgeMesh;

      EdgeMesh = NULL;
      EdgeNum = -1;
      selectedObject = NULL;

      result = TOOLRESULT_DRAWALL;
    }

    return result;
  }

}


