#include <Aztec3DPCH.h>
#include <tools/MGLMoveTool.h>

// Aztec2 includes
#include <views/AztecGLView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <views/UVEditView.h>
#include <views/AztecViewManager.h>

// AztecLib includes
#include <MUIManager.h>
#include <MSystemManager.h>
#include "MEditableMesh.h"

#include <math.h>

namespace AztecGUI {

  MGLMoveToolType::MGLMoveToolType() {
    m_RequiresSel = true;
    middleManipAxis = SCREEN;
  }

  std::string MGLMoveToolType::getName() {
    return "toolMove";
  }

  bool MGLMoveToolType::cancel() {
    // if we cancel while we are dragging, then we revert the changes, ad we are not finished with the tool
    if (m_Dragging) {
      m_Dragging = false;

      {
        Aztec::MVector3 MoveVec, compCentre, compMoveVec;
        
        // Go through the currently selected objects, and move their components.
        Aztec::MBaseObjectPtr BaseObj;
        Aztec::MSceneObjectPtr Obj;
        Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
        Aztec::MEditableMeshPtr meshObj;
        
        scene->getObjectList()->beginIteration();
        while (( BaseObj = scene->getObjectList()->getNext() ) != NULL )
        {
          Obj = AZTEC_CAST(Aztec::MSceneObject, BaseObj);
          if (Obj == NULL) {
            continue;
          }
          
          Aztec::MTreeObjectNodePtr ObjNode;
          Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
          
          ObjNode = scene->getObjectList()->getCurrentNode();
          
          if (compObj != NULL && compObj->isInComponentMode()) {
            Aztec::MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
            editableCompObj->restoreComponentPositions(editableCompObj->getComponentMode());
          } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
            
            Aztec::MTransformObjectPtr XForm = Obj->getTransformObject();
            
            
            if (XForm != NULL) {
              XForm->fetchValues();
            }
          }
        }         
    
        scene->getObjectList()->endIteration();

      }


      return false;
    } else {
      return true;
    }


  }

  bool MGLMoveToolType::inProgress() {
    return m_Dragging;
  }

  int MGLMoveToolType::drawTool(bool Select, const Aztec::MComponentPtr &comp) {
    // Get the current viewport and cast it to a gl wnd.
    Aztec::MVector3  PivotPoint;
    bool DrawManips;
  
    AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(comp);

    if (GLWnd == NULL) {
      return 0;
    }

    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
  
    PivotPoint = getSelectionCentre(comp);
    // we should not update our axis if we are dragging, otherwise
    // the axis would change as we move things!
    if (!m_Dragging) {
      getAxisMatrix(GLWnd);
    }
  
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    glPushMatrix();
    float ScaleFact = GLWnd->getScalingFactor(PivotPoint);
    glTranslatef(PivotPoint.x, PivotPoint.y, PivotPoint.z);
    glMultMatrixf((float*)m_AxisTransform.m);
  
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    if (RequiresSelection()) {
      DrawManips = anythingSelected(comp);
    } else {
      DrawManips = true;
    }
  
    if (DrawManips) {
      switch (middleManipAxis) {
      case SCREEN:
        glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, Select, DRAWAXIS_ALLAXES | DRAWAXIS_ARROWS);
        break;
      case XAXIS:
        glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, Select, DRAWAXIS_ALLAXES | DRAWAXIS_ARROWS | DRAWAXIS_MIDDLE_NORMAL_X);
        break;
      case YAXIS:
        glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, Select, DRAWAXIS_ALLAXES | DRAWAXIS_ARROWS | DRAWAXIS_MIDDLE_NORMAL_Y);
        break;
      case ZAXIS:
        glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, Select, DRAWAXIS_ALLAXES | DRAWAXIS_ARROWS | DRAWAXIS_MIDDLE_NORMAL_Z);
        break;
      }
    
      if (m_Dragging) {
        glPopMatrix();
        glPushMatrix();
        ScaleFact = GLWnd->getScalingFactor(m_MoveStart);
        glTranslatef(m_MoveStart.x, m_MoveStart.y, m_MoveStart.z);
        glScalef(ScaleFact, ScaleFact, ScaleFact);
        glMultMatrixf((float*)m_AxisTransform.m);
        
        glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.2f, 2.0f, false, DRAWAXIS_ALLGRAY | DRAWAXIS_NORMAL | DRAWAXIS_ARROWS);
      
      }
    }
  
    glPopMatrix();
    glPopMatrix();
  
    glPopAttrib();
  
    return 1;
  }

  int MGLMoveToolType::onMouseDown(const Aztec::MMouseEvent &event)
  {
    AztecGLCanvasPtr canvas = AZTEC_CAST(AztecGLCanvas, event.getComponent());

    Aztec::MVector3 customCentre;
    // if we are holding down control, that means we are adjusting our middle
    // manipulator axis
    if (event.getShiftState().ctrlPressed) {
      Aztec::MRay mouse = canvas->getRay(event.getX(), event.getY());
      Aztec::MRay axis;
      axis.Org = getSelectionCentre(event.getComponent());

      ManipAxis oldAxis = middleManipAxis;
      switch (m_PickedManip) {
      case 1:
        middleManipAxis = XAXIS;
        axis.Dir = m_AxisTransform*Aztec::MVector3(1,0,0);
        break;
      case 2:
        middleManipAxis = YAXIS;
        axis.Dir = m_AxisTransform*Aztec::MVector3(0,1,0);
        break;
      case 3:
        middleManipAxis = ZAXIS;
        axis.Dir = m_AxisTransform*Aztec::MVector3(0,0,1);
        break;
      case 4:
        middleManipAxis = SCREEN;
        break;
      }
      m_PickedManip = 4;


      double axisDist = 0.0;
      Aztec::MMath::distanceBetween(axis, mouse, &axisDist);
      customCentre = axis.Org + axis.Dir * axisDist;
    }

    switch (middleManipAxis) {
    case SCREEN:
      getPlaneParams(canvas, m_Constraint, m_AxisTransform);
      break;
    case XAXIS:
      if (m_PickedManip == 4) {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, customCentre, DRAWAXIS_MIDDLE_NORMAL_X);
      } else {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, DRAWAXIS_MIDDLE_NORMAL_X);
      }
      break;
    case YAXIS:
      if (m_PickedManip == 4) {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, customCentre, DRAWAXIS_MIDDLE_NORMAL_Y);
      } else {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, DRAWAXIS_MIDDLE_NORMAL_Y);
      }
      break;
    case ZAXIS:
      if (m_PickedManip == 4) {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, customCentre, DRAWAXIS_MIDDLE_NORMAL_Z);
      } else {
        getPlaneParams(canvas, m_Constraint, m_AxisTransform, DRAWAXIS_MIDDLE_NORMAL_Z);
      }
      break;
    }
  
    m_MoveStart = getSelectionCentre(event.getComponent());

    MXYZToolType::onMouseDown(event);

    return TOOLRESULT_DRAWALL;
  }


  int MGLMoveToolType::onMouseUp(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseUp(event);
  
    endTransform(event.getComponent());
  
    return TOOLRESULT_DRAWALL;
  }

  int MGLMoveToolType::onMouseMove(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseMove(event);
  
    if (!m_Dragging) {
      Aztec::MUIManager::unsetWorldCoordinates();
      Aztec::MUIManager::unsetDiffCoordinates();
      return TOOLRESULT_DRAWNONE;
    }
  
    Aztec::MVector3 OfsVec = m_CurVec;
    OfsVec -= m_DownVec;
  
    Aztec::MUIManager::setWorldCoordinates(m_DownVec);
    Aztec::MUIManager::setDiffCoordinates(OfsVec);

    transformSelection(event.getComponent(), OfsVec);

  
    return TOOLRESULT_DRAWALL;

  }

  int MGLMoveToolType::getDefaultManip() {
    return 4;
  }

  void MGLMoveToolType::transformSelection(const Aztec::MComponentPtr &component, const Aztec::MVector3 &worldOffset) {
    AztecViewPtr view = AztecView::getViewForComponent(component);
    UVEditViewPtr uv = AZTEC_CAST(UVEditView, view);

    if (uv != NULL) {
      transformSelectionUV(worldOffset);
    } else {
      transformSelection3D(worldOffset);
    }
  }

  void MGLMoveToolType::endTransform(const Aztec::MComponentPtr &component) {
    if (m_PickedManip == 1) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSX);
    } else if (m_PickedManip == 2) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSY);
    } else if (m_PickedManip == 3) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSZ);
    } else {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSLATE);
    } 
  }

  class GridSnapper : public Aztec::MEditableComponentisedObject::TransformModifier {
  public:
    GridSnapper(float snapAmount) {
      gridSpacing = snapAmount;
    }
    // TransformModifier methods
    static inline float round(float a, float spacing) {
      a = a / spacing;
      if (a < 0) a -= 0.5;
      if (a > 0) a += 0.5;
      return ((int)a) * spacing;
    }
    bool adjustTransform(const Aztec::MVector3 &originalPos, Aztec::MVector3 &newPos) {
      newPos.x = round(newPos.x, gridSpacing);
      newPos.y = round(newPos.y, gridSpacing);
      newPos.z = round(newPos.z, gridSpacing);
      return true;
    }

    float gridSpacing;
  };

  void MGLMoveToolType::transformSelection3D(const Aztec::MVector3 &worldOffset) {
    Aztec::MVector3 MoveVec, compCentre, compMoveVec;
	
    // Go through the currently selected objects, and move their components.
    Aztec::MBaseObjectPtr BaseObj;
    Aztec::MSceneObjectPtr Obj;
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
	Aztec::MEditableMeshPtr meshObj;

	float gridSpacing = 1;
	bool useComponentCP = 1;
	
    AztecGLView *view = AZTEC_CAST(AztecGLView, AztecViewManager::getCurrentView());
    if (view != NULL) {
	  gridSpacing = view->getCanvas()->getGridSpacing();
	  useComponentCP = view->getCanvas()->getSnapViaComponentCP(); 
	}

	scene->getObjectList()->beginIteration();
    while (( BaseObj = scene->getObjectList()->getNext() ) != NULL )
    {
      Obj = AZTEC_CAST(Aztec::MSceneObject, BaseObj);
      if (Obj == NULL) {
        continue;
      }
      
      Aztec::MTreeObjectNodePtr ObjNode;
      Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
      
      ObjNode = scene->getObjectList()->getCurrentNode();
      
      if (compObj != NULL && compObj->isInComponentMode()) {
        Aztec::MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
        editableCompObj->restoreComponentPositions(editableCompObj->getComponentMode());
        //This bit does Component movement...IE Vertices.
        if (editableCompObj != NULL && editableCompObj->isInComponentMode()) {
          {
            Aztec::MTransformObjectPtr XForm = scene->getTransform(ObjNode);
            
            if (XForm != NULL) {
              MoveVec = scene->worldToObjectSpace(ObjNode, worldOffset);
              MoveVec -= scene->worldToObjectSpace(ObjNode, Aztec::MVector3(0,0,0));
              if (useComponentCP && Aztec::MUIManager::getSnapMode() == Aztec::MUIManager::SNAP_GRID) {
                
                MoveVec.x = floor(MoveVec.x);
                MoveVec.y = floor(MoveVec.y);
                MoveVec.z = floor(MoveVec.z);
                compCentre = compObj->getFlaggedCentre(editableCompObj->getComponentMode());
                
                MoveVec.x -= fmod((MoveVec.x + compCentre.x), gridSpacing);
                MoveVec.y -= fmod((MoveVec.y + compCentre.y), gridSpacing);
                MoveVec.z -= fmod((MoveVec.z + compCentre.z), gridSpacing);
                
              }
            }
            
          }
          
          Aztec::MEditableComponentisedObject::TransformModifier *modifier = NULL;
          GridSnapper snapper(gridSpacing);
          
          if (!useComponentCP && Aztec::MUIManager::getSnapMode() == Aztec::MUIManager::SNAP_GRID) {
            modifier = &snapper;
          }
          
          
          Aztec::MVector3 axisMoveVec = m_AxisTransformInv * worldOffset;
          bool doLocalTransform = Aztec::MUIManager::getSeparateAxisMode() && Aztec::MUIManager::getAxisMode() == Aztec::MUIManager::LOCAL_AXIS;
          Aztec::MMatrix4 transform;
          
          transform.identity();
          
          if (doLocalTransform) {
            transform.m[3][0] = axisMoveVec.x;
            transform.m[3][1] = axisMoveVec.y;
            transform.m[3][2] = axisMoveVec.z;
          } else {
            transform.m[3][0] = MoveVec.x;
            transform.m[3][1] = MoveVec.y;
            transform.m[3][2] = MoveVec.z;
          }
          
          
          editableCompObj->transformComponents(
            editableCompObj->getComponentMode(), 
            transform, 
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            doLocalTransform,modifier);
        }
      } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        //This bit does Object movement...
        {
          Aztec::MTreeObjectNodePtr ParentNode;
          ParentNode = ObjNode->getParent();
          
          MoveVec = scene->worldToObjectSpace(ParentNode, m_CurVec);
          MoveVec -= scene->worldToObjectSpace(ParentNode, m_DownVec);
        }
        
        
        Aztec::MTransformObjectPtr XForm = Obj->getTransformObject();
        Aztec::MVector3          Vec;
        
        
        if (XForm != NULL) {
          Aztec::MVector3 Vec;
          
          XForm->fetchValues();
          Vec = XForm->getTranslateVector(scene->getTime());
          
          if (Aztec::MUIManager::getSnapMode() == Aztec::MUIManager::SNAP_GRID) {
            MoveVec.x -= fmod(MoveVec.x, gridSpacing);
            MoveVec.y -= fmod(MoveVec.y, gridSpacing);
            MoveVec.z -= fmod(MoveVec.z, gridSpacing);
            Vec.x -= fmod(Vec.x, gridSpacing);
            Vec.y -= fmod(Vec.y, gridSpacing);
            Vec.z -= fmod(Vec.z, gridSpacing);
          }
          
          XForm->setTranslateVector(Vec + MoveVec);
          XForm->setParamByName("Tsl", (Vec+MoveVec).convertToString(), false);
        }
      }
    }         

    scene->getObjectList()->endIteration();

    if (m_PickedManip == 1) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSX, false);
    } else if (m_PickedManip == 2) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSY, false);
    } else if (m_PickedManip == 3) {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSZ, false);
    } else {
      UpdateKeys("Move", TRANSFORM_CHAN_TRANSLATE, false);
    } 
  }


  void MGLMoveToolType::transformSelectionUV(const Aztec::MVector3 &worldOffset) {
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

    scene->getSelectedObjectList()->beginIteration();
    
    Aztec::MBaseObjectPtr Obj;
    while ((Obj = scene->getSelectedObjectList()->getNext()) != NULL) {
      Aztec::MSceneObjectPtr sceneObj;
      if ((sceneObj = AZTEC_CAST(Aztec::MSceneObject, Obj)) == NULL) {
        continue;
      }
      Aztec::MMeshPtr Mesh;
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(Aztec::MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
      
      if (Mesh != NULL) {
        
        Aztec::MMeshPtr TextureMesh = Mesh->getTextureMesh();
        Aztec::MEditableComponentisedObjectPtr editableCompObj = AZTEC_CAST(Aztec::MEditableComponentisedObject, TextureMesh);
        
        if (editableCompObj != NULL) {
          editableCompObj->restoreComponentPositions(Aztec::MUIManager::getComponentMode());
          
          Aztec::MVector3 axisMoveVec = m_AxisTransformInv * worldOffset;
          
          bool doLocalTransform = Aztec::MUIManager::getSeparateAxisMode() && Aztec::MUIManager::getAxisMode() == Aztec::MUIManager::LOCAL_AXIS;
          Aztec::MMatrix4 transform;
          
          transform.identity();
          
          if (doLocalTransform) {
            transform.m[3][0] = axisMoveVec.x;
            transform.m[3][1] = axisMoveVec.y;
            transform.m[3][2] = axisMoveVec.z;
          } else {
            transform.m[3][0] = worldOffset.x;
            transform.m[3][1] = worldOffset.y;
            transform.m[3][2] = worldOffset.z;
          }
          
          editableCompObj->transformComponents(
            Aztec::MUIManager::getComponentMode(), 
            transform, 
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            doLocalTransform);
        }
      }
    }
    scene->getSelectedObjectList()->endIteration();

  }

}



