#include <Aztec3DPCH.h>
#include <tools/MGLScaleTool.h>

// Aztec2 includes
#include <views/AztecGLView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>

// AztecLib includes
#include <MUIManager.h>
#include <MSystemManager.h>

// standard includes
#include <math.h>

namespace AztecGUI {

  // This function takes the current gl matrix, and normalises the three main vectors, so
  // scaling does not effect drawing
  static void NormaliseGLMatrix(const Aztec::MVector3 &ScaleVec) {
    float    Mat[16];
    Aztec::MVector3 XVec, YVec, ZVec;
  
    glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
  
    XVec.set(Mat[0], Mat[1], Mat[2]);
    YVec.set(Mat[4], Mat[5], Mat[6]);
    ZVec.set(Mat[8], Mat[9], Mat[10]);
  
    XVec.normalize();
    YVec.normalize();
    ZVec.normalize();
  
    XVec *= (float)fabs(ScaleVec.x);
    YVec *= (float)fabs(ScaleVec.y);
    ZVec *= (float)fabs(ScaleVec.z);
  
    Mat[0] = XVec.x; Mat[1] = XVec.y; Mat[2] = XVec.z;
    Mat[4] = YVec.x; Mat[5] = YVec.y; Mat[6] = YVec.z;
    Mat[8] = ZVec.x; Mat[9] = ZVec.y; Mat[10] = ZVec.z;
  
    glLoadMatrixf(Mat);
  }


  //--------------------
  //  MGLScaleToolType
  //--------------------
  MGLScaleToolType::MGLScaleToolType()
  {
    m_RequiresSel = true;
    m_ScaleVec.set(1,1,1);
  }

  std::string MGLScaleToolType::getName() {
    return "toolScale";
  }

  int MGLScaleToolType::drawTool(bool Select, const Aztec::MComponentPtr &comp)
  {
    // Get the current viewport and cast it to a gl wnd.
    bool           DrawManips;
  
  
    AztecGLViewPtr viewGL = AZTEC_CAST(AztecGLView, comp);

    if (viewGL == NULL) {
      return 0;
    }
  
    AztecGLCanvasPtr GLWnd = viewGL->getCanvas();
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
  
    if (!m_Dragging) {
      m_PivotPoint = scene->getSelectionCentre();
      getAxisMatrix(GLWnd);
    }
  
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    float ScaleFact = GLWnd->getScalingFactor(m_PivotPoint);
    glTranslatef(m_PivotPoint.x, m_PivotPoint.y, m_PivotPoint.z);
    glMultMatrixf((float*)m_AxisTransform.m);
  
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    if (RequiresSelection()) {
      DrawManips = scene->anythingSelected();
    } else {
      DrawManips = true;
    }
  
    if (DrawManips) {
      glDrawScaleIcon(UIConfig::get3DWidgetSize(),0.5f,1, Select);
    }
  
    glPopMatrix();
  
    glPopAttrib();
    return 1;
  }

  int MGLScaleToolType::onMouseDown(const Aztec::MMouseEvent &event)
  {
  
    Aztec::MVector3 Orig, Dir, ViewNorm;
    Aztec::MMatrix4 XFormMat;
  
    {
      AztecGLCanvasPtr GLWnd = AZTEC_CAST(AztecGLCanvas, event.getComponent());
      
      if (GLWnd != NULL)
      {
      
        GLWnd->getInverseTransform().transform(Aztec::MVector3(1,0,0),m_RightVec);
      
        m_RightVec.x = m_RightVec.x;
        m_RightVec.y = m_RightVec.y;
        m_RightVec.z = m_RightVec.z;
      }
    }
    getPlaneParams(AZTEC_CAST(AztecGLCanvas, event.getComponent()), m_Constraint, m_AxisTransform);
  
    m_ScaleVec.set(1,1,1);
  
    MXYZToolType::onMouseDown(event);

    return TOOLRESULT_DRAWALL;
  }

  int MGLScaleToolType::onMouseUp(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseUp(event);
  
    UpdateKeys("Scale", TRANSFORM_CHAN_SCALE);
  
    m_ScaleVec.set(1,1,1);
  
    return TOOLRESULT_DRAWALL;
  }

  int MGLScaleToolType::onMouseMove(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseMove(event);
  
    if (!m_Dragging) {
      return TOOLRESULT_DRAWNONE;
    }
  
    Aztec::MVector3 OfsVec, N;
  
    OfsVec = m_CurVec;
    OfsVec -= m_DownVec;
  
    N.set(1,1,1);
    if (m_PickedManip == 1) { // X Axis
      N.y = 0;
      N.z = 0;
      if ((m_AxisTransform * Aztec::MVector3(1,0,0)).x < 0) {
        N.x *= -1;
      }
    } else if (m_PickedManip == 2) { // Y Axis
      N.x = 0;
      N.z = 0;
      if ((m_AxisTransform * Aztec::MVector3(0,1,0)).y < 0) {
        N.y *= -1;
      }
    } else if (m_PickedManip == 3) { // Z Axis
      N.x = 0;
      N.y = 0;
      if ((m_AxisTransform * Aztec::MVector3(0,0,1)).z < 0) {
        N.z *= -1;
      }
    } else if (m_PickedManip == 4) {
      float    Dist;
      Dist = OfsVec * m_RightVec;
      OfsVec.x = Dist;
      OfsVec.y = OfsVec.x;
      OfsVec.z = OfsVec.x;
    }
  
    m_ScaleVec.x = (float)(1 + 0.1 * OfsVec.x * N.x);
    m_ScaleVec.y = (float)(1 + 0.1 * OfsVec.y * N.y);
    m_ScaleVec.z = (float)(1 + 0.1 * OfsVec.z * N.z);

    Aztec::MMatrix4 localScaleXForm, scaleXForm;

    localScaleXForm.identity();
    localScaleXForm.m[0][0] = m_ScaleVec.x;
    localScaleXForm.m[1][1] = m_ScaleVec.y;
    localScaleXForm.m[2][2] = m_ScaleVec.z;

    scaleXForm = m_AxisTransformInv * localScaleXForm * m_AxisTransform;
  //  scaleXForm = localScaleXForm;

    // Go through the currently selected objects, and move their components.
    Aztec::MBaseObjectPtr BaseObj;
    Aztec::MSceneObjectPtr Obj;
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
    scene->getObjectList()->beginIteration();
    while (( BaseObj = scene->getObjectList()->getNext() ) != NULL ) {
      Obj = AZTEC_CAST(Aztec::MSceneObject, BaseObj);
      if (Obj == NULL) {
        continue;
      }
    
      Aztec::MTreeObjectNodePtr ObjNode;
    
      ObjNode = scene->getObjectList()->getCurrentNode();
      Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
      if (compObj != NULL && compObj->isInComponentMode()) {
        Aztec::MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
        if (editableCompObj != NULL && editableCompObj->isInComponentMode()) {
          editableCompObj->restoreComponentPositions(editableCompObj->getComponentMode(), Aztec::MComponentisedObject::COMPONENT_SELECTED);

          Aztec::MMatrix4 objToWorld;
          Aztec::MMatrix4 worldToObj;
          scene->getWorldTransformMatrix(ObjNode, objToWorld);
          worldToObj = objToWorld;
          worldToObj.inverse();

          Aztec::MMatrix4 transform;
          transform.identity();

          Aztec::MVector3 localPivot = worldToObj * m_PivotPoint;

          worldToObj.m[3][0] = 0;
          worldToObj.m[3][1] = 0;
          worldToObj.m[3][2] = 0;
          worldToObj.normaliseRotation();
          objToWorld.m[3][0] = 0;
          objToWorld.m[3][1] = 0;
          objToWorld.m[3][2] = 0;
          objToWorld.normaliseRotation();

          transform.m[3][0] -= localPivot.x;
          transform.m[3][1] -= localPivot.y;
          transform.m[3][2] -= localPivot.z;

          transform *= objToWorld  * scaleXForm * worldToObj;// * worldToObj;

          transform.m[3][0] += localPivot.x;
          transform.m[3][1] += localPivot.y;
          transform.m[3][2] += localPivot.z;

          editableCompObj->transformComponents(
            editableCompObj->getComponentMode(),
            transform,
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            false);

        }
      } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        Aztec::MTransformObjectPtr XForm;
        Aztec::MVector3          Vec;
      
        XForm = scene->getTransform(scene->getObjectList()->getCurrentNode());
      
        if (XForm != NULL) {
          Aztec::MVector3       Vec;
        
          XForm->fetchValues();
          Vec = XForm->getScaleVector(Obj->getTime());
          Vec.x *= m_ScaleVec.x;
          Vec.y *= m_ScaleVec.y;
          Vec.z *= m_ScaleVec.z;
          XForm->setScaleVector(Vec);
        
          XForm->setParamByName("Scl", XForm->getScaleVector(Obj->getTime()).convertToString(), false);
        }
      }
    }         
    scene->getObjectList()->endIteration();

    UpdateKeys("Scale", TRANSFORM_CHAN_SCALE, false);
  
    return TOOLRESULT_DRAWALL;
  }

  int MGLScaleToolType::getDefaultManip() {
    return 4;
  }

}
