#ifndef AztecGUI_MParameterPlugin_Header
#define AztecGUI_MParameterPlugin_Header


#include <AztecGUICommon.h>

#include <MRefCountedPtr.h>

namespace AztecGUI {

  /**
   * This is the base class for all GUI related extensions to Aztec and 
   * parameters. When an object is selected, it is possible to bring up a 
   * Parameter editor in Aztec to allow a more user friendly and object
   * specific way of editing parameters than the Channel Bar.
   *
   * Each plugin is associated with a specific object type, which is queried
   * using the isUIForObject() method. This function should return true if an 
   * AZTEC_CAST to the known object type is successful.
   *
   * When a query is successful, the UI for aztec will call the createControls()
   * function for the plugin to create the various controls for the UI. The 
   * parent window passed in will be an Empty CWnd that the child should resize
   * to whatever size it needs. If the controls will be dynamic, that is there 
   * are some controls that are only visible at vertain times, the plugin 
   * controls that are inserted should alter the parent window's size to match.
   *
   * The controls that are created are responsible for updating everything 
   * themselves, and keeping track of parameter changes that might be made
   * outside of the Parameter Editor. There is a broad scale notifiction that
   * is executed via the updateGUIValues() method. The plugin can choose to
   * ifnore that if it can keep rtack of changes all by itself.
   */
  class AZTECGUI_EXPORT MParameterPlugin : public Aztec::MRefCountedObject {
  public:
    /**
     * Gets the friendly name to display as the group of this GUIPlugin
     */
    virtual std::string getFriendlyName() = 0;

    /**
     * Returns true if this GUI Plugin handles editing the given object or not
     */
    virtual bool isUIForObject(const Aztec::MBaseObjectPtr &obj) = 0;

    /**
     * This creates all the controls that will edit the given object.
     */
    virtual void createControls(HWND parentWnd, const Aztec::MBaseObjectPtr &object) = 0;

    /**
     * Tells the plugin to destroy any window components it created and free
     * up any memory used.
     */
    virtual void destroyControls() = 0;

    /**
     * This routine should return true if the plugin can create editing controls that
	 * are placed in the channel bar along with the default parameter editing controls.
	 * Most simple parameters are handled with default editors by Aztec, so special
	 * channel bar editors should be the exception.  By default this routine should
	 * return false.
     */
    virtual bool hasChannelControls();

	/**
	 * If there are channel bar controls, then we Aztec will need to know how much
	 * vertical space will be occupied.  The plugin is responsible for computing this.
	 * TBD: the channel bar currently has a fixed width, so the controls have to fit
	 * that size.  in the future, Aztec will probably need to know the width as well.
	 */
	virtual int getChannelControlsHeight();

    /**
     * This creates all the controls for the channel bar that will edit the given object.
	 * By default, this routine should do nothing - normally a plugin will only create Controls
	 * that appear in the Parameter window.
     */
    virtual void createChannelControls(HWND parentWnd, const Aztec::MBaseObjectPtr &object);

    /**
     * Tells the plugin to destroy any window components it created for the channel
	 * bar and free up any memory used.
     */
    virtual void destroyChannelControls();

    /**
     * Revert any changes that has been made over this UI's lifetime.
     */
    virtual void cancelChanges() = 0;

    /**
     * This tells the GUI Plugin that it should update its values from the 
     * object, as things may have changed. The plugin can choose to ignore it
     * if it wishes.
     */
    virtual void updateGUIValues(const Aztec::MBaseObjectPtr &object) = 0;

  protected:
    MParameterPlugin();
    virtual ~MParameterPlugin();
  };

  typedef Aztec::MRefCountedPtr<MParameterPlugin> MParameterPluginPtr;

  /**
   * This is the function typedef that each GUI Plugin DLL must export, in
   * an undecorated name. It should look like this:
   *
   * MParameterPluginPtr getGUIPlugin(int);
   */
  typedef MParameterPluginPtr (*GetGUIPluginFunc)(int);

  /**
   * This is the function typedef that each GUI Plugin DLL must export, in
   * an undecorated name. It should look like this:
   *
   * int getGUIPluginCount();
   */
  typedef int (*GetGUIPluginCountFunc)();

}


#endif