#ifndef MApplication_Header
#define MApplication_Header

#include "MRefCountedPtr.h"

namespace Aztec {

  class IMApp;
  class MApp;

  typedef MRefCountedPtr<IMApp> IMAppPtr;
  typedef MRefCountedPtr<MApp> MAppPtr;

}

#include <gui/MGuiBase.h>
#include <gui/MWindow.h>
#include <string>

namespace Aztec {

  class AZTECGUI_EXPORT IMApp : public virtual IMGuiBase {
  protected:
    IMApp() { };
    virtual ~IMApp() { };
  
  public:
    typedef IMAppPtr Ptr;

    /**
     * This initialises the application. Here we initialise any variables
     * and create any windows needed to run.
     *
     * @return true if initialisation succeeds, false if it fails.
     */
    virtual bool initApp() = 0;

    /**
     * This runs the application.
     *
     * @return the exit code for the application
     */
    virtual int run() = 0;

    /**
     * Cleans up the application
     */
    virtual void cleanup() = 0;

    /**
     * Gets the application's name
     *
     * @return the App's name
     */
    virtual std::string getName() = 0;

    /**
     * This gets the directory that the application was started in.
     */
    virtual std::string getApplicationPath() = 0;

    /**
     * This gets the MWindow that is the main window for the application.
     */
    virtual MWindowPtr getTopLevelWindow() = 0;

    /**
     * This is called whenver a menu, button or similar control is executed. This is called BEFORE any onCommand() component specific method is called.
     *
     * @param command The command that was executed.
     * @param The component that was the source of the command. Sometimes 
     *        this may be null if dealing with menus. Not garaunteed to be 
     *        non-null at all.
     * @return True if it has been handled, and the command shouldn't be passed on to components.
     */
    virtual bool onCommand(const std::string &command, const MComponentPtr &component) = 0;

    /**
     * This is a global mouse event handling function that is called before 
     * any component specific method is called. This allows an application to 
     * provide a very broad similar response to any mouse event.
     *
     * @return true if the event was handled, and should not be processed by 
     *         the component generating the event.
     */
    virtual bool onMouseMove(const MMouseEvent &event) = 0;

    /**
     * This is a global mouse event handling function that is called before 
     * any component specific method is called. This allows an application to 
     * provide a very broad similar response to any mouse event.
     *
     * @return true if the event was handled, and should not be processed by 
     *         the component generating the event.
     */
    virtual bool onMousePressed(const MMouseEvent &event) = 0;

    /**
     * This is a global mouse event handling function that is called before 
     * any component specific method is called. This allows an application to 
     * provide a very broad similar response to any mouse event.
     *
     * @return true if the event was handled, and should not be processed by 
     *         the component generating the event.
     */
    virtual bool onMouseReleased(const MMouseEvent &event) = 0;

    /**
     * This is called when a key has been pressed in a component, or if a key 
     * has been held down, and the keystroke is being repeated.
     */
    virtual bool onKeyPressed(const MKeyEvent &event) = 0;

    /**
     * This is called when a key has been released on the keyboard.
     */
    virtual bool onKeyReleased(const MKeyEvent &event) = 0;
  };

}
  

/**
 * The following header is included via the AZTEC_GUI
 * variable that should be defined for each platform
 * and used to specify an include directory
 * (ie. ../AztecGUICommon/include/gui/$(AZTEC_GUI)
 * when compiling this lib
 */
#include <MAppImpl.h>

#endif

