#ifndef MComponentImpl_Header
#define MComponentImpl_Header

#include <MRefCountedPtr.h>

#include <string>
#include <vector>

#include <windows.h>

namespace Aztec {

  class MComponent;
  typedef MRefCountedPtr<MComponent> MComponentPtr;

  class MContainer;
  typedef MRefCountedPtr<MContainer> MContainerPtr;

  class AZTECGUI_EXPORT MComponent : public virtual Aztec::IMComponent {
  public:
    MComponent();
    MComponent(const std::string &name);
    ~MComponent();

    HWND getHWnd() { return m_hWnd; }

    /**
     * This creates the implementation window
     */
    virtual bool createImpl();

    virtual void destroyImpl();

    /// Handles window messages
    virtual bool wndProc(UINT message, WPARAM wParam, LPARAM lParam, LRESULT *result);

    /// Provides custom handling of WM_COMMAND notify messages
    virtual bool handleWMCommandNotify(int notifyCode, int id);
    virtual bool handleWMNotify(WPARAM wParam, LPARAM lParam);

    virtual bool preTranslateMessage(MSG *msg);

    /**
     * This gets the HBRUSH that is used to fill in the background for the 
     * edit box. This is cached and only created when necessary. If the edit 
     * box doens't have a custome backgronud colour, this returns NULL.
     *
     * @return NULL if no background colour is set, or a HBRUSH for the brush to use.
     */
    HBRUSH getBackgroundBrush();


    // MComponent methods.
    void setName(const std::string name);
    std::string getName();
    void setParent(MContainerPtr parent);
    MContainerPtr getParent();
    void setVisible(bool visible);
    void setPosition(const MPoint2D &pos);
    void setPosition(int X, int Y);
    void setPosition(int X, int Y, int width, int height);
    MPoint2D getPosition();
    void setSize(const MSize2D &size);
    void setSize(int width, int height);
    MRect2D getWindowRect();
    void setWindowRect(const MRect2D &rect);
    MSize2D getSize();
    MSize2D getClientSize();
    MSize2D getMinimumSize();
    void setMinimumSize(const MSize2D &size);
    bool isMinimumSizeSet();
    MSize2D getPreferredSize();
    void setPreferredSize(const MSize2D &size);
    bool isPreferredSizeSet();
    void refresh();
    void setEnabled(bool enabled);
    void setMouseCapture(bool capturing);
    void setMouseCursor(MMouseCursor cursor);
    void setBackgroundColour(const MColour &colour);
    void setForegroundColour(const MColour &colour);

    bool onMouseMove(const MMouseEvent &event);
    bool onMousePressed(const MMouseEvent &event);
    bool onMouseReleased(const MMouseEvent &event);
    bool onMouseDoubleClick(const MMouseEvent &event);
    bool onMouseWheel(const MMouseEvent &event);

    bool onResize(int newWidth, int newHeight);
    bool onShow();
    bool onPaint();
    void onCreate();

    bool onKeyPressed(const MKeyEvent &event);
    bool onKeyReleased(const MKeyEvent &event);

    void addMouseEventListener(const MMouseEventListenerPtr &listener);
    void addKeyEventListener(const MKeyEventListenerPtr &listener);

    HDC paintDC;


    static bool isVKKeyDown(int virtKeyCode);
    static MShiftState MComponent::getStateFromAsync();
    static MKeyEvent::KeyCode getKeyCodeFromVK(int vk);
    static int getVKFromKeyCode(MKeyEvent::KeyCode code);
    static MKeyEvent::KeyCode getKeyCodeFromChar(TCHAR ch);

    bool handleKeyEvent(const MKeyEvent &event);


  protected:
    HWND m_hWnd;

    std::string m_Name;

    /**
     * This is the parent to this component. Note
     * we do not use a reference counted pointer here
     * because of the circular reference this would create.
     */
    MContainer *m_Parent;

    /**
     * This is the preferred size of the component
     */
    MSize2D m_PreferredSize;
    
    /**
     * This is the minimum size of the component
     */
    MSize2D m_MinimumSize;

    HBRUSH bgBrush;

    bool bgColourSet;
    MColour bgColour;
    bool fgColourSet;
    MColour fgColour;

    std::vector<MMouseEventListenerPtr> mouseListeners;
    std::vector<MKeyEventListenerPtr> keyListeners;

    void fireMouseListeners(const MMouseEvent &event);
    void fireKeyListeners(const MKeyEvent &event);

  };

}


#endif
