#include <AztecGUICommonPCH.h>

#include <config/UIConfig.h>
#include <gui/MApplication.h>

#include <scripting/MScriptInterp.h>

#include <fstream>

namespace AztecGUI {

  typedef std::pair<std::string, Aztec::MShiftState> ViewShiftPair; 
  typedef std::map<ViewShiftPair, std::string> QuickToolMap;

  typedef std::map<Colours::Colour, Aztec::MVector4> ColourMap;
  ColourMap colours;

  static bool configLoaded = false;

  static QuickToolMap quickTools;

  static bool g_ChangesMade = false;

  static std::string currentFilename ;

  static float g_WidgetSize = 20.0f;

  static const char * SEP = "\t";

  static std::string getConfigFile() {
    std::string path = Aztec::MApp::getInstance()->getApplicationPath();
    // TODO: make this better, and more general

    std::string configFile = path + 
      // TODO: Make this cleaner foo!
#ifdef _WIN32
      "\\"
#else
      "/"
#endif
      + "userConfig.config";

    return configFile;
  }

  static bool toBool(const std::string &value) {
    if (value == "true") return true;
    if (value == "false") return false;
    return true;
  }

  static float toFloat(const std::string &value) {
    return (float)atof(value.c_str());
  }

  static void setupDefaultColours() {
    colours[Colours::VIEW_BACKGROUND].set(0.4f, 0.4f, 0.4f, 0.0f);
    colours[Colours::VIEW_TITLE].set(0.0f, 0.0f, 0.0f);
    colours[Colours::VIEW_TITLE_CURRENT].set(1.0f, 1.0f, 0.5f);
    colours[Colours::VIEW_BORDER].set(0.4f, 0.4f, 0.4f, 0.0f); // set to 100% transparent, so we just use the default flat 3d colour used by controls.
    colours[Colours::VIEW_BORDER_CURRENT].set(1.0f, 1.0f, 0.5f, 0.5f);
    
    colours[Colours::GRID3D_MAJOR].set(0.25f, 0.25f, 0.25f);
    colours[Colours::GRID3D_MINOR].set(0.35f, 0.35f, 0.35f);
    colours[Colours::GRID3D_AXIS].set(0.0f, 0.0f, 0.0f);
    
  }

  static void setupDefaultKeys() {
    const Aztec::MShiftState CONTROL_DOWN(false, false, false, false, true, false);
    const Aztec::MShiftState CONTROL_SHIFT_DOWN(false, false, false, false, true, true);
    
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Q, "Scene.toolSetCurrent('3D', 'toolSelect')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_W, "Scene.toolSetCurrent('3D', 'toolMove')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_E, "Scene.toolSetCurrent('3D', 'toolRotate')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_R, "Scene.toolSetCurrent('3D', 'toolScale')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Escape, "Scene.toolCancel()");

    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Space, "Scene.viewToggleFullscreen()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_F, "Scene.zoomToFitSelected()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_A, "Scene.zoomToFitAll()");

    UIConfig::addShortcut(Aztec::MKeyEvent::Key_1, "Scene.componentModeSet('object')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_2, "Scene.componentModeSet('point')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_3, "Scene.componentModeSet('facet')");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_4, "Scene.componentModeSet('edge')");

    UIConfig::addShortcut(Aztec::MKeyEvent::Key_BracketLeft, "Scene.selectEdgeLoop()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_BracketRight, "Scene.selectEdgeRing()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Semicolon, "Scene.selectShrink()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Slash, "Scene.selectNone()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_G, "Scene.componentConnect()");

    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Z, CONTROL_DOWN, true, "Scene.undo()");
    UIConfig::addShortcut(Aztec::MKeyEvent::Key_Z, CONTROL_SHIFT_DOWN, true, "Scene.redo()");
  
  }

  void UIConfig::loadConfig() {
    // TODO: Make this better, and more general.

    quickTools[ViewShiftPair("3D", Aztec::MShiftState(true, false, false, true, false, false))] = "KToolPanOrbit";
    quickTools[ViewShiftPair("3D", Aztec::MShiftState(false, true, false, true, false, false))] = "KToolPanTrack";
    quickTools[ViewShiftPair("3D", Aztec::MShiftState(false, false, true, true, false, false))] = "KToolPanZoom";

    quickTools[ViewShiftPair("UV", Aztec::MShiftState(false, true, false, true, false, false))] = "KToolPanTrack";
    quickTools[ViewShiftPair("UV", Aztec::MShiftState(false, false, true, true, false, false))] = "KToolPanZoom";

    setupDefaultColours();

    // now apply our default keys
    setupDefaultKeys();

    // load our shortcut keys.
    std::ifstream in(getConfigFile().c_str());

    if (in.is_open()) {
      in.setf(std::ios_base::skipws);
      while (!in.eof()) {
        char buf[1024];
        std::string str;

        in >> str;

        if (str == "quickTool") {
          std::string view, alt, ctrl, shift, left, mid, right, tool;

          in >> view >> alt >> ctrl >> shift >> left >> mid >> right >> tool;
          quickTools[ViewShiftPair(view, Aztec::MShiftState(toBool(left), toBool(mid), toBool(right), toBool(alt), toBool(ctrl), toBool(shift)))] = tool;
        } else if (str == "colour") {
          std::string col, r, g, b, a;

          in >> col >> r >> g >> b >> a;

          colours[Colours::getColourIndex(col)].set(toFloat(r), toFloat(g), toFloat(b), toFloat(a));
        } else if (str == "key") {
          std::string key, shift, command, type;
          Aztec::MShiftState state;
          bool pressed = true;

          in >> key;
          in >> type;

          if (type == "pressed") {
            pressed = true;
          } else if (type == "released") {
            pressed = false;
          } else {
            // otherwise we have a malformed line, bail out and goto the next line.
            in.getline(buf, 1024);
            continue;
          }

          while (shift != "cmd") {
            in >> shift;
            if (shift == "ctrl") {
              state.ctrlPressed = true;
            } else if (shift == "shift") {
              state.shiftPressed = true;
            } else if (shift == "alt") {
              state.altPressed = true;
            }
          }

          std::ws(in);
          in.getline(buf, 1024);
          command = buf;

          addShortcut(Aztec::MKeyEvent::strToKeyCode(key), state, pressed, command);
        } else if (str == "function") {
          // get the line, and split out the function name and friendly name.
          std::ws(in);
          in.getline(buf, 1024);

          // we do a buf+1 here because the first character is a tab char.
          std::string line = buf;
          int tabloc = line.find('\t');
          int tabloc2 = line.find('\t', tabloc + 1);
          if (tabloc != -1) {
            std::string func = line.substr(0, tabloc);
            std::string friendly = line.substr(tabloc + 1, tabloc2 - tabloc - 1);

            std::string category;
            if (tabloc2 != -1) {
              category = line.substr(tabloc2 + 1);
            } else {
              category = "Miscellanous";
            }

            addFriendlyCommand(friendly, func, category);
          }

        } else {

          // just get the rest of the line.
          in.getline(buf, 1024);
        }
      }
    }


    configLoaded = true;
  }

  std::ofstream& operator <<(std::ofstream &out, const ViewShiftPair &viewShift) {
    out << viewShift.first << SEP;
    out << (viewShift.second.altPressed ? "true" : "false") << SEP; 
    out << (viewShift.second.ctrlPressed ? "true" : "false") << SEP;
    out << (viewShift.second.shiftPressed ? "true" : "false") << SEP;
    out << (viewShift.second.leftMouseDown ? "true" : "false") << SEP;
    out << (viewShift.second.middleMouseDown ? "true" : "false") << SEP;
    out << (viewShift.second.rightMouseDown ? "true" : "false") << SEP;

    return out;
  }

  void UIConfig::saveConfig() {
    std::ofstream out(getConfigFile().c_str());

    // write out quick tools
    for (QuickToolMap::iterator it = quickTools.begin(); it != quickTools.end(); ++it) {
      if (it->second != "") {
        out << "quickTool";
        out << SEP;
        out << it->first;
        out << SEP;
        out << it->second;
        out << "\n";
      }
    }


    // write out colours
    for (ColourMap::iterator it = colours.begin(); it != colours.end(); ++it) {
      out << "colour";
      out << SEP;
      out << Colours::getName(it->first);
      out << SEP;
      out << it->second.x;
      out << SEP;
      out << it->second.y;
      out << SEP;
      out << it->second.z;
      out << SEP;
      out << it->second.w;
      out << "\n";
    }

    // write out the friendly names
    CommandList commands;
    getFriendlyCommands(commands);
    for (int i = 0; i < commands.size(); ++i) {
      if (commands[i].command.length() > 0 && commands[i].friendly.length() > 0) {
        out << "function" << SEP << commands[i].command << SEP << commands[i].friendly << SEP << commands[i].category << "\n";
      }
    }

    // write out the keyboard config
    std::vector<Shortcut> shortcuts;
    getShortcuts(shortcuts);

    for (int i = 0; i < shortcuts.size(); ++i) {
      std::string keyStr = Aztec::MKeyEvent::keyCodeToStr(shortcuts[i].code);

      if (keyStr.length() > 0 && shortcuts[i].command.length() > 0) {
        out << "key";
        out << SEP;
        out << keyStr;
        out << SEP;
        out << (shortcuts[i].pressed ? "pressed" : "released");
        out << SEP;
        if (shortcuts[i].state.shiftPressed) {
          out << "shift" << SEP;
        }
        if (shortcuts[i].state.ctrlPressed) {
          out << "ctrl" << SEP;
        }
        if (shortcuts[i].state.altPressed) {
          out << "alt" << SEP;
        }
        out << "cmd" << SEP;
        out << shortcuts[i].command << "\n";
      }
    }

  }


  std::string UIConfig::getQuickToolFor(const std::string &viewGroup, const Aztec::MShiftState &shiftState) {
    if (!configLoaded) {
      loadConfig();
    }

    return quickTools[std::make_pair(viewGroup, shiftState)];
  }

  bool UIConfig::isGrabManipulator(const Aztec::MShiftState &shiftState) {
    if (!configLoaded) {
      loadConfig();
    }

    // TODO: make this read the config from a config file.
    return (!shiftState.leftMouseDown && !shiftState.rightMouseDown && shiftState.middleMouseDown);
  }

  bool UIConfig::changesMade() {
    return g_ChangesMade;
  }

  void UIConfig::setChanged(bool changed) {
    g_ChangesMade = changed;
  }

  std::string UIConfig::getCurrentFilename() {
    return currentFilename;
  }

  void UIConfig::setCurrentFilename(const std::string &filename) {
    currentFilename = filename;
  }

  float UIConfig::get3DWidgetSize() {
    if (!configLoaded) {
      loadConfig();
    }
    return g_WidgetSize;
  }

  void UIConfig::set3DWidgetSize(float size) {
    if (!configLoaded) {
      loadConfig();
    }
    g_WidgetSize = size;
  }

  Aztec::MVector3 UIConfig::getColour(Colours::Colour colour) {
    if (!configLoaded) {
      loadConfig();
    }

    Aztec::MVector4 c = colours[colour];
    return Aztec::MVector3(c.x, c.y, c.z);
  }

  Aztec::MVector4 UIConfig::getColourWithAlpha(Colours::Colour colour) {
    if (!configLoaded) {
      loadConfig();
    }

    return colours[colour];
  }

  class CommandLess {
  public:

    bool operator()(const std::string &lhs, const std::string &rhs) const { 
      int lhsSize = lhs.size();
      if (lhsSize > 0) {
        if (lhs[lhs.size() - 1] == ';') {
          lhsSize--;
        }
      }
      int rhsSize = rhs.size();
      if (rhsSize > 0) {
        if (rhs[rhs.size() - 1] == ';') {
          rhsSize--;
        }
      }
      
      return lhs.compare(0, lhsSize, rhs, 0, rhsSize) < 0;
    }
  };



  typedef std::pair< Aztec::MKeyEvent::KeyCode, bool > KeyPair;
  typedef std::pair<KeyPair, Aztec::MShiftState> KeyShiftPair;

  typedef std::map<KeyShiftPair, std::string> ShortcutCommandMap;
  typedef std::map<std::string, KeyShiftPair, CommandLess> CommandShortcutMap;



  ShortcutCommandMap keyCmdMap;
  CommandShortcutMap cmdKeyMap;

  void UIConfig::addShortcut(Aztec::MKeyEvent::KeyCode code, Aztec::MShiftState state, bool pressed, const std::string &command) {
    Aztec::MShiftState realState;
    realState.altPressed = state.altPressed;
    realState.ctrlPressed = state.ctrlPressed;
    realState.shiftPressed = state.shiftPressed;
    keyCmdMap[ KeyShiftPair( KeyPair(code, pressed), realState ) ] = command;
    cmdKeyMap[ command ] = KeyShiftPair( KeyPair(code, pressed), realState );
  }

  void UIConfig::addShortcut(Aztec::MKeyEvent::KeyCode code, const std::string &command) {
    keyCmdMap[ KeyShiftPair( KeyPair(code, true), Aztec::MShiftState() ) ] = command;
    cmdKeyMap[ command ] = KeyShiftPair( KeyPair(code, true), Aztec::MShiftState() );
  }

  std::string UIConfig::getShortcut(Aztec::MKeyEvent::KeyCode code, Aztec::MShiftState state, bool pressed) {
    Aztec::MShiftState realState;
    realState.altPressed = state.altPressed;
    realState.ctrlPressed = state.ctrlPressed;
    realState.shiftPressed = state.shiftPressed;
    return keyCmdMap[ KeyShiftPair(KeyPair(code, pressed), state) ];
  }

  bool UIConfig::getShortcut(const std::string &command, Aztec::MKeyEvent::KeyCode &code, Aztec::MShiftState &state, bool &pressed) {
#ifdef _DEBUG
    std::vector<Shortcut> shortcuts;
    getShortcuts(shortcuts);

#endif
    
    
    CommandShortcutMap::iterator it = cmdKeyMap.find(command);
    if (it != cmdKeyMap.end()) {
      code = it->second.first.first;
      state = it->second.second;
      pressed = it->second.first.second;

      return true;
    } else {
      return false;
    }

  }


  typedef std::map<std::string, std::string> FriendCmdMap;
  typedef std::map<std::string, std::pair<std::string, std::string>, CommandLess> CommandFriendMap;

  static FriendCmdMap friendCmdMap;
  static CommandFriendMap cmdFriendMap;

  void UIConfig::addFriendlyCommand(const std::string &friendly, const std::string &command, const std::string &category) {
    // we do an insert because we do not want to overwrite existing values.
    FriendCmdMap::iterator friendIt = friendCmdMap.find(friendly);

    if (friendIt == friendCmdMap.end()) {
      friendCmdMap.insert(FriendCmdMap::value_type(friendly, command));
    } else if (friendIt->second == "") {
      friendIt->second = command;
    }

    CommandFriendMap::iterator commandIt = cmdFriendMap.find(command);

    if (commandIt == cmdFriendMap.end()) {
      cmdFriendMap.insert(CommandFriendMap::value_type(command, std::make_pair(friendly, category)));
    } else {
      if (commandIt->second.first == "") {
        commandIt->second.first = friendly;
        commandIt->second.second = category;
      }
    }
  }

  const std::string& UIConfig::getFriendlyCommand(const std::string &friendly) {
    return friendCmdMap[friendly];
  }

  const std::string& UIConfig::getFriendlyName(const std::string &command) {
    return cmdFriendMap[command].first;
  }

  void UIConfig::getFriendlyCommands(std::vector< UIConfig::Command > &commands) {
    commands.clear();
    for (CommandFriendMap::iterator it = cmdFriendMap.begin(); it != cmdFriendMap.end(); ++it) {
      Command c;
      c.command = it->first;
      c.friendly = it->second.first;
      c.category = it->second.second;
      commands.push_back(c);
    }
  }

  UIConfig::Shortcut::Shortcut(Aztec::MKeyEvent::KeyCode code,
                                      Aztec::MShiftState state,      
                                      bool pressed,                  
                                      const std::string &command) 
    : code(code), state(state), pressed(pressed), command(command)
  {


  }

  UIConfig::Shortcut::Shortcut(const Shortcut &src) 
    : code(src.code), state(src.state), pressed(src.pressed), command(src.command)
  {


  }

  void UIConfig::getShortcuts(std::vector<Shortcut> &shortcuts) {
    shortcuts.clear();
    for (ShortcutCommandMap::iterator it = keyCmdMap.begin(); it != keyCmdMap.end(); ++it) {
      Aztec::MKeyEvent::KeyCode code = it->first.first.first;
      Aztec::MShiftState state = it->first.second;
      bool pressed = it->first.first.second;
      std::string &command = it->second;
      shortcuts.push_back(Shortcut(code, state, pressed, command));
    }
  }


  JSClass aztecUIConfig_class = {
    "AztecUIConfig", JSCLASS_HAS_PRIVATE,
      JS_PropertyStub, JS_PropertyStub,
      JS_PropertyStub, JS_PropertyStub,
      JS_EnumerateStub, JS_ResolveStub,
      JS_ConvertStub,   JS_FinalizeStub
  };
  

  static JSBool js_getCurrentFilename(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
  {
    if (argc != 0) {
      return JS_FALSE;
    }
    
    std::string filename = UIConfig::getCurrentFilename();
    *rval = STRING_TO_JSVAL(JS_NewStringCopyZ(cx, (filename.c_str())));

    return JS_TRUE;
  }


  JSFunctionSpec aztecUIConfig_methods[] = {
    // Object/mesh creation
    {"getCurrentFilename",        js_getCurrentFilename,           0},
    {NULL,                  NULL,                   0} 
  };


  void UIConfig::initialise() {
    Aztec::MScriptInterpreter *interp = Aztec::MScriptInterpreter::getInstance();  

    JSObject *object = 
      JS_DefineObject(interp->GetScriptContext(), 
                      interp->GetGlobalObject(), 
                      "UIConfig", 
                      &aztecUIConfig_class, 
                      0, 
                      JSPROP_ENUMERATE);

    JS_DefineFunctions(interp->GetScriptContext(), object, aztecUIConfig_methods);

  }


}

