#include <AztecGUICommonPCH.h>

#include <gui/MGridLayout.h>

#include <assert.h>

namespace Aztec {

  MGridLayout::GridCell::GridCell() {
  }

  MGridLayout::GridCell::GridCell(const MGridLayout::GridCell &src) {
    component = src.component;
  }


  MConstraint MGridLayout::getConstaint(MGridLayout::Position row, MGridLayout::Position column) {
    return MConstraint((row << 16) | column);
  }

  static void decodeConstraint(const MConstraint &con, MGridLayout::Position &row, MGridLayout::Position &col) {
    col = con.m_Value & 0x0000FFFF;
    row = ((unsigned long)con.m_Value & 0xFFFF0000) >> 16;
  }

  MGridLayout::MGridLayout(int columns, int rows) { 
    cells.resize(rows);
    for (int i = 0; i < rows; ++i) {
      cells[i].resize(columns);
    }

    lastChangedRow = 0;
    lastChangedColumn = 0;
  }

  MGridLayout::~MGridLayout() {
  }

  int MGridLayout::getRowCount() const {
    return cells.size();
  }

  int MGridLayout::getColumnCount() const {
    if (cells.size() > 0) {
      return cells[0].size();
    }
    return 0;
  }


  // Layoutmanager methods
  void MGridLayout::addComponent(MComponentPtr component) {
    // we want to loop through twice, the first time from where we left off, 
    // and the second time looping through all the cells.
    for (int count = 0; count < 2; ++count) {

      // find the first available empty cell, starting from where we left off.
      for (int row = lastChangedRow; row < cells.size(); ++row) {
        for (int col = lastChangedColumn; col < cells[0].size(); ++col) {
          if (cells[row][col].component == NULL) {
            lastChangedRow = row;
            lastChangedColumn = col;
            cells[row][col].component = component;
            return;
          }
        }

        // if we have gotten here, we found no cells, so loop back to the start.
        lastChangedColumn = 0;
      }

      // if we have gotten here, we found no cells, so loop back to the start.
      lastChangedRow = 0;
    }
  }

  void MGridLayout::addComponent(MComponentPtr component, MConstraint constraints) {
    // decode the contraint.
    Position row, col;
    decodeConstraint(constraints, row, col);

    assert(row >= 0);
    assert(row < cells.size());
    assert(col >= 0);
    assert(col < cells[0].size());
    cells[row][col].component = component;
    lastChangedRow = row;
    lastChangedColumn = col;
  }

  void MGridLayout::removeComponent(MComponentPtr component) {
    // find the component
    for (int row = 0; row < cells.size(); ++row) {
      for (int col = 0; col < cells[0].size(); ++col) {
        if (cells[row][col].component == component) {
          cells[row][col].component = NULL;
        }
      }
    }

  }

  void MGridLayout::layoutContainer(MContainerPtr container) {
    MSize2D  contSize;
    float compWidth, compHeight;
    float compLeft, compTop;
    
    contSize = container->getClientSize();

    for (int x = 0; x < getColumnCount(); ++x) {
      for (int y = 0; y < getRowCount(); ++y) {
        MComponentPtr comp = cells[y][x].component;

        if (comp == NULL) {
          continue;
        }
        // if the preferred size is set, then we use that size, and centre
        // the component.
        if (comp->isPreferredSizeSet()) {
          MSize2D compSize = comp->getPreferredSize();
          compWidth = compSize.getWidth();
          compHeight = compSize.getHeight();
          compLeft = ((float)contSize.getWidth() / (float)getColumnCount()) * ((float)x + 0.5) - compWidth/2;
          compTop = ((float)contSize.getHeight() / (float)getRowCount()) * ((float)y + 0.5) - compHeight/2;

        } else {

          // otherwise fill out the entire square with the component.
          compWidth = (float)contSize.getWidth() / (float)getColumnCount();
          compHeight = (float)contSize.getHeight() / (float)getRowCount();
          compLeft = compWidth * x;
          compTop = compHeight * y;
        }

        comp->setPosition(compLeft, compTop, compWidth, compHeight );
      }
    }
  }

  /**
   * This gets the minimum size of this layout. 
   * <P>
   * the minimum size of a grid layout is the largest minimum size of 
   * all the components multiplied by the number of grid cells in each
   * direction.
   */
  MSize2D MGridLayout::getMinimumSize(MContainerPtr container) {
    MSize2D largestSize;

    for (int index = 0; index < container->getComponentCount(); index++) {
      MComponentPtr comp = container->getComponent(index);
      MSize2D size;

      size = comp->getMinimumSize();

      if (size.getWidth() > largestSize.getWidth()) {
        largestSize.setWidth(size.getWidth());
      }
      if (size.getHeight() > largestSize.getHeight()) {
        largestSize.setHeight(size.getHeight());
      }
    }

    largestSize.setWidth(largestSize.getWidth() * getColumnCount());
    largestSize.setHeight(largestSize.getHeight() * getRowCount());

    return largestSize;
  }

  MSize2D MGridLayout::getPreferredSize(MContainerPtr container) {
    MSize2D largestSize;

    for (int index = 0; index < container->getComponentCount(); index++) {
      MComponentPtr comp = container->getComponent(index);
      MSize2D size;

      size = comp->getPreferredSize();

      if (size.getWidth() > largestSize.getWidth()) {
        largestSize.setWidth(size.getWidth());
      }
      if (size.getHeight() > largestSize.getHeight()) {
        largestSize.setHeight(size.getHeight());
      }
    }

    largestSize.setWidth( largestSize.getWidth() * getColumnCount() );
    largestSize.setHeight( largestSize.getHeight() * getRowCount() );

    return largestSize;
  }

  MConstraint MGridLayout::getConstraint(const MComponentPtr &component) {
    for (int x = 0; x < getColumnCount(); ++x) {
      for (int y = 0; y < getRowCount(); ++y) {
        const MComponentPtr &comp = cells[y][x].component;

        if (comp == component) {
          return getConstaint(y, x);
        }
      }
    }

    return MConstraint();
  }

}

