#include <AztecGUICommonPCH.h>

#include <gui/MComboBox.h>
#include <gui/win32/MContainerImpl.h>
#include <gui/win32/MAppImpl.h>

#include <algorithm>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  MComboBox::MComboBox()
    : MComponent()
  {
    m_SelIndex = -1;
  }

  MComboBox::~MComboBox() {
  }

  void MComboBox::setInternalValue(const std::string &value) {
    m_Value = value;
  }

  bool MComboBox::createImpl() {
    MApp *app;
    HWND parentHWND = NULL;

    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    m_hWnd = ::CreateWindow("COMBOBOX", m_Value.c_str(), WS_CHILD | CBS_AUTOHSCROLL | CBS_DROPDOWNLIST | CBS_HASSTRINGS,
                            0, 10, 0, 100, parentHWND, NULL, app->getHInstance(), NULL);

    if (m_hWnd != 0) {
      app->registerComponent(m_hWnd, this);

      ::SendMessage(m_hWnd, WM_SETFONT, (DWORD)::GetStockObject(DEFAULT_GUI_FONT), 0);

      // now place any items we have in our list into the combo box.
      for (int i = 0; i < m_Values.size(); ++i) {
        ::SendMessage(m_hWnd, CB_ADDSTRING, 0, (DWORD)m_Values[i].c_str());
      }

      // now set the currently selected item if any.
      ::SendMessage(m_hWnd, CB_SETCURSEL, m_SelIndex, 0);

      // and set the current text in the combo box if we have any.
      ::SetWindowText(m_hWnd, m_Value.c_str());

      onCreate();

      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }

  bool MComboBox::handleWMCommandNotify(int notifyCode, int id) {
    if (MComponent::handleWMCommandNotify(notifyCode, id)) {
      return true;
    }

    if (notifyCode == CBN_SELCHANGE) {
      // if our selection has changed, update our current value
      char buf[1024];
      ::GetWindowText(m_hWnd, buf, 1024);
      m_Value = buf;
      m_SelIndex = ::SendMessage(m_hWnd, CB_GETCURSEL, 0, 0);

      // call the event handler
      onChanged();

      notifyListeners();
    }

    return false;
  }

  void MComboBox::setValue(const std::string &value) {
    if (value == m_Value) {
      return;
    }

    m_Value = value;
    if (m_hWnd != NULL) {
      ::SetWindowText(m_hWnd, value.c_str());
    }

    // see if we have an exact match
    std::vector<std::string>::iterator it = std::find(m_Values.begin(), m_Values.end(), value);

    if (it != m_Values.end()) {
      setSelectedIndex(it - m_Values.begin());
    }
  }

  std::string MComboBox::getValue() {
    return m_Value;
  }

  void MComboBox::addItem(const std::string &item) {
    DWORD result;

    if (m_hWnd != NULL) {
      result = ::SendMessage(m_hWnd, CB_ADDSTRING, 0, (DWORD)item.c_str());

      if (result == CB_ERR || result == CB_ERRSPACE) {
        return;
      }
    }

    m_Values.push_back(item);
  }

  void MComboBox::insertItem(const std::string &item, int index) {
    DWORD result;

    result = ::SendMessage(m_hWnd, CB_ADDSTRING, index, (DWORD)item.c_str());

    if (result == CB_ERR || result == CB_ERRSPACE) {
      return;
    }

    m_Values.insert(m_Values.begin() + index, item);

  }

  void MComboBox::removeItem(const std::string &item) {
    removeItem(findItem(item));
  }

  void MComboBox::removeItem(int index) {
    if (index < 0) {
      return;
    }

    DWORD result;

    result = ::SendMessage(m_hWnd, CB_DELETESTRING, index, 0);

    if (result == CB_ERR) {
      return;
    }

    m_Values.erase(m_Values.begin() + index);
  }

  void MComboBox::removeAllItems() {
    ::SendMessage(m_hWnd, CB_RESETCONTENT, 0, 0);

    m_Values.clear();
  }

  int MComboBox::findItem(const std::string &item) {
    std::vector<std::string>::iterator it;
    int index = 0;

    for (it = m_Values.begin(); it != m_Values.end(); ++it) {
      if (*it == item) {
        return index;
      }
    }

    return -1;
  }


  int MComboBox::getItemCount() {
    return m_Values.size();
  }


  int MComboBox::getSelectedIndex() {
    return m_SelIndex;
  }

  std::string MComboBox::getSelectedItem() {
    if (m_SelIndex == -1) {
      return "";
    }
    return m_Values[m_SelIndex];
  }

  void MComboBox::setSelectedIndex(int index) {
    m_SelIndex = index;
    if (m_hWnd != NULL) {
      ::SendMessage(m_hWnd, CB_SETCURSEL, index, 0);
    } else {
      // call the event handlers directly.
      if (index >= 0) {
        m_Value = m_Values[index];
      } else {
        m_Value = "";
      }

      onChanged();
    }
  }

  void MComboBox::setSelectedItem(const std::string &item) {
    DWORD result;

    result = ::SendMessage(m_hWnd, CB_SELECTSTRING, 0, (DWORD)item.c_str());

    if (result == CB_ERR) {
      return;
    }

    m_SelIndex = result;
  }

  void MComboBox::showDropDown() {
    ::SendMessage(m_hWnd, CB_SHOWDROPDOWN, TRUE, 0);
  }

  void MComboBox::hideDropDown() {
    ::SendMessage(m_hWnd, CB_SHOWDROPDOWN, FALSE, 0);
  }

  bool MComboBox::isDropDownVisible() {
    return ::SendMessage(m_hWnd, CB_GETDROPPEDSTATE, 0, 0) == TRUE;
  }

  bool MComboBox::onChanged() {
    return false;
  }

  void MComboBox::addListener(const ComboBoxListenerPtr &listener) {
    listeners.push_back(listener);
  }

  void MComboBox::setPosition(int X, int Y, int width, int height) {
    ::SetWindowPos(m_hWnd, NULL, X, Y, width, height + 100, SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComboBox::setSize(const MSize2D &size) {
    ::SetWindowPos(m_hWnd, NULL, size.getLeft(), size.getTop(), size.getWidth(), size.getHeight() + 100, SWP_NOMOVE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComboBox::setSize(int width, int height) {
    ::SetWindowPos(m_hWnd, NULL, 0, 0, width, height+100, SWP_NOMOVE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  bool MComboBox::isMinimumSizeSet() {
    return true;
  }

  MSize2D MComboBox::getMinimumSize() {
    MSize2D minSize(160,30);

    return MSize2D::getLargest(minSize, MComponent::getMinimumSize());
  }

  void MComboBox::notifyListeners() {
    for (int i = 0; i < listeners.size(); ++i) {
      listeners[i]->onListen(this);
    }
  }



}

