#include <AztecGUICommonPCH.h>
#include <gui/MComponent.h>

#include <gui/win32/MAppImpl.h>
#include <gui/win32/MContainerImpl.h>

namespace Aztec {

  MComponent::MComponent() {
    m_hWnd = NULL;
    m_Parent = NULL;

    m_MinimumSize.setWidth(-1);
    m_MinimumSize.setHeight(-1);
    m_PreferredSize.setWidth(-1);
    m_PreferredSize.setWidth(-1);

    paintDC = NULL;

    bgColourSet = false;
    fgColourSet = false;
  }

  MComponent::MComponent(const std::string &name) 
    : m_Name(name) 
  {
    m_hWnd = NULL;
    m_Parent = NULL;

    m_MinimumSize.setWidth(-1);
    m_MinimumSize.setHeight(-1);
    m_PreferredSize.setWidth(-1);
    m_PreferredSize.setHeight(-1);
  }

  MComponent::~MComponent() {
    m_Parent = NULL;
  }

  bool MComponent::createImpl() {
    if (m_hWnd != NULL) {
      // if we have already had ourselves created, don't create us again.
      return true;
    }

    MApp *app;
    HWND parentHWND = NULL;

    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    m_hWnd = ::CreateWindow((LPCTSTR)app->getWindowClass("component"), NULL, WS_CHILD,
                            0, 10, 0, 10, parentHWND, NULL, app->getHInstance(), NULL);

    if (m_hWnd != 0) {
      app->registerComponent(m_hWnd, this);

      onCreate();

      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }

  void MComponent::destroyImpl() {
    if (m_hWnd != NULL) {
      ::DestroyWindow(m_hWnd);
      m_hWnd = NULL;
      m_Parent = NULL;
    }
  }


  static MShiftState getStateFromFlags(DWORD flags) {
    MShiftState shift;
    shift.ctrlPressed = (flags & MK_CONTROL) > 0;
    shift.shiftPressed = (flags & MK_SHIFT) > 0;
    shift.leftMouseDown = (flags & MK_LBUTTON) > 0;
    shift.rightMouseDown = (flags & MK_RBUTTON) > 0;
    shift.middleMouseDown = (flags & MK_MBUTTON) > 0;
  
    short r,l,m;
    r = GetAsyncKeyState(VK_RMENU);
    l = GetAsyncKeyState(VK_LMENU);
    m = GetAsyncKeyState(VK_MENU);
  
    if (r <0 || l < 0 || m < 0) {
      shift.altPressed = true;
    } else {
      shift.altPressed = false;
    }

    return shift;
  }

  bool MComponent::isVKKeyDown(int virtKeyCode) {
    return GetAsyncKeyState(virtKeyCode) < 0;
  }

  MShiftState MComponent::getStateFromAsync() {
    MShiftState shift;
    shift.ctrlPressed = isVKKeyDown(VK_CONTROL) || isVKKeyDown(VK_LCONTROL) || isVKKeyDown(VK_RCONTROL);
    shift.shiftPressed = isVKKeyDown(VK_SHIFT) || isVKKeyDown(VK_LSHIFT) || isVKKeyDown(VK_RSHIFT);
    shift.leftMouseDown = isVKKeyDown(VK_LBUTTON);
    shift.rightMouseDown = isVKKeyDown(VK_RBUTTON);
    shift.middleMouseDown = isVKKeyDown(VK_MBUTTON);
  
    shift.altPressed = isVKKeyDown(VK_RMENU) ||isVKKeyDown(VK_LMENU) || isVKKeyDown(VK_MENU);

    return shift;
  }

#ifndef VK_OEM_PLUS
#define VK_OEM_PLUS 0xBB
#endif

#ifndef VK_OEM_COMMA
#define VK_OEM_COMMA 0xBC
#endif

#ifndef VK_OEM_MINUS
#define VK_OEM_MINUS 0xBD
#endif

#ifndef VK_OEM_PERIOD
#define VK_OEM_PERIOD 0xBE
#endif

#ifndef VK_OEM_2
#define VK_OEM_2 0xBF
#endif

  typedef std::map<int, MKeyEvent::KeyCode> VKKeyCodeMap;
  typedef std::map<MKeyEvent::KeyCode, int> KeyCodeVKMap;

  typedef std::map<TCHAR, MKeyEvent::KeyCode> CharKeyCodeMap;
  typedef std::map<MKeyEvent::KeyCode, TCHAR> KeyCodeCharMap;

  static VKKeyCodeMap vk_key;
  static KeyCodeVKMap key_vk;

  static CharKeyCodeMap char_key;
  static KeyCodeCharMap key_char;

  static void addKey(int vk, MKeyEvent::KeyCode code) {
    vk_key[vk] = code;
    key_vk[code] = vk;
  }

  static void addKey(int vk, MKeyEvent::KeyCode code, TCHAR ch) {
    vk_key[vk] = code;
    key_vk[code] = vk;

    char_key[ch] = code;
    key_char[code] = ch;
  }

  static void createKeyMaps() {
    if (vk_key.size() != 0) {
      return;
    }
    // we add them in a second time here to ensure they take pecedence in the map, regardless of how addKey works.
    addKey(0, MKeyEvent::Key_None);
    addKey(0, MKeyEvent::Key_Unknown);
    addKey(0, MKeyEvent::Key_None);

    addKey(VK_LBUTTON, MKeyEvent::Key_LeftButton);
    addKey(VK_RBUTTON, MKeyEvent::Key_RightButton);
    addKey(VK_MBUTTON, MKeyEvent::Key_MiddleButton);
      
    addKey(VK_BACK, MKeyEvent::Key_Backspace);
    addKey(VK_TAB, MKeyEvent::Key_Tab);
      
    addKey(VK_CLEAR, MKeyEvent::Key_Clear);
    addKey(VK_RETURN, MKeyEvent::Key_Return);
      
    addKey(VK_SHIFT, MKeyEvent::Key_Shift);
    addKey(VK_CONTROL, MKeyEvent::Key_Control);
    addKey(VK_MENU, MKeyEvent::Key_Alt);
    addKey(VK_PAUSE, MKeyEvent::Key_Pause);
    addKey(VK_CAPITAL, MKeyEvent::Key_CapsLock);
      
    addKey(VK_ESCAPE, MKeyEvent::Key_Escape);
      
    addKey(VK_SPACE, MKeyEvent::Key_Space, ' ');
    addKey(VK_PRIOR, MKeyEvent::Key_Prior);
    addKey(VK_NEXT, MKeyEvent::Key_Next);
    addKey(VK_END, MKeyEvent::Key_End);
    addKey(VK_HOME, MKeyEvent::Key_Home);
    addKey(VK_LEFT, MKeyEvent::Key_Left);
    addKey(VK_UP, MKeyEvent::Key_Up);
    addKey(VK_RIGHT, MKeyEvent::Key_Right);
    addKey(VK_DOWN, MKeyEvent::Key_Down);
    addKey(VK_SELECT, MKeyEvent::Key_Unknown);
    addKey(VK_PRINT, MKeyEvent::Key_Print);
    addKey(VK_EXECUTE, MKeyEvent::Key_SysReq);
    addKey(VK_SNAPSHOT, MKeyEvent::Key_Unknown);
    addKey(VK_INSERT, MKeyEvent::Key_Insert);
    addKey(VK_DELETE, MKeyEvent::Key_Delete);
    addKey(VK_HELP, MKeyEvent::Key_Help);
      
    addKey(VK_LWIN, MKeyEvent::Key_Unknown);
    addKey(VK_RWIN, MKeyEvent::Key_Unknown);
    addKey(VK_APPS, MKeyEvent::Key_Unknown);
      
    addKey(VK_NUMPAD0, MKeyEvent::Key_0);
    addKey(VK_NUMPAD1, MKeyEvent::Key_1);
    addKey(VK_NUMPAD2, MKeyEvent::Key_2);
    addKey(VK_NUMPAD3, MKeyEvent::Key_3);
    addKey(VK_NUMPAD4, MKeyEvent::Key_4);
    addKey(VK_NUMPAD5, MKeyEvent::Key_5);
    addKey(VK_NUMPAD6, MKeyEvent::Key_6);
    addKey(VK_NUMPAD7, MKeyEvent::Key_7);
    addKey(VK_NUMPAD8, MKeyEvent::Key_8);
    addKey(VK_NUMPAD9, MKeyEvent::Key_9);
    addKey(VK_MULTIPLY, MKeyEvent::Key_Asterisk, '*');
    addKey(VK_ADD, MKeyEvent::Key_Plus, '+');
    addKey(VK_SEPARATOR, MKeyEvent::Key_Comma, ',');
    addKey(VK_SUBTRACT, MKeyEvent::Key_Minus, '-');
    addKey(VK_DECIMAL, MKeyEvent::Key_Period, '.');
    addKey(VK_DIVIDE, MKeyEvent::Key_Slash, '/');
    addKey(VK_F1, MKeyEvent::Key_F1);
    addKey(VK_F2, MKeyEvent::Key_F2);
    addKey(VK_F3, MKeyEvent::Key_F3);
    addKey(VK_F4, MKeyEvent::Key_F4);
    addKey(VK_F5, MKeyEvent::Key_F5);
    addKey(VK_F6, MKeyEvent::Key_F6);
    addKey(VK_F7, MKeyEvent::Key_F7);
    addKey(VK_F8, MKeyEvent::Key_F8);
    addKey(VK_F9, MKeyEvent::Key_F9);
    addKey(VK_F10, MKeyEvent::Key_F10);
    addKey(VK_F11, MKeyEvent::Key_F11);
    addKey(VK_F12, MKeyEvent::Key_F12);
    addKey(VK_F13, MKeyEvent::Key_F13);
    addKey(VK_F14, MKeyEvent::Key_F14);
    addKey(VK_F15, MKeyEvent::Key_F15);
    addKey(VK_F16, MKeyEvent::Key_F16);
    addKey(VK_F17, MKeyEvent::Key_F17);
    addKey(VK_F18, MKeyEvent::Key_F18);
    addKey(VK_F19, MKeyEvent::Key_F19);
    addKey(VK_F20, MKeyEvent::Key_F20);
    addKey(VK_F21, MKeyEvent::Key_F21);
    addKey(VK_F22, MKeyEvent::Key_F22);
    addKey(VK_F23, MKeyEvent::Key_F23);
    addKey(VK_F24, MKeyEvent::Key_F24);

    addKey(VkKeyScan('['), MKeyEvent::Key_BracketLeft, '[');
    addKey(VkKeyScan(']'), MKeyEvent::Key_BracketRight, ']');

    addKey(VkKeyScan('('), MKeyEvent::Key_ParenLeft, '(');
    addKey(VkKeyScan(')'), MKeyEvent::Key_ParenRight, ')');
      
    addKey(VK_NUMLOCK, MKeyEvent::Key_NumLock);
    addKey(VK_SCROLL, MKeyEvent::Key_ScrollLock);

    addKey('0', MKeyEvent::Key_0, '0');
    addKey('1', MKeyEvent::Key_1, '1');
    addKey('2', MKeyEvent::Key_2, '2');
    addKey('3', MKeyEvent::Key_3, '3');
    addKey('4', MKeyEvent::Key_4, '4');
    addKey('5', MKeyEvent::Key_5, '5');
    addKey('6', MKeyEvent::Key_6, '6');
    addKey('7', MKeyEvent::Key_7, '7');
    addKey('8', MKeyEvent::Key_8, '8');
    addKey('9', MKeyEvent::Key_9, '9');
      
    addKey('A', MKeyEvent::Key_A, 'A');
    addKey('B', MKeyEvent::Key_B, 'B');
    addKey('C', MKeyEvent::Key_C, 'C');
    addKey('D', MKeyEvent::Key_D, 'D');
    addKey('E', MKeyEvent::Key_E, 'E');
    addKey('F', MKeyEvent::Key_F, 'F');
    addKey('G', MKeyEvent::Key_G, 'G');
    addKey('H', MKeyEvent::Key_H, 'H');
    addKey('I', MKeyEvent::Key_I, 'I');
    addKey('J', MKeyEvent::Key_J, 'J');
    addKey('K', MKeyEvent::Key_K, 'K');
    addKey('L', MKeyEvent::Key_L, 'L');
    addKey('M', MKeyEvent::Key_M, 'M');
    addKey('N', MKeyEvent::Key_N, 'N');
    addKey('O', MKeyEvent::Key_O, 'O');
    addKey('P', MKeyEvent::Key_P, 'P');
    addKey('Q', MKeyEvent::Key_Q, 'Q');
    addKey('R', MKeyEvent::Key_R, 'R');
    addKey('S', MKeyEvent::Key_S, 'S');
    addKey('T', MKeyEvent::Key_T, 'T');
    addKey('U', MKeyEvent::Key_U, 'U');
    addKey('V', MKeyEvent::Key_V, 'V');
    addKey('W', MKeyEvent::Key_W, 'W');
    addKey('X', MKeyEvent::Key_X, 'X');
    addKey('Y', MKeyEvent::Key_Y, 'Y');
    addKey('Z', MKeyEvent::Key_Z, 'Z');

    addKey(VkKeyScan('='), MKeyEvent::Key_Equal, '=');
    addKey(VkKeyScan('-'), MKeyEvent::Key_Minus, '-');
    addKey(VkKeyScan(','), MKeyEvent::Key_Comma, ',');
    addKey(VkKeyScan('.'), MKeyEvent::Key_Period, '.');
    addKey(VkKeyScan('/'), MKeyEvent::Key_Slash, '/');
    addKey(VkKeyScan(';'), MKeyEvent::Key_Semicolon, ';');
    addKey(VkKeyScan('\''), MKeyEvent::Key_Apostrophe, '\'');

    // we add them in a second time here to ensure they take pecedence in the map, regardless of how addKey works.
    addKey(0, MKeyEvent::Key_None);
    addKey(0, MKeyEvent::Key_Unknown);
    addKey(0, MKeyEvent::Key_None);
  }

  MKeyEvent::KeyCode MComponent::getKeyCodeFromVK(int vk) {
    createKeyMaps();

    VKKeyCodeMap::iterator it = vk_key.find(vk);

    if (it != vk_key.end()) {
      return it->second;
    } else {
      return MKeyEvent::Key_Unknown;
    }
  }

  int MComponent::getVKFromKeyCode(MKeyEvent::KeyCode code) {
    createKeyMaps();

    KeyCodeVKMap::iterator it = key_vk.find(code);

    if (it !=key_vk.end()) {
      return it->second;
    } else {
      return 0;
    }
  }

  MKeyEvent::KeyCode MComponent::getKeyCodeFromChar(TCHAR ch) {
    createKeyMaps();

    CharKeyCodeMap::iterator it = char_key.find(ch);
    if (it != char_key.end()) {
      return it->second;
    } else {
      it = char_key.find(toupper(ch));
      if (it != char_key.end()) {
        return it->second;
      }
      return MKeyEvent::Key_Unknown;
    }
  }

  bool MComponent::wndProc(UINT message, WPARAM wParam, LPARAM lParam, LRESULT *result) {
    bool handled = false;

//    if (message == WM_MOUSEACTIVATE) {
//      *result = MA_NOACTIVATEANDEAT;
//      return true;
//    }
    // if we have a component, then we can pass it along.
    if (message == WM_PAINT) {
      PAINTSTRUCT ps;
      HDC hdc = BeginPaint(this->m_hWnd, &ps);

      this->paintDC = hdc;

      handled = this->onPaint();

      this->paintDC = NULL;

      EndPaint(this->m_hWnd, &ps);
      *result = 0;

    } else if (message == WM_ERASEBKGND) {
      handled = true;
      *result = TRUE;

    } else if ( message == WM_MOUSEMOVE ||
                message == WM_LBUTTONDOWN ||
                message == WM_LBUTTONUP ||
                message == WM_LBUTTONDBLCLK ||
                message == WM_RBUTTONDOWN ||
                message == WM_RBUTTONUP ||
                message == WM_RBUTTONDBLCLK ||
                message == WM_MBUTTONDOWN ||
                message == WM_MBUTTONUP ||
                message == WM_MBUTTONDBLCLK ||
                message == WM_MOUSEWHEEL) {

      DWORD fwKeys = LOWORD(wParam);        // key flags 
      int xPos = (short)LOWORD(lParam);  // horizontal position of cursor 
      int yPos = (short)HIWORD(lParam);  // vertical position of cursor 

      MShiftState state = getStateFromFlags(fwKeys);

      const DWORD tick = ::GetTickCount();
      const MPoint2D mousePos(xPos, yPos);

      if (message == WM_MOUSEMOVE) {
        MMouseEvent event(MMouseEvent::MOVE, tick, mousePos, state, this);

        handled = MApp::getInstance()->onMouseMove(event);

        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMouseMove(event);
        }
      } else if (message == WM_LBUTTONDOWN) {
        MMouseEvent event(MMouseEvent::LBUTTON_DOWN, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMousePressed(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMousePressed(event);
        }
      } else if (message == WM_LBUTTONUP) {
        MMouseEvent event(MMouseEvent::LBUTTON_UP, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMouseReleased(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMouseReleased(event);
        }
      } else if (message == WM_LBUTTONDBLCLK) {
        MMouseEvent event(MMouseEvent::LBUTTON_DBLCLK, tick, mousePos, state, this);
        handled = this->onMouseDoubleClick(event);
        fireMouseListeners(event);

      } else if (message == WM_RBUTTONDOWN) {
        MMouseEvent event(MMouseEvent::RBUTTON_DOWN, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMousePressed(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMousePressed(event);
        }
      } else if (message == WM_RBUTTONUP) {
        MMouseEvent event(MMouseEvent::RBUTTON_UP, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMouseReleased(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMouseReleased(event);
        }
      } else if (message == WM_RBUTTONDBLCLK) {
        MMouseEvent event(MMouseEvent::RBUTTON_DBLCLK, tick, mousePos, state, this);
        handled = this->onMouseDoubleClick(event);
        fireMouseListeners(event);

      } else if (message == WM_MBUTTONDOWN) {
        MMouseEvent event(MMouseEvent::MBUTTON_DOWN, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMousePressed(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMousePressed(event);
        }
      } else if (message == WM_MBUTTONUP) {
        MMouseEvent event(MMouseEvent::MBUTTON_UP, tick, mousePos, state, this);
        handled = MApp::getInstance()->onMouseReleased(event);
        fireMouseListeners(event);

        if (!handled) {
          handled = this->onMouseReleased(event);
        }
      } else if (message == WM_MBUTTONDBLCLK) {
        MMouseEvent event(MMouseEvent::MBUTTON_DBLCLK, tick, mousePos, state, this);
        handled = this->onMouseDoubleClick(event);
        fireMouseListeners(event);
      } else if (message == WM_MOUSEWHEEL) {
        int delta = (short) HIWORD(wParam);    // wheel rotation

        if (delta > 0) {
          MMouseEvent event(MMouseEvent::WHEEL_UP, tick, mousePos, state, this);
          handled = this->onMouseWheel(event);
          fireMouseListeners(event);
        } else {
          MMouseEvent event(MMouseEvent::WHEEL_DOWN, tick, mousePos, state, this);
          handled = this->onMouseWheel(event);
          fireMouseListeners(event);
        }
      }

    } else if (message == WM_WINDOWPOSCHANGED) {
      WINDOWPOS *pos = (WINDOWPOS*)lParam;

      if ( !IsIconic(m_hWnd) && !(pos->flags & SWP_NOSIZE) ) {

        handled |= this->onResize(pos->cx, pos->cy);

        // Renable the painting between a pos changing and pos changed event, but only if it is a top level window.
        /**
         @todo This code made resizing of windows very smooth, but with the 
         horrible side effect of causing the windows desktop to be replainted. 
         This needs to have another look at it in the future to see if we can 
         make it smooth with the icky side effects.
        if (AZTEC_CAST(MWindow, this) != NULL) {
          LockWindowUpdate(NULL);
        }
        */

      }

      if ( pos->flags & SWP_SHOWWINDOW ) {
        handled |= this->onShow();
      }
    } else if (message == WM_CLOSE) {
      // see if we are closing an MWindow.
      MWindowPtr window = AZTEC_CAST(MWindow, this);
      if (window != NULL) {
        handled = window->onClose();
      }

    } else if (message == WM_GETMINMAXINFO) {
      MINMAXINFO *info = (MINMAXINFO*)lParam;

      MSize2D size = this->getMinimumSize();

      RECT windowRect, clientRect;

      ::GetWindowRect(this->m_hWnd, &windowRect);
      ::GetClientRect(this->m_hWnd, &clientRect);

      info->ptMinTrackSize.x = size.getWidth() > 64 ? size.getWidth() : 64;
      info->ptMinTrackSize.y = size.getHeight() > 64 ? size.getHeight() : 64;
    } else if (message == WM_KEYDOWN) {

      handled = handleKeyEvent(MKeyEvent(MKeyEvent::KEY_DOWN, 
                                         ::GetTickCount(), 
                                         getKeyCodeFromVK(wParam), 
                                         getStateFromAsync(), 
                                         this));

    } else if (message == WM_KEYUP) {

      handled = handleKeyEvent(MKeyEvent(MKeyEvent::KEY_UP, 
                                         ::GetTickCount(), 
                                         getKeyCodeFromVK(wParam), 
                                         getStateFromAsync(), 
                                         this));
    }
    return handled;
  }

  bool MComponent::handleWMCommandNotify(int notifyCode, int id) {
    return false;
  }

  bool MComponent::handleWMNotify(WPARAM wParam, LPARAM lParam) {
    return false;
  }

  bool MComponent::preTranslateMessage(MSG *msg) {
    return false;
  }

  HBRUSH MComponent::getBackgroundBrush() {
    if (!bgColourSet) {
      return NULL;
    }

    if (bgBrush == NULL) {
      // mix the brush colour with the standard colour.
      DWORD orig = ::GetSysColor(COLOR_WINDOW);

      Aztec::MVector3 a, b;

      a.set((float)GetRValue(orig) / 255.0f, (float)GetGValue(orig) / 255.0f, (float)GetBValue(orig) / 255.0f);
      b.set(bgColour.r, bgColour.g, bgColour.b);

      b = (1.0f - bgColour.a) * a + (bgColour.a) * b;

      Aztec::MColour c(b.x, b.y, b.z);


      bgBrush = ::CreateSolidBrush(RGB(c.getRuchar(), c.getGuchar(), c.getBuchar()));
    }

    return bgBrush;
  }

  // MComponent methods.
 
  void MComponent::setName(const std::string name) {
    m_Name = name;
  }

  std::string MComponent::getName() {
    return m_Name;
  }

  void MComponent::setParent(MContainerPtr parent) {
    if (m_Parent == &*parent) {
      return;
    }

    m_Parent = &*parent;

    if (parent != NULL) {
      parent->addComponent(this);
    }

    // create the physical windows implementation if needed.
    createImpl();

    HWND parentHWnd = NULL;

    if (parent != NULL) {
      parentHWnd = parent->m_hWnd;
    }

    ::SetParent(m_hWnd, parentHWnd);

    // now we adjust the window styles to reflect reality.
    if (parentHWnd != NULL) {
      // make sure that the child window does have WS_CHILD
      // and does not have the WS_POPUP style.
      long style;
      style = ::GetWindowLong(m_hWnd, GWL_STYLE);
      ::SetWindowLong(m_hWnd, GWL_STYLE, style & (~WS_POPUP));

      style = ::GetWindowLong(m_hWnd, GWL_STYLE);
      ::SetWindowLong(m_hWnd, GWL_STYLE, style | WS_CHILD);

    } else {
      // make sure that the child window does NOT have WS_CHILD
      // and DOES have the WS_POPUP style.
      long style;
      style = ::GetWindowLong(m_hWnd, GWL_STYLE);
      ::SetWindowLong(m_hWnd, GWL_STYLE, style & (~WS_CHILD));

      style = ::GetWindowLong(m_hWnd, GWL_STYLE);
      ::SetWindowLong(m_hWnd, GWL_STYLE, style | WS_POPUP);
    }
  }

  MContainerPtr MComponent::getParent() {
    return m_Parent;
  }

  void MComponent::setVisible(bool visible) {
    ::ShowWindow(m_hWnd, visible ? SW_SHOW : SW_HIDE);
  }

  void MComponent::setPosition(const MPoint2D &pos) {
    ::SetWindowPos(m_hWnd, NULL, pos.x, pos.y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComponent::setPosition(int X, int Y) {
    ::SetWindowPos(m_hWnd, NULL, X, Y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComponent::setPosition(int X, int Y, int width, int height) {
    ::SetWindowPos(m_hWnd, NULL, X, Y, width, height, SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  MPoint2D MComponent::getPosition() {
    MRect2D rect = getWindowRect();
    return MPoint2D(rect.x1, rect.y1);
    
  }

  void MComponent::setSize(const MSize2D &size) {
    ::SetWindowPos(m_hWnd, NULL, size.getLeft(), size.getTop(), size.getWidth(), size.getHeight(), SWP_NOMOVE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComponent::setSize(int width, int height) {
    ::SetWindowPos(m_hWnd, NULL, 0, 0, width, height, SWP_NOMOVE | SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  MSize2D MComponent::getSize() {
    return getWindowRect();
  }

  MSize2D MComponent::getClientSize() {
    RECT rect;
    ::GetClientRect(m_hWnd, &rect);

    return MSize2D(rect.left, rect.top, rect.right, rect.bottom);
  }

  MSize2D MComponent::getMinimumSize() {
    if (isMinimumSizeSet()) {
      return m_MinimumSize;
    }

    // otherwise return a tiny rectangle.
    return MSize2D(10, 10);
  }

  void MComponent::setMinimumSize(const MSize2D &size) {
    m_MinimumSize = size;
  }

  bool MComponent::isMinimumSizeSet() {
    if (m_MinimumSize.getWidth() < 0 || m_MinimumSize.getHeight() < 0 ) {
      return false;
    }

    return true;
  }

  MSize2D MComponent::getPreferredSize() {
    if (isPreferredSizeSet()) {
      return m_PreferredSize;
    }

    // otherwise return out minimum size.
    return getMinimumSize();
  }

  void MComponent::setPreferredSize(const MSize2D &size) {
    m_PreferredSize = size;
  }

  bool MComponent::isPreferredSizeSet() {
    if (m_PreferredSize.getWidth() < 0 || m_PreferredSize.getHeight() < 0 ) {
      return false;
    }

    return true;
  }

  MRect2D MComponent::getWindowRect() {
    MRect2D result;
    RECT rect;
    HWND parentHWnd;

    // this gets the rect of the window 
    // in screen coordinates.
    ::GetWindowRect(m_hWnd, &rect);

    parentHWnd = ::GetParent(m_hWnd);

    // if we have no parent, then we are a top level window
    if (parentHWnd == NULL) {
      return MRect2D( rect.left, rect.top, rect.right, rect.bottom );
    }

    // otherewise we need to convert to the parents local coords.
    POINT topleft, botright;

    topleft.x = rect.left;
    topleft.y = rect.top;
    botright.x = rect.right;
    botright.y = rect.bottom;
    ::ScreenToClient(parentHWnd, &topleft);
    ::ScreenToClient(parentHWnd, &botright);

    return MRect2D(topleft.x, topleft.y, botright.x, botright.y);
  }

  void MComponent::setWindowRect(const MRect2D &rect) {
    ::SetWindowPos(m_hWnd, NULL, rect.x1, rect.y1, rect.x2 - rect.x1, rect.y2 - rect.y1, SWP_NOZORDER | SWP_NOOWNERZORDER);
  }

  void MComponent::refresh() {
    if (m_hWnd != NULL) {
      ::InvalidateRect(m_hWnd, NULL, FALSE);
      ::UpdateWindow(m_hWnd);
    }
  }

  void MComponent::setEnabled(bool enabled) {
    ::EnableWindow(m_hWnd, enabled ? TRUE : FALSE);
  }

  void MComponent::setMouseCapture(bool capturing) {
    if (capturing) {
      ::SetCapture(m_hWnd);
    } else {
      ::SetCapture(NULL);
    }
  }

  void MComponent::setMouseCursor(MMouseCursor cursor) {
    HCURSOR c = NULL;

    switch (cursor) {
    case Mouse_Normal:
      c = ::LoadCursor(NULL, IDC_ARROW);
      break;
    case Mouse_SizeAll:
      c = ::LoadCursor(NULL, IDC_SIZEALL);
      break;
    case Mouse_SizeEastWest:
      c = ::LoadCursor(NULL, IDC_SIZEWE);
      break;
    case Mouse_SizeNorthSouth:
      c = ::LoadCursor(NULL, IDC_SIZENS);
      break;
    }

    if (c == NULL) {
      c = ::LoadCursor(NULL, IDC_ARROW);
    }

    ::SetCursor(c);
  }

  void MComponent::setBackgroundColour(const MColour &colour) {
    // if the colour has changed, clean up any old brush we had, if any.
    if (colour != bgColour && bgBrush != NULL) {
      ::DeleteObject(bgBrush);
      bgBrush = NULL;
    }

    bgColour = colour;

    // if we have an alpha of zero, then we have basically unset the colour, so indicate as such.
    if (colour.a <= 0.0) {
      bgColourSet = false;
    } else {
      bgColourSet = true;
    }
    refresh();
  }

  void MComponent::setForegroundColour(const MColour &colour) {
    fgColour = colour;
    if (colour.a <= 0.0) {
      fgColourSet = false;
    } else {
      fgColourSet = true;
    }
    refresh();
  } 

  bool MComponent::onMouseMove(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMousePressed(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseReleased(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseDoubleClick(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onMouseWheel(const MMouseEvent &event) {
    return false;
  }

  bool MComponent::onResize(int newWidth, int newHeight) {
    return false;
  }

  bool MComponent::onShow() {
    return false;
  }


  bool MComponent::onPaint() {
    return false;

  }

  void MComponent::onCreate() {

  }

  bool MComponent::onKeyPressed(const MKeyEvent &event) {
    return false;
  }

  bool MComponent::onKeyReleased(const MKeyEvent &event) {
    return false;
  }

  void MComponent::addMouseEventListener(const MMouseEventListenerPtr &listener) {
    mouseListeners.push_back(listener);
  }

  void MComponent::addKeyEventListener(const MKeyEventListenerPtr &listener) {
    keyListeners.push_back(listener);
  }

  void MComponent::fireMouseListeners(const MMouseEvent &event) {
    for (unsigned i = 0; i < mouseListeners.size(); ++i) {
      mouseListeners[i]->onMouseEvent(this, event);
    }
  }

  void MComponent::fireKeyListeners(const MKeyEvent &event) {
    for (unsigned  i = 0; i < keyListeners.size(); ++i) {
      keyListeners[i]->onKeyEvent(this, event);
    }
  }

  bool MComponent::handleKeyEvent(const MKeyEvent &event) {
    bool handled = false;
    fireKeyListeners(event);
    handled = this->onKeyPressed(event);
    
    if (!handled) {
      handled = MApp::getInstance()->onKeyPressed(event);
    }

    return handled;
  }


}
