#include <AztecGUICommonPCH.h>

#include <gui/win32/MScrolledContainerImpl.h>
#include <gui/win32/MAppImpl.h>

#include <gui/MBorderLayout.h>

#include <assert.h>



namespace Aztec {

  MScrolledContainer::MScrolledContainer() {
    horzVisibility = Automatic;
    vertVisibility = Automatic;
    vertVisible = false;
    horzVisible = false;
  }

  void MScrolledContainer::setVisibility(ScrollBarEnum scroller, VisiblityEnum visibility) {
    if (scroller == VerticalScroller) vertVisibility = visibility;
    if (scroller == HorizontalScroller) horzVisibility = visibility;
    if (visibility == Always) {
      setScrollVisible(scroller, true);
    } else if (visibility == Never) {
      setScrollVisible(scroller, false);
    } else {
      doLayout();
    }
  }

  MScrolledContainer::VisiblityEnum MScrolledContainer::getVisibility(ScrollBarEnum scroller) {
    if (scroller == VerticalScroller) {
      return vertVisibility;
    } else if (scroller == HorizontalScroller) {
      return horzVisibility;
    } else {
      assert(0);
      return Never;
    }
  }

  bool MScrolledContainer::onScrolled(ScrollBarEnum scroller) {
    return false;
  }

  int MScrolledContainer::getComponentCount() {
    if (content == NULL) {
      return 0;
    } else {
      return content->getComponentCount();
    }
  }

  Aztec::MComponentPtr MScrolledContainer::getComponent(int index) {
    if (content == NULL) {
      return NULL;
    } else {
      return content->getComponent(index);
    }
  }

  void MScrolledContainer::addComponent(Aztec::MComponentPtr component) {
    if (component == content) return;
    content->addComponent(component);
  }

  void MScrolledContainer::addComponent(Aztec::MComponentPtr component, int index) {
    if (component == content) return;
    content->addComponent(component, index);
  }

  void MScrolledContainer::addComponent(Aztec::MComponentPtr component, Aztec::MConstraint constraints) {
    if (component == content) return;
    content->addComponent(component, constraints);
  }

  void MScrolledContainer::addComponent(Aztec::MComponentPtr component, int index, Aztec::MConstraint constraints) {
    if (component == content) return;
    content->addComponent(component, index, constraints);
  }

  void MScrolledContainer::removeComponent(int index) {
    content->removeComponent(index);
  }

  void MScrolledContainer::removeComponent(Aztec::MComponentPtr component) {
    content->removeComponent(component);
  }

  int MScrolledContainer::getComponentIndex(Aztec::MComponentPtr component) {
    if (content == NULL) {
      return -1;
    } else {
      return content->getComponentIndex(component);
    }
  }

  void MScrolledContainer::setLayoutManager(Aztec::MLayoutManagerPtr layout) {
    content->setLayoutManager(layout);
  }

  MLayoutManagerPtr MScrolledContainer::getLayoutManager() {
    if (content == NULL) {
      return NULL;
    } else {
      return content->getLayoutManager();
    }
  }


  void MScrolledContainer::doLayout() {
    if (content != NULL) {
      MSize2D childSize = getChildSize();
      MSize2D size = getClientSize();

      bool changed = false;
      if (vertVisibility == Automatic) {
        if (childSize.getHeight() > size.getHeight()) {
          changed |= (vertVisible != true);
          size.adjustWidth(-2);
          setScrollVisible(VerticalScroller, true);
        } else {
          changed |= (vertVisible != false);
          setScrollVisible(VerticalScroller, false);
        }
      }

      if (horzVisibility == Automatic) {
        if (childSize.getWidth() > size.getWidth()) {
          changed |= (horzVisible != true);
          size.adjustHeight(-2);
          setScrollVisible(HorizontalScroller, true);
        } else {
          changed |= (horzVisible != false);
          setScrollVisible(HorizontalScroller, false);
        }


      }
      if (!changed) {
        MRect2D validPositions = getValidScrollRect();
        scrollPos.limitToRect(validPositions);

//        if (vertVisible) {
          size.setHeight(childSize.getHeight());
//        }
        if (horzVisible) {
          size.setWidth(childSize.getWidth());
        }
        content->setPosition(-scrollPos.x,-scrollPos.y, size.getWidth(), size.getHeight());
      }
    }
  }

  bool MScrolledContainer::createImpl() {
    if (MContainer::createImpl()) {

      content = new MContainer();
      content->setLayoutManager(new MBorderLayout());

      // call the MContainer's addComponent, because we don't want to call 
      // our overridden version.
      MContainer::addComponent(content, MConstraint());
      return true;
    } else {
      return false;
    }
  }

  bool MScrolledContainer::wndProc(UINT message, WPARAM wParam, LPARAM lParam, LRESULT *result) {

    int diffX = 0;
    int diffY = 0;

    if (message == WM_VSCROLL) {

      switch (LOWORD(wParam)) {
      case SB_LINEUP:
        diffY = -64;
        break;
      case SB_LINEDOWN:
        diffY = 64;
        break;
      case SB_PAGEUP:
        diffY = - (getClientSize().getHeight() * 0.9);
        break;
      case SB_PAGEDOWN:
        diffY = getClientSize().getHeight() * 0.9;
        break;
      case SB_THUMBTRACK:
        diffY = (HIWORD(wParam)) - scrollPos.y;
        break;
      case SB_THUMBPOSITION:
        diffY = (HIWORD(wParam)) - scrollPos.y;
        break;
      }
    }
    if (message == WM_HSCROLL) {

      switch (LOWORD(wParam)) {
      case SB_LINEUP:
        diffX = -64;
        break;
      case SB_LINEDOWN:
        diffX = 64;
        break;
      case SB_PAGEUP:
        diffX = - (getClientSize().getWidth() * 0.9);
        break;
      case SB_PAGEDOWN:
        diffX = getClientSize().getWidth() * 0.9;
        break;
      case SB_THUMBTRACK:
        diffX = (HIWORD(wParam)) - scrollPos.x;
        break;
      case SB_THUMBPOSITION:
        diffX = (HIWORD(wParam)) - scrollPos.x;
        break;
      }
    }


    scrollContainer(diffX, diffY);


    return MContainer::wndProc(message, wParam, lParam, result);
  }


  void MScrolledContainer::setScrollVisible(ScrollBarEnum scroller, bool visible) {

    bool &scrollVisibleFlag = (scroller == VerticalScroller) ? vertVisible : horzVisible;
    DWORD scrollerWindowStyle = (scroller == VerticalScroller) ? WS_VSCROLL : WS_HSCROLL;

    if (scrollVisibleFlag == visible) return;
    scrollVisibleFlag = visible;

    if (visible) {
      ::SetWindowLong(m_hWnd, GWL_STYLE, ::GetWindowLong(m_hWnd, GWL_STYLE) | scrollerWindowStyle);
    } else {
      ::SetWindowLong(m_hWnd, GWL_STYLE, ::GetWindowLong(m_hWnd, GWL_STYLE) & ~scrollerWindowStyle);
    }
    // we have to call set window pos here so the window styles get refreshed.
    RECT r;
    ::GetWindowRect(m_hWnd, &r);
    ::SetWindowPos(m_hWnd, NULL, 0,0,r.right - r.left,r.bottom-r.top, SWP_NOMOVE | SWP_NOZORDER | SWP_NOOWNERZORDER);
    doLayout();
  }

  void MScrolledContainer::scrollContainer(int dx, int dy) {
    MSize2D clientSize = getClientSize();
    MSize2D childSize = getChildSize();

    // if our viewport is bigger than our child component, then we do nothing
    if (clientSize.getWidth() >= childSize.getWidth()) {
      dx = 0;
    } 
    
    if (clientSize.getHeight() >= childSize.getHeight()) {
      dy = 0;
    } 
    
    if (dx != 0 || dy != 0) {
      // limit how much we can move based on where we are, how big our viewport is, and how big our child is.
      MPoint2D newPos(scrollPos.x + dx, scrollPos.y + dy);
      MRect2D validPositions = getValidScrollRect();
      newPos.limitToRect(validPositions);

      dx = newPos.x - scrollPos.x;
      dy = newPos.y - scrollPos.y;
    }

    if (dx != 0 || dy != 0) {
      BOOL result = ScrollWindow(m_hWnd, -dx, -dy, NULL, NULL);
      scrollPos.x += dx;
      scrollPos.y += dy;
    }


    updateScrollInfo();
  }

  MRect2D MScrolledContainer::getValidScrollRect() {
    MSize2D clientSize = getClientSize();
    MSize2D childSize = getChildSize();
    MSize2D validPositions(childSize.getWidth() - clientSize.getWidth(), childSize.getHeight() - clientSize.getHeight());
    
    if (validPositions.getWidth() < 0) {
      validPositions.setWidth(0);
    }
    if (validPositions.getHeight() < 0) {
      validPositions.setHeight(0);
    }

    return validPositions;
  }

  MSize2D MScrolledContainer::getChildSize() {
    if (content != NULL) {
      return content->getPreferredSize();
    } else {
      return MSize2D();
    }
  }

  void MScrolledContainer::updateScrollInfo() {

    MRect2D validRect = getValidScrollRect();

    if (horzVisible) {
      SCROLLINFO info;
      info.cbSize = sizeof(SCROLLINFO);
      info.fMask = SIF_PAGE;
      info.nPage = getClientSize().getWidth();

      SetScrollInfo(m_hWnd, SB_HORZ, &info, FALSE);
      SetScrollRange(m_hWnd, SB_HORZ, 0, getChildSize().getWidth(), FALSE);
      SetScrollPos(m_hWnd, SB_HORZ, scrollPos.x, TRUE);
    }
    if (vertVisible) {
      SCROLLINFO info;
      info.cbSize = sizeof(SCROLLINFO);
      info.fMask = SIF_PAGE;
      info.nPage = getClientSize().getHeight();

      SetScrollInfo(m_hWnd, SB_VERT, &info, FALSE);
      SetScrollRange(m_hWnd, SB_VERT, 0, getChildSize().getHeight(), FALSE);
      SetScrollPos(m_hWnd, SB_VERT, scrollPos.y, TRUE);
    }
  }



}

