#include <AztecGUICommonPCH.h>

#include <gui/win32/MTextFieldImpl.h>
#include <gui/win32/MContainerImpl.h>
#include <gui/win32/MAppImpl.h>

#include <assert.h>

namespace Aztec {

  MTextField::MTextField(int width)
    : MComponent(), minSize(-1,-1)
  {
    lpfnOldWndProc = NULL;
    bgBrush = NULL;
    textWidth = width;
  }

  MTextField::~MTextField() {
  }

  void MTextField::setInternalValue(const std::string &value) {
    m_Value = value;
  }

  void MTextField::addAcceptListener(const TextListenerPtr &listener) {
    acceptListeners.push_back(listener);
  }

  void MTextField::addCancelListener(const TextListenerPtr &listener) {
    cancelListeners.push_back(listener);
  }

  void MTextField::fireAcceptListeners() {
    fireListeners(acceptListeners);
  }

  void MTextField::fireCancelListeners() {
    fireListeners(cancelListeners);
  }

  //
  // Here is a sample subclass function.
  //
  static LONG FAR PASCAL SubClassFunc(   HWND hWnd,
                 WORD Message,
                 WORD wParam,
                 LONG lParam)
  {
    MTextFieldPtr field = AZTEC_CAST(MTextField, MApp::getInstance()->getComponent(hWnd));

    assert(field != NULL);

     //
     // When the focus is in an edit control inside a dialog box, the
     //  default ENTER key action will not occur.
     //
    if (Message == WM_KEYDOWN) {
      bool handled = field->handleKeyEvent(MKeyEvent(MKeyEvent::KEY_DOWN, ::GetTickCount(), field->getKeyCodeFromVK(wParam), field->getStateFromAsync(), field));
       
      if (wParam == VK_TAB || wParam == VK_RETURN) {
        HWND parent = ::GetParent(hWnd);
        HWND next = NULL;

        if (parent != NULL) {
          next = ::GetNextDlgTabItem(parent, hWnd, !(::GetAsyncKeyState(VK_SHIFT) & 0x8000));
          ::SetFocus(next);
        }

        field->fireAcceptListeners();
        field->onAccepted();

        return 0;
      } 
      if (wParam == VK_ESCAPE) {
        field->fireCancelListeners();
        field->onCancelled();
        ::SetFocus(NULL);
      }
    } else if (Message == WM_CHAR) {
      // If we have a char message, ignore any tabs, sicne they are handled by WM_KEYDOWN

      bool handled = field->handleKeyEvent(MKeyEvent(MKeyEvent::KEY_DOWN, ::GetTickCount(), field->getKeyCodeFromChar(wParam), field->getStateFromAsync(), field));

      if (wParam == VK_TAB || wParam == VK_RETURN || wParam == VK_ESCAPE) {
        return 0;
      }
    }  else if (Message == WM_KEYUP) {
      bool handled = field->handleKeyEvent(MKeyEvent(MKeyEvent::KEY_UP, ::GetTickCount(), field->getKeyCodeFromVK(wParam), field->getStateFromAsync(), field));
        

    } else if (Message == WM_NCPAINT) {
      // BUG: For some reason, when a region passes through this window proc, 
      // and then NC_PAINT message is handled, the control's frame isn't 
      // always redrawn. The hack is to set wParam to 1, which means the 
      // entire frame should be drawn.
      wParam = 1;
    }
    
    return CallWindowProc(field->lpfnOldWndProc, hWnd, Message, wParam, lParam);
  }


  bool MTextField::createImpl() {
    MApp *app;
    HWND parentHWND = NULL;

    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    m_hWnd = ::CreateWindow("EDIT", m_Value.c_str(), WS_CHILD | ES_AUTOHSCROLL | WS_TABSTOP,
                            0, 10, 0, 10, parentHWND, NULL, app->getHInstance(), NULL);

    if (m_hWnd != 0) {
      // now subclass us
      lpfnOldWndProc = (WNDPROC)SetWindowLong(m_hWnd, GWL_WNDPROC, (DWORD) SubClassFunc);
      
      app->registerComponent(m_hWnd, this);

      ::SendMessage(m_hWnd, WM_SETFONT, (DWORD)::GetStockObject(DEFAULT_GUI_FONT), 0);

      ::SetWindowLong(m_hWnd, GWL_EXSTYLE, WS_EX_CLIENTEDGE);

      onCreate();

      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }

  void MTextField::setValue(const std::string &value) {
    if (value == m_Value) {
      return;
    }

    m_Value = value;
    if (m_hWnd != NULL) {
      ::SetWindowText(m_hWnd, value.c_str());
    }
  }

  std::string MTextField::getValue() {
    return m_Value;
  }

  bool MTextField::onChanged() {
    return false;
  }

  bool MTextField::onAccepted() {
    return false;
  }

  bool MTextField::onCancelled() {
    return false;
  }

  bool MTextField::isMinimumSizeSet() {
    return true;
  }

  MSize2D MTextField::getMinimumSize() {
    // get the size of our font
    if (minSize.getWidth() < 0 || minSize.getHeight() < 0) {
      HDC dc = ::GetDC(m_hWnd);      

      SIZE size;
      char buf[1024];
      for (int i = 0; i < textWidth; ++i) {
        buf[i] = 'w';
      }
      buf[textWidth] = '\0';

      HFONT editFont = (HFONT)::SendMessage(m_hWnd, WM_GETFONT, 0, 0);

      if (editFont != NULL) {
        ::SelectObject(dc, editFont);
      }

      ::GetTextExtentPoint(dc, buf, textWidth, &size);

      POINT p;

      p.x = size.cx;
      p.y = size.cy;

      if (minSize.getHeight() < 0) {
        minSize.setHeight(p.y + 2 * ::GetSystemMetrics(SM_CYEDGE) + 4);
      }

      if (minSize.getWidth() < 0) {
        minSize.setWidth(p.x);
      }

      ::ReleaseDC(m_hWnd, dc);
    }

    // now take our custom minimum size and our computed minimum size and see which is largest.
    const MSize2D s = m_MinimumSize;

    if (s.getWidth() > minSize.getWidth()) {
      minSize.setWidth(s.getWidth());
    }
    if (s.getHeight() > minSize.getHeight()) {
      minSize.setHeight(s.getHeight());
    }

    return MSize2D::getLargest(minSize, MComponent::getMinimumSize());
  
  }

  void MTextField::fireListeners(const std::vector<TextListenerPtr> &v) {
    MTextFieldPtr field = this;
    for (std::vector<TextListenerPtr>::const_iterator i = v.begin(); i != v.end(); ++i) {
      (*i)->onListen(field);
    }
  }

}

