#ifndef MatrixH
#define MatrixH

#include "ModelGeneric.h"

namespace Aztec {
  class MMath;
  class MVector3;
  class MVector4;
  class Matrix3;
  class MMatrix4;
  class MRay;
  class MPlane;
}

#include "MStr.h"
#include "EulerAngles.h"
#include "MStreams.h"
#include <string>
#ifndef WIN32
#include <math.h>
#endif


namespace Aztec {

#ifndef M_PI
#define M_PI      3.141592653589792328462643383279502884197169399375105
#endif

#define DegToRad(x) ((x)*M_PI/180.0)
#define RadToDeg(x) ((x)*180.0/M_PI)

  /**
   * This is the global math class that provies general math
   * functions and things like that.
   */
  class MGENEXPORT MMath {
  public:
    /**
     * This calculates the smallest distance between the two rays,
     * and provides how far along the rays that closest distance is.
     * This implementation was taken from the Magic Math library.
     * @param ray1 The first ray to use
     * @param ray2 The second ray
     * @param ray1Dist How far along the first ray the closest point is.
     * @param ray2Dist How far along the second ray the closest point is.
     */
    static double distanceBetween(const MRay &ray1, const MRay &ray2, double *ray1Dist = NULL, double *ray2Dist = NULL);

  };

  //----------------------------------------------------------------------------------------
  //  MVector3
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MVector3
  {
  public:
    float x, y, z;
    
    // These functions create and set internal data.
    
    MVector3 () { x = y = z = 0; }
    MVector3 (float x, float y, float z) { set(x, y, z);};
    MVector3 (const MVector3 &Src) { x = Src.x; y = Src.y ; z = Src.z;};
    
    void set (float nx, float ny, float nz) { x = nx; y = ny; z = nz; }
    void set (const MVector3 &v) { x = v.x; y = v.y; z = v.z ;}
    
    float dotProduct(const MVector3 &B) const;
    MVector3 crossProduct(const MVector3 &B) const;
    MVector3 addVector(const MVector3 &B) const;
    MVector3 subVector(const MVector3 &B) const;
    MVector3 scalarbyVector(float B) const;
    
    // Make the Vector Unit Length
    void normalize();
    // Returns length of Vector
    float length() const;
    // Returns Length of Vector Squared
    float length2() const;
    
    
    MVector3& operator=  (const MVector3 &v) { x = v.x; y = v.y; z = v.z; return *this;};
    MVector3& operator+= (const MVector3 &v) { x += v.x; y += v.y; z += v.z; return *this;};
    MVector3& operator-= (const MVector3 &v) { x -= v.x; y -= v.y; z -= v.z; return *this;};
    // Scalar multiplication assignment
    MVector3& operator*= (float a) { x *= a; y *= a; z *= a; return *this;};
    // Scalar division assignment
    MVector3& operator/= (float a) { x /= a; y /= a; z /= a; return *this;};
    // Cross Product Assignment
    MVector3& operator/= (const MVector3 &v) { set(crossProduct(v)); return *this;};
    
    MVector3 operator+  (const MVector3 &v) const { return MVector3(x+v.x, y+v.y, z+v.z); };
    MVector3 operator-  (const MVector3 &v) const { return MVector3(x-v.x, y-v.y, z-v.z); };
    MVector3 operator* (float a) const { return MVector3(x*a, y*a, z*a);};
    MVector3 operator/ (float a) const { return MVector3(x/a, y/a, z/a);};
    
    // Dot Product
    float operator*     (const MVector3 &v) const { return x*v.x + y*v.y + z*v.z; };
    // Cross Product
    MVector3 operator/ (const MVector3 &v) const { return crossProduct(v); };
    
    // In case some one want to use MVector3 as if it were the old TVector type, we use this
    // operational overload to simulate an array of float variables.
    float& operator[] (int i);
    const float& operator[] (int i) const;
    
    // Class conversion operators
    MVector3& operator= (const MVector4 &v);
    
    
    // String conversion operations
    MStr convertToString(const MStr &FormatStr = MStr("%.2f %.2f %.2f")) const;
    bool convertFromString(const MStr &Str);
    
  };
  
  MGENEXPORT bool operator==(const MVector3 &LHS, const MVector3 &RHS);
  
  // Non-member MVector3 operators
  MGENEXPORT MVector3 operator*(float f, const MVector3 &V);
  
  //----------------------------------------------------------------------------------------
  //  Matrix3
  //----------------------------------------------------------------------------------------
  
  class MGENEXPORT Matrix3
  {
  public:
    MVector3 m1, m2, m3;
    
  public:
    Matrix3 () { };
    void set(const MVector3 &x, const MVector3 &y, const MVector3 &z);
    void set(float m11, float m12, float m13,
  	   float m21, float m22, float m23,
       float m31, float m32, float m33);
    Matrix3 (float m11, float m12, float m13,
  	   float m21, float m22, float m23,
       float m31, float m32, float m33) { set(m11,m12,m13,m21,m22,m23,m31,m32,m33); };
    Matrix3 (MVector3 &x, MVector3 &y, MVector3 &z) { set(x,y,z); };
    ~Matrix3 () { };
    
    Matrix3& operator+= (Matrix3 &m);
    Matrix3& operator-= (Matrix3 &m);
    Matrix3& operator*= (Matrix3 &m);
    Matrix3& operator*= (float s);
    
    void transpose ();
    void inverse ();
    float determinant ();
    void transform (MVector3 &f) const;
    void transform (const MVector3 &Src, MVector3& Dest) const;
    void transform (float x, float y, float z, MVector3 &t) const;
    void identity ();
    
    void splitHoriz(MVector3 &A, MVector3 &B, MVector3 &C);
    void splitVert(MVector3 &A, MVector3 &B, MVector3 &C);
    
    void makeHoriz(MVector3 &A, MVector3 &B, MVector3 &C);
    void makeVert(MVector3 &A, MVector3 &B, MVector3 &C);
    
    Matrix3& makeRotationMatrix(float X, float Y, float Z);
    Matrix3& makeRotationMatrix(MVector3 &A);

    // In case some one want to use MVector3 as if it were the old TVector type, we use this
    // operational overload to simulate an array of float variables.
    
    MVector3& Matrix3::operator[] (int i);
  };
  
  //----------------------------------------------------------------------------------------
  //  MVector4
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MVector4
  {
  public:
    float    x,y,z,w;
    
  public:
    MVector4();
    MVector4(const MVector4 &Src);
    MVector4(const MVector3 &Src);
    MVector4(float nx, float ny, float nz, float nw);
    
    // Memeber functions.
    void set(float nx, float ny, float nz, float nw = 1.0);
    
    // Operators
    
    float& operator[](int i);
    float operator[](int i) const;
  };
  
  //----------------------------------------------------------------------------------------
  //  MMatrix4
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MMatrix4
  {
  public:
    float    m[4][4];
    
  public:
    MMatrix4();
    MMatrix4(const float Src[4][4]);
    MMatrix4(const float Src[16]);
    MMatrix4(const double Src[16]);
    MMatrix4(const MMatrix4 &Src);
    MMatrix4(const MVector3 &srcAxis, const MVector3 &axisToMatch);
    
    MMatrix4& operator=(const MMatrix4 &RHS);
    MMatrix4& operator=(const Matrix3 &RHS);
    MMatrix4& operator+=(const MMatrix4 &RHS);
    MMatrix4& operator*=(const MMatrix4 &RHS);
    
    MMatrix4 operator+(const MMatrix4 &RHS) const;
    MMatrix4 operator-(const MMatrix4 &RHS) const;
    
    MMatrix4 operator*(const MMatrix4 &RHS) const;
    MVector4 operator*(const MVector4 &RHS) const;
    MVector3 operator*(const MVector3 &RHS) const;

    MRay operator*(const MRay &rhs) const;
    
    MMatrix4& inverse();
    MMatrix4& transpose();
    
    MMatrix4& identity();
    
    void convertFromEuler(MVector3 src, int Order = EulOrdXYZs);
    // Converts the inner rotation part of thematrix to euler angles in the given rotation order
    void convertToEuler(MVector3 &Dest, int Order = EulOrdXYZs);

    MMatrix4& rotateWithVector(const MVector3 axis, float ang);

    /**
     * This takes the rotation part of the matrix (the top left 3x3 matrix)
     * and normalises the three vectors contained in there.
     */
    void normaliseRotation();

    /**
     * This converts this matrix into a viewing frustum. This code
     * was adapted from Mark Morley's 
     * <A href=http://www.markmorley.com/opengl/frustumculling.html>page</A>. 
     *
     * @param frustum an array of 6 planes to receive the frustum
     */
    void extractFrustum(MPlane frustum[]);

    /**
     * This creates a matrix that transforms one normal vector to another
     */
    void createRotationMatrix(const MVector3 &fromNormal, const MVector3 &toNormal);
  };
  
  
  // rotates vector x, using u as an axis, by angle 'ang'.
  MGENEXPORT void rotateWithVector(const MVector3 &x, MVector3 u, float ang, MVector3 &Res);
  
  
  //----------------------------------------------------------------------------------------
  //  MRay
  //----------------------------------------------------------------------------------------
  
  // This represents an infinite ray, or just a positioned vector in space.
  // If it is a ray, it is interpreted as a parametric equation for the line
  // x(t) = Ox + t*Dx
  // y(t) = Oy + t*Dy
  // z(t) = Oz + t*Dz
  
  class MGENEXPORT MRay
  {
  public:
    MVector3       Org, Dir;
    
    MRay() {};
    MRay(const MVector3 &O, const MVector3 &D) { Org = O; Dir = D;};
    MRay(float ox, float oy, float oz, float dx, float dy, float dz) {Org.set(ox, oy, oz); Dir.set(dx, dy, dz);}
    
    void set(const MVector3 &O, const MVector3 &D) { Org = O; Dir = D;}
    void set(float ox, float oy, float oz, float dx, float dy, float dz) {Org.set(ox, oy, oz); Dir.set(dx, dy, dz);}
    void set(const MVector3 &P1, const MVector3 &P2, float Mult);
    
    MVector3 intersectWithPlane(const MPlane &P) const;
    
    
  };
  
  //----------------------------------------------------------------------------------------
  //  MPlane
  //----------------------------------------------------------------------------------------
  
  // This represents an infinite plane in space, that satisfies the equation
  // Ax + By + Cz + D = 0
  class MGENEXPORT MPlane
  {
  public:
    float A, B, C, D;
    
    MPlane() {};
    MPlane(float nA, float nB, float nC, float nD) { A = nA; B = nB; C = nC; D = nD;};
    MPlane(const MVector3 &Origin, const MVector3 &Normal);
    
    void set(float nA, float nB, float nC, float nD) { A = nA; B = nB; C = nC; D = nD;};
    void set(const MVector3 &Origin, const MVector3 &Normal);

	MVector3 normal() const;

	friend float operator*(const MVector3 &v, const MPlane &p) {
		return v.x * p.A + v.y * p.B + v.z * p.C+ p.D;
	}
  };

  
  /**
   * This is a representation of a hyper-complex number, known as a 
   * Quaternion. This is chiefly used for representing orientation information.
   * It can also be used for spline interpolation and a few other little
   * things.
   *
   * This class was adapted Laurent Schmalen quaternion classes.
   */
  class MQuaternion {
  public:
    MQuaternion();
    MQuaternion(float angle, float x = 0.0, float y = 0.0, float z = 0.0);

    /**
     * Multiplies the two quaternions. This essentially concatenate the
     * two rotations.
     */
    friend MQuaternion operator *(const MQuaternion &lhs, const MQuaternion &rhs) ;
    /**
     * Assignment multiplication operator. 
     */
    inline const MQuaternion& operator *=(const MQuaternion &rhs);

    /**
     * This calculate the conjugate of the quaternion.
     */
    inline const MQuaternion& operator ~();

    /**
     * This inverts the quaternion.
     */
    inline const MQuaternion& operator -();

    /**
     * Normalises a quaternion.
     */
    inline const MQuaternion& normalize();

    /**
     * This linearly interpolates the two qaternions and places
     * the result into this Quaternion. It performs spherical linear 
     * interpolation, so the resulting quaternion takes the shortest
     * path along a sphere between the two otrientations.
     */
    inline void slerp(const MQuaternion &from, const MQuaternion &to, const float t);

    // some additional Quaternion functions
    // getting the exponent of a quaternion
    inline const MQuaternion& exp();
    inline const MQuaternion& log();    

    // conversion functions
    inline const MQuaternion& fromAxis(const float angle, float x, float y, float z);
    inline void toMatrix(float matrix[3][3]) const;
    inline void toMatrix(MMatrix4 &matrix) const;

    inline std::string toString() const;

  protected:
    float W, X, Y, Z;

  };


  // streaming functions
  MOutputStream &operator <<(MOutputStream &out, const MVector3 &vec);
  MInputStream &operator >>(MInputStream &in, MVector3 &vec);

  MOutputStream &operator <<(MOutputStream &out, const MMatrix4 &mat);
  MInputStream &operator >>(MInputStream &in, MMatrix4 &mat);


}
#endif
