
#ifndef MMESH_H 
#define MMESH_H

#include "ModelGeneric.h"

namespace Aztec {

  class MMesh;
  class MAnimMesh;

  typedef MRefCountedPtr<MMesh> MMeshPtr;
  typedef MRefCountedPtr<MAnimMesh> MAnimMeshPtr;
}

#define TRIFLAGS     Aztec::AztecFlags
#define VERTFLAGS    Aztec::AztecFlags

#define VERTEX_SELECTED       ((Aztec::AztecFlags)Aztec::MComponentisedObject::COMPONENT_SELECTED)
#define VERTEX_VISIBLE        (Aztec::AztecFlags)0x02
#define VERTEX_FLAGFORCHANGE  (Aztec::AztecFlags)0x04
#define VERTEX_DRAW_FEEDBACK  (Aztec::AztecFlags)0x08
#define VERTEX_NORMAL_OKAY    (Aztec::AztecFlags)0x10
#define VERTEX_NORMAL_NEEDS_UPDATE  (Aztec::AztecFlags)0x20
#define VERTEX_HAS_COLOUR     (Aztec::AztecFlags)0x40

#define TRIANGLE_SELECTED        ((Aztec::AztecFlags)Aztec::MComponentisedObject::COMPONENT_SELECTED)
#define TRIANGLE_VISIBLE         (Aztec::AztecFlags)0x02
#define TRIANGLE_FLAGFORCHANGE   (Aztec::AztecFlags)0x04
#define TRIANGLE_FLAGFORCHANGE2  (Aztec::AztecFlags)0x08
#define TRIANGLE_HAS_COLOUR      (Aztec::AztecFlags)0x10
#define TRIANGLE_NORMAL_VALID    (Aztec::AztecFlags)0x20

#define EDGE_SELECTED   ((Aztec::AztecFlags)Aztec::MComponentisedObject::COMPONENT_SELECTED)
#define EDGE_VISIBLE    (Aztec::AztecFlags)0x02
#define EDGE_FLAGFORCHANGE ((Aztec::AztecFlags)0x04)

#include "MShapeObject.h"
#include "MBaseObject.h"
#include "MComponentisedObject.h"

#include "MMath.h"
#include "MMaterial.h"

#include <MVertex.h>

#include <vector>
#include <set>

namespace Aztec {

  //------------
  //  MInternalVertex
  //------------
  class MGENEXPORT MInternalVertex : public MFlagObject {
  public:
    MVector3 m_Pos;
    MVector3 m_Normal;
	MVector3 m_VertexColour; 
    MRGBAFloat m_FeedbackColor;
	MVector3 vertexColour; 

    MInternalVertex() { setFlag(VERTEX_VISIBLE);};
    MInternalVertex(const MInternalVertex &Src);
    MInternalVertex(const MVector3 &Src);
    MInternalVertex(float x, float y, float z);

    /**
     * This returns the position of the vertex. 
     * @deprecated Use of this should be removed ASAP!
     */
    inline operator MVector3& () {
      return m_Pos;
    }

    void invalidateNormal() {
      unsetFlag(VERTEX_NORMAL_OKAY);
    }

    /**
     * This gets a transformation matrix which will transform (0,0,1) to 
     * the normal of this triangle.
     */
    MMatrix4 getMatrix();

    MInternalVertex& operator=(const MVector3 &RHS);
  };

  //--------------------
  //  MInternalTriangle
  //--------------------
  class MGENEXPORT MInternalTriangle {
  public:
   
    MInternalTriangle();
    MInternalTriangle(int a, int b, int c);
    MInternalTriangle(const MInternalTriangle &src);
    MInternalTriangle& operator=(const MInternalTriangle &Src);

    /**
     * Gets the index into the mesh's vertex array for the given triangle vertex. 
     */
    int getVertex(int index) const;
    void setVertex(int index, int vertexIndex);

    const MVector3& getNormal() const;
    const MVector3& getCentre() const;
    const MVector3& getColour() const;

    AztecFlags getFlags() const;
    void setFlag(AztecFlags flag);
    void unsetFlag(AztecFlags flag);
    void assignFlag(AztecFlags flag);
    bool isFlagged(AztecFlags flag) const;
    void toggleFlag(AztecFlags flag);

    AztecFlags getEdgeFlag(int index) const;
    void setEdgeFlag(int index, AztecFlags flag);
    void unsetEdgeFlag(int index, AztecFlags flag);
    void assignEdgeFlag(int index, AztecFlags flag);
    bool isEdgeFlagged(int index, AztecFlags flag) const;
    void toggleEdgeFlag(int index, AztecFlags flag);
    /**
     * @return The edge that contains the two given vertices. -1 if no edge has those two vertices.
     */
    int getEdge(int VertA, int VertB) const;

  private:
    MFlagObject triangleFlag;
    MFlagObject edgeFlags[3];
    int         m_Vertices[3];
    MVector3    m_Normal, m_Centre, colour;

    friend class MMesh;
    friend class MEditableMesh;
  };

  typedef MInternalTriangle MTriangle;
  //------------------
  //  MMesh
  //------------------
  class MGENEXPORT MMesh : public MNamedObject, 
                           public virtual MComponentisedObject {
  public:
    friend class MEditableMesh;
    friend class MAnimMesh;
    friend class TriangleGetter;

	  MMesh();
    MMesh(MMeshPtr src);
	  virtual ~MMesh();
    
    // Class related
    virtual MStr getClassName() {return MStr("MMesh");};
    virtual MStr getParentClassName() {return MStr("MBaseObject");};
    virtual MBaseObjectPtr createNew();
    void setFrom(MBaseObjectPtr SrcObj);
    
    int getNumVerts() const;
    int getNumTris() const;
    
    MVector3 getVertexPosition(int index) const;
	void setVertexPosition(int index, MVector3 pos);
    MVector3 getVertexNormal(int index) const;

    void setVertexFlag(int index, AztecFlags flag);
    void unsetVertexFlag(int index, AztecFlags flag);
    bool isVertexFlagged(int index, AztecFlags flag) const;

    // Triangle functions
    const MInternalTriangle* getTriangle(int triIndex) const;
    int getTriangleVertex(int triIndex, int cornerIndex) const;

    const MVector3& getTriangleNormal(int triIndex) const;
    const MVector3& getTriangleCentre(int triIndex) const;
    const MVector3& getTriangleColour(int triIndex) const;

    AztecFlags getTriangleFlags(int triIndex) const;
    void setTriangleFlag(int triIndex, AztecFlags flag);
    void unsetTriangleFlag(int triIndex, AztecFlags flag);
    void assignTriangleFlag(int triIndex, AztecFlags flag);
    bool isTriangleFlagged(int triIndex, AztecFlags flag) const;
    void toggleTriangleFlag(int triIndex, AztecFlags flag);

    AztecFlags getTriangleEdgeFlag(int triIndex, int edgeIndex) const;
    void setTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag);
    void unsetTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag);
    void assignTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag);
    bool isTriangleEdgeFlagged(int triIndex, int edgeIndex, AztecFlags flag) const;
    void toggleTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag);

    /**
     * Tests to see if the edge defined by the vertices <code>from</code> 
     * and <code>to</code> is flagged with the given flags or not.
     */
    bool isEdgeFlagged(int from, int to, Aztec::AztecFlags flag);

    Aztec::AztecFlags getEdgeFlag(int from, int to);

    /**
     * Sets a flag on the edge defined by the two vertices <code>from</code> 
     * and <code>to</code> is flagged with the given flags or not.
     */
    void setEdgeFlag(int from, int to, Aztec::AztecFlags flag);

    void unsetEdgeFlag(int from, int to, Aztec::AztecFlags flag);

    void assignEdgeFlag(int from, int to, Aztec::AztecFlags flag);

    /**
     * @return The edge that contains the two given vertices. -1 if no edge has those two vertices.
     */
    int getTriangleEdge(int triIndex, int VertA, int VertB) const;


    /**
     * @deprecated
     */
    MVertex getVertex(int Num) {
      return MVertex(this, Num);
    }
    int getTriVert(int TriNum, int VertNum) const ;
    int getTriFromEdge(int EdgeNum, int &Tri, int &Edge) const;
    int getVertsFromEdge(int EdgeNum, int *VertA, int *VertB);
    int getVertsFromEdge(int triangle, int edge, int *VertA, int *VertB);
    int getTriangleNeighbour(int triIndex, int edgeIndex);

    /**
     * This gets the vertices that are connected to the given vertex by 
     * mvisible edges. The resulting vertices are given in a consitent 
     * order, so that each consecutive edge extracted could create 
     * a list of neighbouring triangles/polygons.
     */
    void getVerticesConnecting(int vertex, std::vector<int> &vertices, bool includeHidden = false);

    /**
     * Gets all the triangles in the mesh that have the given vertex.
     */
    void getTrianglesWithVertex(int vertex, std::vector<int> &triangles);

    /**
     * Gets a list of all the triangles that have both of the vertices. 
     * Should only ever return zero, one or two triangles for a well 
     * defined manifold mesh.
     */
    void getTrianglesWithEdge(int vertexFrom, int vertexTo, std::vector<int> &triangles);
    
    /**
     * This gets all the points that make up the polygon starting from the 
     * given edge, and continuousling turning the same direction as we walk
     * along it. This function only works on edges with the EDGE_VISIBLE flag 
     * set. It considers any edges that are invisible to indicate the interior
     * of a polygon.
     * 
     * The first point will always be vertexFrom, and the second point will always be vertexTo. 
     *
     * @return true if the edge loop was closed, false if it was open.
     */
    bool getPolygonVerticesFromEdge(int vertexFrom, int vertexTo, bool left, std::vector<int> &points);

    /**
     * This figures out on which side the triangle appears for the edge going 
     * from vertexFrom to vertexTo.
     *
     * @return -1 for left, 1 for right, and 0 for that edge isn't on the triangle.
     */
    int getSideOfEdgeForTriangle(int vertexFrom, int vertexTo, int triangleIndx);

    void getPolygonVerticesFromTriangle(int triangle, std::vector<int> &points);

    int getNumVertsFlagged(AztecFlags Flags);
    int getNumTrisFlagged(AztecFlags Flags);
    
    int createGLVertexArray(float *&Array);
    
    void setTextureMesh(MMeshPtr Mesh);
    MMeshPtr getTextureMesh() { return m_TextureMesh; }
    
    virtual int flagVerticesOnTris(TRIFLAGS TriFlags, VERTFLAGS VertFlags);
    // Gets the bounding box of the mesh in world space
    virtual int getMeshExtents(MVector3 &Min, MVector3 &Max);
    // Gets the bounding box of the mesh after the given transformation.
    virtual int getMeshExtents(MVector3 &Min, MVector3 &Max, const MMatrix4 &XForm);
    
    // Bulk Vertex and triagle flag setting
    void setVertexFlags(AztecFlags Flags);
    void unsetVertexFlags(AztecFlags Flags);
    void setTriangleFlags(AztecFlags Flags);
    void unsetTriangleFlags(AztecFlags Flags);
    
    void calculateNormals();
    
    virtual MMeshPtr convertToMesh();
    virtual bool drawObject(const MBaseObjectPtr &baseObj, MSceneViewFlags ViewFlags);
    virtual int selectComponent(AztecFlags Mode, int CompNum, bool TargetSel);
    virtual int isComponentSelected(AztecFlags Mode, int CompNum);
    virtual void selectComponentToggle(AztecFlags Mode, int CompNum);

	void setVertexColour(int index, const MVector3 &colour);
	MVector3 getVertexColour(int index);
	void unsetVertexColour(int index);

	void setTriangleColour(int index, const MVector3 &colour);
	MVector3 getTriangleColour(int index);
	void unsetTriagleColour(int index);

    /**
     * This looks at the given triangle, and if it has any hidden edges
     * it will travel across the mesh surface flagging any connected 
     * triangles.
     */
    void flagPolygon(int triIndex, AztecFlags flag, MAction action = atSet);

    void getPolygonFromTriangle(int tri, std::set<int> &trisInPoly);

    // MComponentisedObject methods
    bool isInComponentMode(AztecFlags type = POINT_TYPE | EDGE_TYPE | FACET_TYPE);
    void setComponentMode(AztecFlags type);
    AztecFlags getComponentMode();
    bool hasComponentType(AztecFlags type);
    int getComponentCount(AztecFlags type);
    bool flagComponent(AztecFlags type, int index, AztecFlags flag = COMPONENT_SELECTED, MAction action = atSet);
    bool isComponentFlagged(AztecFlags type, int index, AztecFlags flag = COMPONENT_SELECTED);
    bool hasComponentsFlagged(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED);
    bool setComponentFeedbackColour(AztecFlags type, int index, const MVector4 &colour);
    bool unsetComponentFeedbackColour(AztecFlags type, int index);
    MVector4 getComponentFeedbackColour(AztecFlags type, int index);
    bool flagConnected(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    bool growFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    bool shrinkFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    MVector3 getComponentCentre(AztecFlags type, int index);
    MVector3 getFlaggedCentre(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED);
    MVector3 getComponentNormal(AztecFlags type, int index);


    // MUndoableObject methods
    void finishWithUndoNode();

  protected:
    AztecFlags componentMode;

    MFlagObject* getFlagObjectForComponent(AztecFlags type, int index);

    /**
     * This is called when the normals of some of the components may have
     * changed.
     */
    void invalidateNormals();

    void cleanUpMesh();

    typedef std::vector<MInternalVertex*> InternalVertexArray;
    /**
     * These are the actual vertices in the mesh.
     */
    InternalVertexArray m_Verts;
    /**
     * These are the actual triangles in the mesh.
     */
    MInternalTriangle **m_Tris;
    
    /**
     * This is the texture mesh. It defines how textures
     * are applies to the triangles of the mesh.
     */
    MMeshPtr m_TextureMesh;

    /**
     * The number of triangles.
     */
    int m_NumTris;

    // this does the actual work of the polygon flagging.
    // it will not traval to the edge specified by ignoreEdge
    void doFlagPolygon(int triIndex, AztecFlags flag, MAction action = atSet, int ignoreEdge = -1);
  
    /**
     * Gets the edge in the triangle that is adjacent to the given edge 
     * and vertex.
     *
     * For example, in the diagram:
     *
     * 
     *      /\
     *  a  /  \  v
     *    /    \
     *   /      \
     *  / vertex \
     * +----------
     *   edge
     *
     * The function will return edge a, because it is adjacent to edge, and 
     * also uses the given vertex.
     */
    int getNextEdgeAroundVertex(int triangle, int edge, int vertex);

    bool normalsInvalid;

    void drawSelectableMesh(MSceneViewFlags ViewFlags);

    void flagConnectedVertices(int index, AztecFlags flag, std::vector<bool> &visited);

  
    class MeshUndoNode : public MBaseUndoNode {
    public:
      MeshUndoNode(const MMeshPtr &m);

      // MBaseUndoNode Methods
      MUndoableObject* getObject();
      void undo();
      void redo();

      std::vector<MInternalVertex> verts;
      std::vector<MInternalTriangle> tris;
      MMeshPtr textureMesh;
      AztecFlags storedComponentMode;

      MMeshPtr mesh;
    };

    typedef MRefCountedPtr<MeshUndoNode> MeshUndoNodePtr;

    MeshUndoNodePtr undoNode;

    void ensureUndo();

    void copyToUndo(MeshUndoNode *node);
    void swapWithUndo(MeshUndoNode *node);
    friend class MeshUndoNode;

};

}
#endif
