#ifndef Aztec_PluginManager_Header
#define Aztec_PluginManager_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MPluginManager;

  typedef MRefCountedPtr<MPluginManager> MPluginManagerPtr;
}

#include "MBaseObject.h"
#include "MListsTrees.h"
#include "LogFile.h"
#include "MUndoManager.h"
#include <MSceneRenderer.h>
#include <translator/MImageTranslator.h>
#include <translator/MSceneTranslator.h>
#include <translator/MMaterialTranslator.h>

#include <map>

namespace Aztec {

  class MGENEXPORT MPluginManager : public MRefCountedObject {
  public:
    typedef std::vector<MImageTranslatorPtr> ImageTranslatorVector;
    typedef std::vector<MSceneTranslatorPtr> SceneTranslatorVector;

    int registerObject(const MBaseObjectPtr &object);
    MBaseObjectPtr createObjectFromDefault(const MStr &className);

    int registerSceneTranslator(const MSceneTranslatorPtr &translator);
    int getSceneTranslatorCount();
    MSceneTranslatorPtr getSceneTranslator(int index);
    MSceneTranslatorPtr getSceneTranslatorThatImports(const MStr &filename);
    MSceneTranslatorPtr getSceneTranslatorThatExports(const MStr &filename);

    int registerImageTranslator(const MImageTranslatorPtr &translator);
    int getImageTranslatorCount();
    MImageTranslatorPtr getImageTranslator(int index);
    MImageTranslatorPtr getImageTranslatorThatImports(const MStr &filename);
    MImageTranslatorPtr getImageTranslatorThatExports(const MStr &filename);

    int registerMaterialTranslator(const MMaterialTranslatorPtr &translator);
    int getMaterialTranslatorCount();
    MMaterialTranslatorPtr getMaterialTranslator(int index);
    MMaterialTranslatorPtr getMaterialTranslatorThatImports(const MStr &filename);
   
    int registerPrimitive(const MNamedObjectPtr &namedObject, const MStr &menuName);
    int getPrimitiveCount();
    MNamedObjectPtr getPrimitive(int index);


    /**
     * This enum is used for functions so client code can retrieve a number 
     * of translators based on their ability to import and/or export data.
     */
    typedef enum { DONT_CARE, MUST, MUST_NOT } CapabilityEnum;

    /**
     * This gets a list of image translators that have certain capabilities. 
     *
     * @param canImport If DONT_CARE is specified for the canImport argument, 
     *                  than it doesn't matter if the translator can import 
     *                  or not. If it is MUST, then a translator will only be 
     *                  returned if it can import files. If it is MUST_NOT, 
     *                  then the translator will only be returned if it cannot 
     *                  import files.
     * @param canExport If DONT_CARE is specified for the canExport argument, 
     *                  than it doesn't matter if the translator can export 
     *                  or not. If it is MUST, then a translator will only be 
     *                  returned if it can export files. If it is MUST_NOT, 
     *                  then the translator will only be returned if it cannot 
     *                  export files.
     *
     * @return the list of translators that matches the given criteria
     */
    ImageTranslatorVector getImageTranslators(
        CapabilityEnum canImport = DONT_CARE, 
        CapabilityEnum canExport = DONT_CARE);

    /**
     * This gets a list of scene translators that have certain capabilities. 
     *
     * @param canImport If DONT_CARE is specified for the canImport argument, 
     *                  than it doesn't matter if the translator can import 
     *                  or not. If it is MUST, then a translator will only be 
     *                  returned if it can import files. If it is MUST_NOT, 
     *                  then the translator will only be returned if it cannot 
     *                  import files.
     * @param canExport If DONT_CARE is specified for the canExport argument, 
     *                  than it doesn't matter if the translator can export 
     *                  or not. If it is MUST, then a translator will only be 
     *                  returned if it can export files. If it is MUST_NOT, 
     *                  then the translator will only be returned if it cannot 
     *                  export files.
     *
     * @return the list of translators that matches the given criteria
     */
    SceneTranslatorVector getSceneTranslators(
        CapabilityEnum canImport = DONT_CARE, 
        CapabilityEnum canExport = DONT_CARE);
    
    void registerRenderer(const MSceneRendererPtr &renderer);
    int getRendererCount();
    MSceneRendererPtr getRenderer(int index);
    MSceneRendererPtr getRenderer(const std::string &name);

  protected:

    typedef std::map<std::string, MBaseObjectPtr> ObjectMap;
    typedef std::map<std::string, MSceneRendererPtr> RendererMap;
    typedef std::vector<MMaterialTranslatorPtr> MaterialTranslatorVector;

    ImageTranslatorVector imageTranslators;
    SceneTranslatorVector sceneTranslators;
    MaterialTranslatorVector materialTranslators;

    ObjectMap objectTemplates;
    std::vector<MNamedObjectPtr> primitives;
   
    RendererMap renderers;
  };
  
}

#endif 
