#ifndef MStreams_Header
#define MStreams_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MInputStream;
  class MOutputStream;

}

#include "MStr.h"
#include <string>

namespace Aztec {

  class MGENEXPORT MStream {
  public:
    /**
     * virtual destructor so every derived class will 
     * definately have viertual destructors.
     */
    virtual ~MStream() { };

    virtual void close() = 0;

    virtual bool isAtEOF() = 0;
  };

  /**
   * This is a base class for an input stream class made 
   * to work with streams. It proviedes a simple interface
   * to read information from a stream..
   */
  class MGENEXPORT MInputStream : public MStream {
  public:
    /**
     * Reads in one character and returns it
     */
    virtual char readChar() = 0;

    /**
     * Reads in one unsigned character and returns it
     */
    virtual unsigned char readUnsignedChar() = 0;

    /**
     * Reads in one short integer and returns it.
     */
    virtual short readShort() = 0;

    /**
     * Reads in one unsigned short integer and returns it.
     */
    virtual unsigned short readUnsignedShort() = 0;

    /**
     * Reads in one integer and returns it.
     */
    virtual int readInt() = 0;

    /**
     * Reads in one unsigned integer and returns it.
     */
    virtual unsigned int readUnsignedInt() = 0;

    /**
     * Reads in one float and returns it.
     */
    virtual float readFloat() = 0;

    /**
     * Reads in one double and returns it.
     */
    virtual double readDouble() = 0;

    /**
     * Reads in one boolean and returns it.
     */
    virtual bool readBoolean() = 0;

    /**
     * Reads in a string and returns it.
     */ 
    virtual MStr readString() = 0;

    /** 
     * Reads in a buffer of bytes. Note that the buffer 
     * received may not necessarily by the physical 
     * representation of that buffer.
     *
     * @param buffer The buffer to receive the data.
     * @param bufSize The size of the buffer to receive the data
     * @return The number of bytes used in the buffer.
     */
    virtual int readBuffer(void *buffer, int bufSize) = 0;


  };

  /**
   * This is the vase class for all output streams. It 
   * provides an interface to operate on writable streams.
   */
  class MGENEXPORT MOutputStream : public MStream {
  public:
    /**
     * Writes the given char to the stream.
     */
    virtual void writeChar(char value) = 0;

    /**
     * Writes the given char to the stream.
     */
    virtual void writeUnsignedChar(unsigned char value) = 0;

    /**
     * Writes the given short to the stream.
     */
    virtual void writeShort(short value) = 0;

    /**
     * Writes the given unsigned short to the stream.
     */
    virtual void writeUnsignedShort(unsigned short value) = 0;

    /**
     * Writes the given integer to the stream.
     */
    virtual void writeInt(int value) = 0;

    /**
     * Writes the given unsigned integer to the stream.
     */
    virtual void writeUnsignedInt(unsigned int value) = 0;

    /**
     * Writes the given float to the stream.
     */
    virtual void writeFloat(float value) = 0;

    /**
     * Writes the given double to the stream.
     */
    virtual void writeDouble(double value) = 0;

    /**
     * Writes the given boolean to the stream.
     */
    virtual void writeBoolean(bool value) = 0;

    /**
     * Writes the null terminates string to the stream.
     */
    virtual void writeString(const char *value) = 0;

    /** 
     * This writes out a set of bytes. Note that data given may
     * not be the physical data written to the stream. It 
     * could be compressed, or anything. The onyl guarantee is that
     * when read back, it will read back exactly what is put in.
     *
     * @param buffer The buffer to write.
     * @param bufSize The number of bytes to write.
     */
    virtual void writeBuffer(const void *buffer, int bufSize) = 0;
  };

  MOutputStream &operator <<(MOutputStream &out, char ch);
  MOutputStream &operator <<(MOutputStream &out, int i);
  MOutputStream &operator <<(MOutputStream &out, long i);
  MOutputStream &operator <<(MOutputStream &out, float fl);
  MOutputStream &operator <<(MOutputStream &out, double dbl);
  MOutputStream &operator <<(MOutputStream &out, bool b);
  MOutputStream &operator <<(MOutputStream &out, BYTE b);
  MOutputStream &operator <<(MOutputStream &out, WORD w);
  MOutputStream &operator <<(MOutputStream &out, DWORD dw);
  MOutputStream &operator <<(MOutputStream &out, const MStr &str);
  MOutputStream &operator <<(MOutputStream &out, const char *str);
  MOutputStream &operator <<(MOutputStream &out, const std::string &str);

  MInputStream &operator >>(MInputStream &in, char &ch);
  MInputStream &operator >>(MInputStream &in, int &i);
  MInputStream &operator >>(MInputStream &in, long &i);
  MInputStream &operator >>(MInputStream &in, float &fl);
  MInputStream &operator >>(MInputStream &in, double &dbl);
  MInputStream &operator >>(MInputStream &in, bool &b);
  MInputStream &operator >>(MInputStream &in, BYTE &b);
  MInputStream &operator >>(MInputStream &in, WORD &w);
  MInputStream &operator >>(MInputStream &in, DWORD &dw);
  MInputStream &operator >>(MInputStream &in, MStr &str);
  MInputStream &operator >>(MInputStream &in, char *str);
  MInputStream &operator >>(MInputStream &in, std::string &str);

}

#endif
