#ifndef MArrayParameter_Header
#define MArrayParameter_Header

#include "ModelGeneric.h"

namespace Aztec {
  
  class MArrayParameter;
  typedef MRefCountedPtr<MArrayParameter> MArrayParameterPtr;

}

#include <params/MAggregateParameter.h>
#include <MParameterObject.h>

namespace Aztec {

  /**
   * MArrayParameter is a parameter that contains an array of values. This
   * is an interface class so that various objects can provide their own
   * implementation. For example, in a mesh class, there would be an array
   * of vectors containing the position of each point in the mesh. It would
   * be very inefficient to have an array of actual MVector3Parameter objects,
   * so we provide an interface so that the Mesh class can provide a much more
   * efficient implementation which access the mesh internal storage.
   */
  class MGENEXPORT MArrayParameter : public virtual MParameterObject {
  public:
    /**
     * This gets the number of parameters in this aggregate
     */
    virtual int getElementCount() = 0;

    /**
     * Adds an element to the end of the array.
     *
     * @return The index of the element.
     */
    virtual int addElement() = 0;

    /**
     * Removes an element from the array.
     * @param index The index of the element.
     */
    virtual void removeElement(int index) = 0;

    /**
     * Sets the number of elements in the array. If the array's size needs
     * to be reduced, then elements are removed from the end. If an array
     * needs to be exapnded, some default value is used to fill out the array.
     */
    virtual void resize(int size) = 0;

    /**
     * This gets the parameter member at the given index
     */
    virtual MParameterObjectPtr getElement(int index) = 0;

    virtual bool getElement(int index, MStr &value) = 0;
    virtual bool getElement(int index, int &value) = 0;
    virtual bool getElement(int index, float &value) = 0;
    virtual bool getElement(int index, bool &value) = 0;
    virtual bool getElement(int index, MVector3 &value) = 0;
    virtual bool getElement(int index, MAggregateParameterPtr &value) = 0;
    virtual bool getElement(int index, MArrayParameterPtr &array) = 0;

    virtual bool setElement(int index, const MStr &value) = 0;
    virtual bool setElement(int index, int value) = 0;
    virtual bool setElement(int index, float value) = 0;
    virtual bool setElement(int index, bool value) = 0;
    virtual bool setElement(int index, const MVector3 &value) = 0;
    virtual bool setElement(int index, const MAggregateParameterPtr &value) = 0;
    virtual bool setElement(int index, const MArrayParameter &value) = 0;

    virtual int getElementType() const = 0;

    // MParameterObject methods
    int getDataType() const;

    bool getValueString(MStr &value);
    bool getValueInteger(int &value);
    bool getValueFloat(float &value);
    bool getValueBoolean(bool &value);
    bool getValueVector(MVector3 &value);
    bool setValueString(const MStr &value);
    bool setValueInteger(int value);
    bool setValueFloat(float value);
    bool setValueBoolean(bool value);
    bool setValueVector(const MVector3 &value);

    // MBaseObject methods
    virtual MStr getClassName() { return "MArrayParameter"; }
  
  protected:
    MArrayParameter() {
    }

    ~MArrayParameter() {
    }

  };

  /**
   * This is just the ArrayParameter class with all the element getting
   * and setting methods implemented to do nothing and just return false.
   */
  class MGENEXPORT MArrayParameterAdapter : public virtual MArrayParameter {
  public:
    // MArrayParameter methods
    bool getElement(int index, MStr &value) { return false; }
    bool getElement(int index, int &value) { return false; }
    bool getElement(int index, float &value) { return false; }
    bool getElement(int index, bool &value) { return false; }
    bool getElement(int index, MVector3 &value) { return false; }
    bool getElement(int index, MAggregateParameterPtr &value) { return false; }
    bool getElement(int index, MArrayParameterPtr &array) { return false; }

    bool setElement(int index, const MStr &value) { return false; }
    bool setElement(int index, int value) { return false; }
    bool setElement(int index, float value) { return false; }
    bool setElement(int index, bool value) { return false; }
    bool setElement(int index, const MVector3 &value) { return false; }
    bool setElement(int index, const MAggregateParameterPtr &value) { return false; }
    bool setElement(int index, const MArrayParameter &value) { return false; }

    // MParameterObject methods
    bool setValueParameter(const MParameterObjectPtr &value);

  };

}

#endif
