#include "StdAfx.h"

#include "MClusterModifier.h"
#include "MMesh.h"
#include "MEditableMesh.h"

#include "MSystemManager.h"

#include <params/MParameterFactory.h>
#include <params/MArrayParameter.h>

#include <GL/gl.h>
#include <assert.h>
#include <stdio.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  class IntArrayElementParam : public MIntParameterAdapter {
  public:
    IntArrayElementParam(MArrayParameter *array, int index) 
      : array(array), index(index)
    {
      setStorable(true);
    }

    // MIntParameter methods
    int getValue() {
      int value = 0;
      array->getElement(index, value);
      return value;
    }

    // MParameterObject methods
    MStr getShortName() const {
      char buf[16];
      sprintf(buf, "p[%i]", index);
      return buf; 
    }

    MStr getLongName() const {
      char buf[16];
      sprintf(buf, "p[%i]", index);
      return buf; 
    }

    MStr getFriendlyName() const {
      char buf[16];
      sprintf(buf, "p[%i]", index);
      return buf; 
    }

    MBaseObject* getOwner() const {
      return array->getOwner();
    }

    void setOwner(MBaseObject *newOwner) {
      assert(newOwner == getOwner());
    }

    bool isInputParameter() {
      return true;
    }

    bool setValueInteger(int value) {
      return array->setElement(index, value);
    }

    bool createKeyAt(long time) {
      return false;
    }

    bool setFromParameter(MParameterObjectPtr src) {
      return setValueParameter(src);
    }

    bool isConnectable() {
      return false;
    }

    bool isConnected() {
      return false;
    }

    bool setInputParameter(const MParameterObjectPtr &input) {
      return false;
    }

    MParameterObjectPtr getInputParameter() {
      return NULL;
    }



  protected:
    MArrayParameter *array;
    int index;
  };

  class ClusterArrayParam : public MArrayParameterAdapter {
  public:
    ClusterArrayParam(MClusterModifier *modifier) {
      clusterMod = modifier;
      setStorable(true);
    }

    ~ClusterArrayParam() {
    }

    int getElementCount() {
      return clusterMod->getNumPoints();
    }

    int addElement() {
      clusterMod->activePoints.push_back(MClusterModifier::PointPair(-1, 1.0));
      return clusterMod->activePoints.size() - 1;
    }

    void removeElement(int index) {
      clusterMod->activePoints.erase(clusterMod->activePoints.begin() + index);
    }

    void resize(int size) {
      clusterMod->activePoints.resize(size);
    }

    MParameterObjectPtr getElement(int index) {
      return new IntArrayElementParam(this, index);
    }

    bool getElement(int index, MStr &value) {
      int intval;
      if (getElement(index, intval)) {
        value = intToStr(intval);
        return true;
      }

      return false;
    }

    bool getElement(int index, int &value) {
      if (index < 0 || index >= (int)clusterMod->activePoints.size()) {
        return false;
      }
      value = clusterMod->activePoints[index].pointIndex;
      return true;
    }

    bool getElement(int index, float &value) {
      int intval;
      if (getElement(index, intval)) {
        value = (float)intval;
        return true;
      }

      return false;
    }

    bool setElement(int index, const MStr &value) {
      return setElement(index, strToInt(value));
    }

    bool setElement(int index, int value) {
      clusterMod->activePoints[index].pointIndex = value;
      return true;
    }

    bool setElement(int index, float value) {
      return setElement(index, (int)value);
    }

    int getElementType() const {
      return TYPE_INTEGER;
    }

    // MParameterObject methods
    int getDataMeaning() const {
      return MEANING_ANY;
    }

    MStr getShortName() const {
      return "p";
    }

    MStr getLongName() const {
      return "points";
    }

    MStr getFriendlyName() const {
      return "Points";
    }

    MBaseObject* getOwner() const {
      return clusterMod;
    }

    void setOwner(MBaseObject *newOwner) {
      assert(newOwner == clusterMod);
    }

    bool isInputParameter() {
      return true;
    }

    bool setValueParameter(const MParameterObjectPtr &value) {
      return false;
    }

    bool createKeyAt(long time) {
      return false;
    }

    bool setFromParameter(MParameterObjectPtr src) {
      return setValueParameter(src);
    }

    bool isConnectable() {
      return false;
    }

    bool isConnected() {
      return false;
    }

    bool setInputParameter(const MParameterObjectPtr &input) {
      return false;
    }

    MParameterObjectPtr getInputParameter() {
      return NULL;
    }

    MParameterObjectPtr createCopy() {
      // TODO: return a parameter object that will contain a copy of 
      // these values

      return NULL;
    }

    // MBaseObject methods
    MStr getClassName() { return "ClusterArrayParam"; }
  
  protected:
    MClusterModifier *clusterMod;
  };


  //---------------------
  //  MClusterModifier
  //---------------------

  MClusterModifier::MClusterModifier() {
  
    addParameter(m_ControlParam = MParameterFactory::createObject("ctrlObj", "controlObject", "Control Object"));
    addParameter(lastControlObject = MParameterFactory::createObject("cachedCtrlObj", "cachedControlObject", "Cached Control Object"));
    addParameter(lastInputShapeParam = MParameterFactory::createObject("cachedInMesh", "cachedInputMesh", "Cached Input Mesh"));
    addParameter(m_InitialXFormInv = MParameterFactory::createMatrix("xforminv", "xforminv", "xforminv"));

    lastControlObject->setVisible(false);
    m_InitialXFormInv->setVisible(false);

    ClusterArrayParam *pointsParam = new ClusterArrayParam(this);
    pointsParam->setVisible(false);

    addParameter(pointsParam);

    m_PivotSet = false;

    outputMesh = new MEditableMesh();
  }

  MClusterModifier::~MClusterModifier()
  {
    setControllingObject(NULL);
  
    cleanPoints();
  }

  MBaseObjectPtr MClusterModifier::createNew()
  {
    MClusterModifierPtr NewObj;
    NewObj = new MClusterModifier;
  
    return NewObj;
  }

  void MClusterModifier::setFrom(MBaseObjectPtr SrcObj) {
    if (SrcObj == NULL)
      return;
  
    MMeshModifier::setFrom(SrcObj);
  
    MClusterModifierPtr Obj;
  
    Obj = AZTEC_CAST(MClusterModifier, SrcObj);
  
    if (Obj == NULL) {
      return;
    }

    // TODO: implement this part
  
  }

  bool MClusterModifier::doUpdateObject() {
    MMeshModifier::doUpdateObject();
    if (modifyObject()) {
      MMeshShapePtr shape = AZTEC_CAST(MMeshShape, getOutputParameter()->getValue());

      if (shape == NULL) {
        new MMeshShape(outputMesh);
      } else {
        shape->setMeshObject(outputMesh);
      }

    }
    return true;
  }

  bool MClusterModifier::checkInputObject() {
  
    // if our last and current shapes are the same, then we do
    // not need to do any initialisation.
    if (lastInputShapeParam->getValue() == getInputShape()) {
      // return true indicating a valid shape if it isn't null.
      return getInputShape() != NULL;
    }

    // if we have gotten here, then the last shape and current shape were
    // different, so we need to intialise everything.

    cleanPoints();
    MMeshShapePtr lastInputShape = getInputShape();

    if (getInputParameter()->isConnected()) {
      lastInputShapeParam->setInputParameter( getInputParameter()->getInputParameter() );
    } else {
      lastInputShapeParam->setValue(lastInputShape);
    }

    if (lastInputShape != NULL) {
      // see if the input object is a MMesh derived object
      MMeshPtr Mesh = lastInputShape->getMeshObject();
    
      // If it is not, reject it as the input node
      if (Mesh == NULL) {
        return false;
      }
    
      // Set up the cluster using the vertices marked for changing.
      int      n;

      activePoints.clear();

      for (n=0; n<Mesh->getNumVerts(); n++) {
        if (Mesh->isVertexFlagged(n, VERTEX_FLAGFORCHANGE)) {
          activePoints.push_back(PointPair(n, 1.0));
        }
      }
    
    }

    return true;
  }

  int MClusterModifier::modifyObject() {
    if (!checkInputObject()) {
      return 0;
    }
    getControllingObject();

    if (getInputShape() == NULL || m_ControlParam->getValue() == NULL) {
      return 0;
    }
  
    MMeshPtr InMesh;
    MSystemManagerPtr SysMan;
  
    SysMan = Aztec::getSystemManager();
  
    InMesh = getInputShape()->getMeshObject();
  
    InMesh->updateObject();
    // update our input mesh, but keep our flag information.
    outputMesh->setFromMesh(InMesh, false, true);
  
    outputMesh->unsetFlag((AztecFlags)0xFFFFFFFF);
    outputMesh->setFlag(InMesh->getFlags());
  
    // transform the vertices according to the formula
    // w.TObject^-1.TCluster.TObject.v - w.v + v = vCLocal
    // which was derived in ClusterModifers.doc
    MMatrix4  objectXForm, objectXFormInv, clusterXForm, xform, clusterOrig;
    float w;
    MScenePtr scene;
    
    scene = SysMan->getScene();

    scene->getWorldTransformMatrix(scene->getObjectList()->findObject(m_ControlParam->getValue()), clusterXForm);
    scene->getWorldTransformMatrix(scene->getObjectList()->findObject(InMesh), objectXForm);

    objectXFormInv = objectXForm;
    objectXFormInv.inverse();

    clusterXForm = m_InitialXFormInv->getValue() * clusterXForm;

    xform = objectXFormInv;
    xform *= clusterXForm;
    xform *= objectXForm;

    for (unsigned int index = 0; index < activePoints.size(); ++index) {
      MVector3    TempVec, srcVec;
    
      w = activePoints[index].pointWeight;

      srcVec = InMesh->getVertexPosition(activePoints[index].pointIndex);

      TempVec = w * (xform * srcVec);
      TempVec -= w * srcVec;
      TempVec += srcVec;

      outputMesh->setVertexPosition(activePoints[index].pointIndex, TempVec);
    }

    outputMesh->calculateNormals();
  
    return 1;
  }

  void MClusterModifier::setControllingObject(MSceneObjectPtr Obj) {
    m_ControlParam->setValue(Obj);
  }

  MSceneObjectPtr MClusterModifier::getControllingObject() {
    if (lastControlObject->getValue() != m_ControlParam->getValue()) {
      lastControlObject->setValue(m_ControlParam->getValue());
      if (lastControlObject->getValue() != NULL) {
        MSceneObject *controlObject = AZTEC_CAST(MSceneObject, lastControlObject->getValue());
        if (m_PivotSet == false) {
          MVector3    TranslateVec;
          MMatrix4 xform;
          
          TranslateVec = controlObject->getTransformObject()->getTranslateVector(getTime());
          m_PivotPoint = TranslateVec;
          
          {
            MSystemManagerPtr SysMan;
            SysMan = Aztec::getSystemManager();
            SysMan->getScene()->getWorldTransformMatrix(SysMan->getScene()->getObjectList()->findObject(controlObject), xform);
          }
          xform.inverse();
          m_InitialXFormInv->setValueMatrix(xform);
        }
        
      } else {
        MMatrix4 identity;
        identity.identity();
        m_InitialXFormInv->setValueMatrix(identity);
      }
    }
    return AZTEC_CAST(MSceneObject, lastControlObject->getValue());
  }

  int MClusterModifier::getNumPoints() {
    return activePoints.size();
  }

  void MClusterModifier::setPointWeight(int index, float weight) {
    activePoints[index].pointWeight = weight;
  }

  bool MClusterModifier::isPointSelected(int index) {
    return activePoints[index].selected = true;
  }

  int MClusterModifier::drawObject(const MBaseObjectPtr &baseObj, MSceneViewFlags ViewFlags) {
    if (outputMesh == NULL) {
      return 0;
    }
  
    unsigned int               n;
    MSystemManagerPtr SysMan;
  
    SysMan = Aztec::getSystemManager();
  
    if ((ViewFlags.m_ComponentMode == MComponentisedObject::POINT_TYPE)) {
      if (!isFlagged(OBJECTFLAG_SELECTED))
        return 1;
    
      if (ViewFlags.m_SelectMethod == smNone)
      {
      
        glColor3f(0,1,1);
      
        glBegin(GL_POINTS);
        for (n = 0; n < activePoints.size(); n++) {
          if (activePoints[n].selected) {
            glColor3f(1,1,0);
          } else {
            glColor3f(0,1,activePoints[n].pointWeight);
          }
        
          glVertex3fv((float*)&outputMesh->getVertexPosition(activePoints[n].pointIndex));
        }
        glEnd();
      
      } else {
        glPushName(SELECTITEM_VERTEX);
        glPushName(-1);
        for (n = 0; n < activePoints.size(); n++) {
          glLoadName(n);
          glBegin(GL_POINTS);
          glVertex3fv((float*)&outputMesh->getVertexPosition(activePoints[n].pointIndex));
          glEnd();
        }
        glPopName();
        glPopName();
      }
    }
  
    return 1;
  }

  int MClusterModifier::selectComponent(AztecFlags Mode, int CompNum, bool TargetSel) {
    if (Mode & MComponentisedObject::POINT_TYPE) {
      if (TargetSel) {
        activePoints[CompNum].selected = true;
      } else {
        activePoints[CompNum].selected = false;
      }
    
      return 1;
    }
  
    return 0;
  }

  bool MClusterModifier::hasComponentsSelected(AztecFlags Mode) {
    unsigned int         n;
  
    for (n = 0; n < activePoints.size(); n++) {
      if (activePoints[n].selected) {
        return true;
      }
    }
  
    return false;
  }

  void MClusterModifier::cleanPoints() {
    activePoints.clear();
  }

}
