#include "StdAfx.h"

#include "MDAGraph.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  using namespace std;

  MDAGraph::MDAGraph() {
  }

  MDAGraph::~MDAGraph() {
  }

  void MDAGraph::addNode(MDAGNode *node) {
    ensureUndo();
    allNodes.insert(node);
  }

  void MDAGraph::removeNode(MDAGNode *node) {
    ensureUndo();
    allNodes.erase(node);
  }

  void MDAGraph::flagOutputs(MDAGNode *node, AztecFlags flag) {
    // we iterate over all the nodes in the graph from the
    // given node onwards.
    MDAGraphDownstreamIterator it(node);

    ++it;
    while (!it.atEnd()) {
      it->setFlag(flag);

      ++it;
    }
  }

  bool MDAGraph::doesNodeOutputTo(MDAGNode *from, MDAGNode *to) {
    // here we iterate over all the outputs, and see if we come across
    // our node.
    MDAGraphDownstreamIterator it(from);

    while (!it.atEnd()) {
      if ((*it) == to) {
        return true;
      }

      ++it;
    }

    // if we couldn't find it, then return false.
    return false;

  }

  void MDAGraph::finishWithUndoNode() {
  }

  void MDAGraph::ensureUndo() {
    doEnsureUndo(undo, this);
  }

  MDAGraph::Undo::Undo(MDAGraph *graph) {
    object = graph;
    stored = graph->allNodes;
  }

  MUndoableObject* MDAGraph::Undo::getObject() {
    return object;
  }

  void MDAGraph::Undo::undo() {
    std::swap(object->allNodes, stored);
  }

  void MDAGraph::Undo::redo() {
    std::swap(object->allNodes, stored);
  }
  


  MDAGraphDownstreamIterator::MDAGraphDownstreamIterator() {
    graph = NULL;
    curNode = NULL;
  };

  MDAGraphDownstreamIterator::MDAGraphDownstreamIterator(MDAGNode *node) {
    graph = NULL;
    curNode = node;

    visited.insert(curNode);
    addOutputs(curNode);
  }

  MDAGraphDownstreamIterator::MDAGraphDownstreamIterator(MDAGraph *graph, MDAGNode *node) {
    this->graph = graph;
    curNode = node;

    visited.insert(curNode);
    addOutputs(curNode);
  }

  MDAGraphDownstreamIterator::MDAGraphDownstreamIterator(const MDAGraphDownstreamIterator &src) 
    : graph(src.graph), curNode(src.curNode), visited(src.visited), tovisit(src.tovisit)
  {
  }

  MDAGraphDownstreamIterator::~MDAGraphDownstreamIterator() {
  }

  MDAGNode* MDAGraphDownstreamIterator::operator*() {
    return curNode;
  }

  MDAGNode* MDAGraphDownstreamIterator::operator->() {
    return curNode;
  }

  MDAGraphDownstreamIterator& MDAGraphDownstreamIterator::operator++() {
    // We can't do anything if we have no more nodes to visit.
    if (tovisit.size() == 0) {
      curNode = NULL;
      return *this;
    }

    // here we pop off what we currently have

    MDAGNode *newNode = tovisit.top();
    tovisit.pop();

    visited.insert(newNode);
    addOutputs(newNode);

    curNode = newNode;

    return *this;
  }

  MDAGraphDownstreamIterator MDAGraphDownstreamIterator::operator++(int) {
    MDAGraphDownstreamIterator iter(*this);

    ++iter;

    return iter;
  }


  bool MDAGraphDownstreamIterator::haveVisited(MDAGNode *node) {
    return (visited.find(node) != visited.end());
  }

  void MDAGraphDownstreamIterator::addOutputs(MDAGNode *node) {
    // canonly add outputs if we have a non null node
    if (node != NULL) {
      // iterate over the output nodes, and add them
      // if we haven't visited them yet
      MDAGNode::NodeList::iterator outIter;

      for (outIter = node->outputNodes.begin(); outIter != node->outputNodes.end(); ++outIter) {
        if (!haveVisited(*outIter)) {
          tovisit.push(*outIter);
        }
      }

    }
  }


  MDAGraphUpstreamIterator::MDAGraphUpstreamIterator() {
    graph = NULL;
    curNode = NULL;
  };

  MDAGraphUpstreamIterator::MDAGraphUpstreamIterator(MDAGNode *node) {
    graph = NULL;
    curNode = node;

    visited.insert(curNode);
    addInputs(curNode);
  }

  MDAGraphUpstreamIterator::MDAGraphUpstreamIterator(MDAGNode *node, const std::vector<MDAGNode*> &ignoreList) {
    for (unsigned int i = 0; i < ignoreList.size(); ++i) {
      visited.insert(ignoreList[i]);
    }

    graph = NULL;
    curNode = node;

    visited.insert(curNode);
    addInputs(curNode);
  }

  MDAGraphUpstreamIterator::MDAGraphUpstreamIterator(MDAGraph *graph, MDAGNode *node) {
    this->graph = graph;
    curNode = node;

    visited.insert(curNode);
    addInputs(curNode);
  }

  MDAGraphUpstreamIterator::MDAGraphUpstreamIterator(const MDAGraphUpstreamIterator &src) 
    : graph(src.graph), curNode(src.curNode), visited(src.visited), tovisit(src.tovisit)
  {
  }

  MDAGraphUpstreamIterator::~MDAGraphUpstreamIterator() {
  }

  MDAGNode* MDAGraphUpstreamIterator::operator*() {
    return curNode;
  }

  MDAGNode* MDAGraphUpstreamIterator::operator->() {
    return curNode;
  }

  MDAGraphUpstreamIterator& MDAGraphUpstreamIterator::operator++() {
    // We can't do anything if we have no more nodes to visit.
    if (tovisit.size() == 0) {
      curNode = NULL;
      return *this;
    }

    // here we pop off what we currently have

    MDAGNode *newNode = tovisit.top();
    tovisit.pop();

    visited.insert(newNode);
    addInputs(newNode);

    curNode = newNode;

    return *this;
  }

  MDAGraphUpstreamIterator MDAGraphUpstreamIterator::operator++(int) {
    MDAGraphUpstreamIterator iter(*this);

    ++iter;

    return iter;
  }


  bool MDAGraphUpstreamIterator::haveVisited(MDAGNode *node) {
    return (visited.find(node) != visited.end());
  }

  void MDAGraphUpstreamIterator::addInputs(MDAGNode *node) {
    // canonly add outputs if we have a non null node
    if (node != NULL) {
      // iterate over the output nodes, and add them
      // if we haven't visited them yet
      MDAGNode::NodeList::iterator outIter;

      for (outIter = node->inputNodes.begin(); outIter != node->inputNodes.end(); ++outIter) {
        if (!haveVisited(*outIter)) {
          tovisit.push(*outIter);
        }
      }

    }
  }


}
