#include "StdAfx.h"

#include "MEditableMesh.h"

#include <assert.h>
#include <list>
#include <algorithm>

#include "MSystemManager.h"

#include <GL/glu.h>

#define MVERTEXLIST_GRANUL       128

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
#ifdef _DEBUG
  static void checkMesh(MEditableMesh *mesh);
#else
  #define checkMesh(mesh)
#endif

  //------------------
  //  MMeshEdgeData
  //------------------
  // Just a helper class to help with some various mesh editing stuff.
  class MGENEXPORT MMeshEdgeData : public MBaseObject {
  protected:
    std::vector< std::set<int> > m_List;
    int m_NumVerts;
   
  public:
    MMeshEdgeData();
    virtual ~MMeshEdgeData();

    void setVertexNum(int NumVerts);
    void addFace(int TriNum, MInternalTriangle *Tri);
    void addFace(int TriNum, int VertA, int VertB, int VertC);

    bool edgeOnFace(int VertA, int VertB, int TriA);
  };


  //----------------
  //  MEditableMesh
  //----------------
  
  MEditableMesh::MEditableMesh() {
    m_Snapshot = NULL;

  }

  MEditableMesh::MEditableMesh(MMeshPtr src) 
    : MMesh(src)
  {
    m_Snapshot = NULL;
  }
  
  MEditableMesh::~MEditableMesh() {
    m_Snapshot = NULL;

    setTextureMesh(NULL);
  }
  
  MBaseObjectPtr MEditableMesh::createNew() {
    MEditableMeshPtr NewMesh;
    
    NewMesh = new MEditableMesh;
    NewMesh->setFrom(this);
    
    return NewMesh;
  }
  
  void MEditableMesh::setFrom(MBaseObjectPtr SrcObj) {
    if (SrcObj == NULL) {
      return;
    }
    
    MMesh::setFrom(SrcObj);

    m_Snapshot = NULL;
  }
  
  void MEditableMesh::setFromMesh(MMeshPtr Mesh, int Append, bool KeepFlags) {
    if (Mesh == NULL) {
      return;
    }
    ensureUndo();
    
    VERTFLAGS   *VertFlags;
    TRIFLAGS    *TriFlags;
    
    VertFlags = NULL;
    TriFlags = NULL;
    
    if (Append == 0) {
      if (KeepFlags) {
        if (Mesh->getNumTris() == m_NumTris) {
          TriFlags = new TRIFLAGS[m_NumTris];
          
          for (int n = 0; n<m_NumTris; n++) {
            TriFlags[n] = m_Tris[n]->getFlags();
          }
        }
        if (Mesh->getNumVerts() == getNumVerts()) {
          VertFlags = new AztecFlags[getNumVerts()];
          
          for (int n = 0; n<getNumVerts(); n++) {
            VertFlags[n] = m_Verts[n]->getFlags();
          }
        }
      }
      
      // First Clear the mesh
      {
        setTextureMesh(NULL);
        
        
        // Check for the special case of when the number of vertices and triangles stays the same
        if (getNumVerts() == Mesh->getNumVerts() && m_NumTris == Mesh->getNumTris())
        {
          InternalVertexArray::iterator SrcVert, srcEnd = Mesh->m_Verts.end();
          InternalVertexArray::iterator DestVert, destEnd = m_Verts.end();
          MInternalTriangle   **SrcTri, **DestTri;

          for (SrcVert = Mesh->m_Verts.begin(), DestVert = m_Verts.begin(); 
               SrcVert != srcEnd; 
               ++SrcVert, ++DestVert) 
          {
            **DestVert = **SrcVert;
          }
          
          SrcTri = Mesh->m_Tris;
          DestTri = m_Tris;
          
          for (int n=m_NumTris-1; n>=0; n--)
          {
            **DestTri = **SrcTri;
            SrcTri ++;
            DestTri ++;
          }
          if (TriFlags)
          {
            for (int n = 0; n<m_NumTris; n++)
            {
              m_Tris[n]->unsetFlag(0xFFFFFFFF);
              m_Tris[n]->setFlag(TriFlags[n]);
            }
            delete[] TriFlags;
          }
          if (VertFlags)
          {
            for (int n = 0; n<getNumVerts(); n++)
            {
              m_Verts[n]->unsetFlag(0xFFFFFFFF);
              m_Verts[n]->setFlag(VertFlags[n]);
            }
            delete[] VertFlags;
          }
          
          if (Mesh->m_TextureMesh != NULL) {
            setTextureMesh(AZTEC_CAST(MMesh, Mesh->m_TextureMesh->createNew()));
          } else {
            setTextureMesh(NULL);
          }
          
          return;
        }
        
        for (int n = 0; n<getNumVerts(); n++) {
          if (m_Verts[n]) {
            delete m_Verts[n];
          }
        }
        m_Verts.clear();

        for (int n = 0; n<m_NumTris; n++) {
          if (m_Tris[n]) {
            delete m_Tris[n];
          }
        }
        delete[] m_Tris;
        
        m_Tris = NULL;
        m_NumTris = 0;
      }
   }
   
   if (Mesh == NULL)
     return;
   
   if (getNumVerts() + Mesh->getNumVerts() <= 0 || Mesh->getNumVerts() == 0)
     return;
   if (m_NumTris + Mesh->getNumTris() <= 0 || Mesh->getNumTris() == 0)
     return;
   
   MInternalTriangle **NewTris;
  
   int oldVertexCount = m_Verts.size();
   
   // Create the new arrays
   m_Verts.resize(m_Verts.size() + Mesh->getNumVerts());
   NewTris = new MInternalTriangle*[m_NumTris + Mesh->getNumTris()];
   
   
   // Fill up the new arrays with the old values
   for (int n=0; n<m_NumTris; n++) {
     NewTris[n] = m_Tris[n];
   }
   
   // Create the new vertices and triangles
   for (int n=0; n<Mesh->getNumVerts(); n++) {
     m_Verts[n + oldVertexCount] = new MInternalVertex(*Mesh->m_Verts[n]);
   }
   
   for (int n=0; n<Mesh->getNumTris(); n++) {
     MInternalTriangle   *Tri;
     Tri = new MInternalTriangle;
     *Tri = *Mesh->m_Tris[n];
     
     Tri->setVertex(0, Tri->getVertex(0) + oldVertexCount);
     Tri->setVertex(1, Tri->getVertex(1) + oldVertexCount);
     Tri->setVertex(2, Tri->getVertex(2) + oldVertexCount);
     
     NewTris[n + m_NumTris] = Tri;
   }
   
   // Create the new texture mesh, if applicable
   if (Mesh->m_TextureMesh != NULL) {
     setTextureMesh( AZTEC_CAST(MMesh, Mesh->m_TextureMesh->createNew() ) );
   } else {
     setTextureMesh( NULL);
   }
   
   if (m_Tris) {
     delete[] m_Tris;
   }
   
   m_Tris = NewTris;
   m_NumTris += Mesh->getNumTris();
   
   assignFlags(Mesh->getFlags());
   
   if (Append == 0)
   {
     int n;
     if (TriFlags)
     {
       for (n = 0; n<m_NumTris; n++)
       {
         m_Tris[n]->unsetFlag(0xFFFFFFFF);
         m_Tris[n]->setFlag(TriFlags[n]);
       }
       delete[] TriFlags;
     }
     if (VertFlags)
     {
       for (n = 0; n<getNumVerts(); n++)
       {
         m_Verts[n]->unsetFlag(0xFFFFFFFF);
         m_Verts[n]->setFlag(VertFlags[n]);
       }
       delete[] VertFlags;
     }
   }
   
   
  }

  void MEditableMesh::takeSnapshot() {
    if (m_Snapshot == NULL) {
      m_Snapshot = new MEditableMesh;
    }
  
    m_Snapshot->setFromMesh(this);
  }

  void MEditableMesh::retreiveSnapshot() {
    ensureUndo();
    if (m_Snapshot != NULL) {
      setFromMesh(m_Snapshot);
    }
  }


  int MEditableMesh::separateFaces(TRIFLAGS Flags) {
    ensureUndo();
    int         *VertsInfo;
    int         NumFlaggedTris, n, OldNumXYZ, NumSeamXYZ;
    MInternalTriangle   **ppTri;
  
    NumFlaggedTris = 0;
  
    ppTri = m_Tris;
    for (n=0; n < m_NumTris; n++, ppTri++) {
      if (!*ppTri) {
        continue;
      }
      if ((*ppTri)->isFlagged(Flags)) {
        NumFlaggedTris++;
      }
    }
  
    if (NumFlaggedTris == 0) {
      return 0;
    }
  
    VertsInfo = new int[getNumVerts()];
  
    memset(VertsInfo, 0, getNumVerts()*sizeof(int));
  
    // Got through and flag the vertices if they are on a non-detached face
    // or on a detached face.  Those which are on both need to be duplicated.
    ppTri = m_Tris;
    for (n=0; n < m_NumTris; n++, ppTri++) {
    
      if (!*ppTri) {
        continue;
      }
    
      if ((*ppTri)->isFlagged(Flags)) {
        VertsInfo[(*ppTri)->getVertex(0)] |= 0x01;
        VertsInfo[(*ppTri)->getVertex(1)] |= 0x01;
        VertsInfo[(*ppTri)->getVertex(2)] |= 0x01;
      } else {
        VertsInfo[(*ppTri)->getVertex(0)] |= 0x10;
        VertsInfo[(*ppTri)->getVertex(1)] |= 0x10;
        VertsInfo[(*ppTri)->getVertex(2)] |= 0x10;
      }
    }
  
    NumSeamXYZ = 0;
    for (n = 0; n<getNumVerts(); n++) {
      if (VertsInfo[n] == 0x11) {
        NumSeamXYZ ++;
      }
    }
    if (NumSeamXYZ == 0) {
      delete[] VertsInfo;
      return 0;
    }
  
    OldNumXYZ = getNumVerts();
  
    // Add the seam vertices to the current mesh
    {
      MVector3 *SeamVerts;
    
      SeamVerts = new MVector3[NumSeamXYZ];
    
      NumSeamXYZ = 0;
      for (n = 0; n<getNumVerts(); n++) {
        if (VertsInfo[n] == 0x11) {
          SeamVerts[NumSeamXYZ] = *m_Verts[n];
          NumSeamXYZ ++;
        }
      }
    
      addVertexArray(SeamVerts, NumSeamXYZ);
    
      delete[] SeamVerts;
    }
  
  
    // Go thorugh the VertsInfo array, and change it so that the array represents which of the enw vertices
    // should be used
    NumSeamXYZ = 0;
    for (n = 0; n<OldNumXYZ; n++) {
      if (VertsInfo[n] == 0x11) {
        VertsInfo[n] = NumSeamXYZ;
        NumSeamXYZ++;
      } else {
        VertsInfo[n] = -1;
      }
    }
  
  
  
    // Go through the flagged triangles, and adjust the seam vertices so they are referencing
    // the newly made vertices
    ppTri = m_Tris;
    for (n=0; n < m_NumTris; n++, ppTri++) {
    
      if (!*ppTri) {
        continue;
      }
    
      if ((*ppTri)->isFlagged(Flags)) {
        for (int nvert = 0; nvert < 3; nvert++) {
          if (VertsInfo[(*ppTri)->getVertex(nvert)] != -1) {
            (*ppTri)->setVertex(nvert, OldNumXYZ + VertsInfo[(*ppTri)->getVertex(nvert)]);
          }
        }
      }
    }
  
  
    delete[] VertsInfo;
  
    return NumSeamXYZ;
  }

  static inline int getValue(const std::vector<int> &vec, int index) {
    return vec[(index + vec.size()) % vec.size()];
  }

  int MEditableMesh::collapseVertices(AztecFlags Flags) {
    std::vector<int> verts;
    for (int n = 0; n < getNumVerts(); n++) {
      if (m_Verts[n]->isFlagged(Flags)) {
        verts.push_back(n);
      }
    }

    int newVertex = collapseVertices(verts);

    // now flag the new vertex
    setVertexFlag(newVertex, Flags);

    return newVertex;
  }

  int MEditableMesh::collapseVertices(const std::vector<int> &vertices) {
    // if we have no vertices to collapse, don't do anything.
    if (vertices.size() == 0) {
      return - 1;
    }

    ensureUndo();

    checkMesh(this);

    MVector3 AveragePos;
  
    std::vector<bool> vertsSelected;
    vertsSelected.assign(getNumVerts(), false);

    // first, make sure there are some vertices to deal with.
    for (int n=0; n<vertices.size(); n++) {
      AveragePos += *m_Verts[vertices[n]];      
      vertsSelected[vertices[n]] = true;
    }
  
    AveragePos *= 1.0/vertices.size();
  
    int newVertex = getNumVerts();
    addVertex(AveragePos);

    std::vector<bool> trisDone;
    trisDone.assign(getNumTris(), false);

    std::vector<int> trisToDelete;

    typedef std::vector<int> Polygon;
    typedef std::list<Polygon> PolygonList;

    PolygonList newPolygons;

    for (int tri = 0; tri < getNumTris(); ++tri) {
      // don't bother checking this triangle if it has already been done.
      if (trisDone[tri]) {
        continue;
      }
      
      // check to see if any of the vertices are flagged
      for (int corner = 0; corner < 3; ++corner) {
        if (vertsSelected[getTriangleVertex(tri, corner)]) {
          continue;
        }
        // we now have to get the points for this polygon, and recreate 
        // the polygons after the collapsing.

        Polygon poly;
        getPolygonVerticesFromTriangle(tri, poly);
        
        // see if there is more than one selected vertex.
        int selectedCount = 0;
        for (int i = 0; i < poly.size(); ++i) {
          if (vertsSelected[poly[i]]) {
            ++selectedCount;
          }
        }

        // we only bother with polygons that have 2 ore more vertices selected.
        if (selectedCount < 2) {
          continue;
        }

        // if the selected count is all the points, or all the points 
        // except 1, we just flag the triangles for deletion, and leave 
        // it at that.
        if (selectedCount + 1 < poly.size()) {
          // now we split up the polygons by the selected vertices. 
          int startVertex;

          // find a selected vertex as a starting point
          for (startVertex = 0; startVertex < poly.size(); ++startVertex) {
            if (vertsSelected[getValue(poly, startVertex)]) {
              break;
            }
          }

          // we should always find a vertex
          assert(startVertex < poly.size());

          int a = startVertex;

          do {
            while (vertsSelected[getValue(poly, a)]) {
              ++a;
            }
            if (a == startVertex) {
              break;
            }

            // a is now the first non selected vertex.
            int b = a;
          
            // now we find the last non selected vertex from this group.
            while (!vertsSelected[getValue(poly, b)]) {
              ++b;
            }

            --b;
            // b is now the last non-selected vertex.
          
            // we must construct a polygon that contains the new vertex, 
            // and these non selected vertices.
            //
            // If we only have 1 non selected vertex, then we do not make a 
            // polygon, because it would only consist of one point.

            if (a != b) {
              Polygon newPoly;
              newPoly.push_back(newVertex);
              while (a != b) {
                newPoly.push_back(getValue(poly,a));
                ++a;
              }
              newPoly.push_back(getValue(poly,b));

              newPolygons.push_back(newPoly);
            }
            // now move the first vertex onto the next one
            ++a;

            a %= poly.size();
            b %= poly.size();
          } while (a != startVertex);
        }

        std::set<int> trisInPoly;
        getPolygonFromTriangle(tri, trisInPoly);
        for (std::set<int>::iterator i = trisInPoly.begin(); i != trisInPoly.end(); ++i) {
          trisToDelete.push_back(*i);
          trisDone[*i] = true;
        }

        break;
      }
    }
  
    // delete all the old triangles
    
    std::sort(trisToDelete.begin(), trisToDelete.end());

    for (int i = (int)trisToDelete.size() - 1; i >= 0; --i) {
      deleteTriangle(trisToDelete[i]);
    }

    // reassign any left over points
    for (int tri = 0; tri < getNumTris(); ++tri) {
      // check to see if any of the vertices are flagged
      for (int corner = 0; corner < 3; ++corner) {
        if (vertsSelected[getTriangleVertex(tri, corner)]) {
          setTriangleVertex(tri, corner, newVertex);
        }
      }
    }

    // remake all the polygons
    for (PolygonList::iterator it = newPolygons.begin(); it != newPolygons.end(); ++it) {
      triangulatePolygon(*it);
    }

    // now delete any flagged vertices.
    deleteVertexArray(&vertices[0], vertices.size());

    calculateNormals();

    checkMesh(this);
  
    // the new vertex will be its old index, minus the vertices we deleted. 
    return newVertex - vertices.size();
  }

  class MVertexList {
  public:
    int         *m_Verts;
    int         m_NumVerts, m_NumAllocVerts;
  
    MVertexList();
    virtual ~MVertexList();
  
    void addVertex(int Vert);
  };

  MVertexList::MVertexList() {
    m_Verts = NULL;
    m_NumVerts = 0;
    m_NumAllocVerts = 0;
  }

  MVertexList::~MVertexList() {
    if (m_Verts) {
      delete[] m_Verts;
    }
  }

  void MVertexList::addVertex(int Vert) {
 
    if (m_NumVerts == m_NumAllocVerts) {
      int      *NewVerts;
    
      NewVerts = new int[m_NumAllocVerts + MVERTEXLIST_GRANUL];
      m_NumAllocVerts += MVERTEXLIST_GRANUL;
    
      for (int n=0; n<m_NumVerts; n++) {
        NewVerts[n] = m_Verts[n];
      }
    
      if (m_Verts) {
        delete[] m_Verts;
      }
    
      m_Verts = NewVerts;
    }
  
    m_Verts[m_NumVerts] = Vert;
    m_NumVerts ++;
  }

  int MEditableMesh::weldVertices(VERTFLAGS Flags, float Threshold) {
    ensureUndo();
    bool           All;
    float          Threshold2;
    MSystemManagerPtr SysMan;
  
    SysMan = Aztec::getSystemManager();
  
    Threshold2 = Threshold * Threshold;
  
    // If Flags is 0, then we are effecting all vertices.
    if (Flags == 0) {
      All = true;
    } else {
      All = false;
    }
  
  
    // Go through the mesh, and place all the vertices into various containers.
  
    int      NumBoxesPerSide, n, i, j, k;
    MVector3 BoxSize, MeshMin, MeshMax;
  
    MVertexList  ****BoxVerts;
  
    NumBoxesPerSide = 1 + getNumVerts() / 500;
  
    getMeshExtents(MeshMin, MeshMax);
  
    MeshMin += MVector3(-0.1f, -0.1f, -0.1f);
    MeshMax += MVector3(0.1f,0.1f,0.1f);
    BoxSize = MeshMax - MeshMin;
    BoxSize *= 1.0f / NumBoxesPerSide;
  
    // Create the array used to store the vertices
    BoxVerts = new MVertexList***[NumBoxesPerSide];
    for (i=0; i<NumBoxesPerSide; i++)
    {
      BoxVerts[i] = new MVertexList**[NumBoxesPerSide];
      for (j=0; j<NumBoxesPerSide; j++)
      {
        BoxVerts[i][j] = new MVertexList*[NumBoxesPerSide];
        for (k=0; k<NumBoxesPerSide; k++)
        {
          BoxVerts[i][j][k] = new MVertexList;
        }
      }
    }
  
  
    // Go through the mesh, and place the vertices in the appropriate container.
    for (n=0; n<getNumVerts(); n++) {
      MInternalVertex *V;
      int         ip, jp, kp, in, jn, kn;
    
      V = m_Verts[n];
    
      if ( !(All || V->isFlagged(Flags)) ) {
        continue;
      }
    
      ip = (int)((V->m_Pos.x - MeshMin.x) / BoxSize.x);
      jp = (int)((V->m_Pos.y - MeshMin.y) / BoxSize.y);
      kp = (int)((V->m_Pos.z - MeshMin.z) / BoxSize.z);
    
      BoxVerts[ip][jp][kp]->addVertex(n);
      for (i=-1; i<=1; i++) {
        for (j=-1; j<=1; j++) {
          for (k=-1; k<=1; k++) {
            in = (int)((V->m_Pos.x - MeshMin.x + (Threshold * i)) / BoxSize.x);
            jn = (int)((V->m_Pos.y - MeshMin.y + (Threshold * i)) / BoxSize.y);
            kn = (int)((V->m_Pos.z - MeshMin.z + (Threshold * i)) / BoxSize.z);
          
            if (in != ip || jn != jp || kn != kp) {
              BoxVerts[in][jn][kn]->addVertex(n);
            }
          }
        }
      }
    
    }
  
  
    // Now that we have boxified the vertex data, we can go through and weld the similar vertices.
    unsetVertexFlags(~VERTEX_VISIBLE);
  
    int      *TargetVerts;
  
    TargetVerts = new int[getNumVerts()];
    for (i=0; i<getNumVerts(); i++) {
      TargetVerts[i] = -1;
    }
  
    for (i=0; i<NumBoxesPerSide; i++) {
      for (j=0; j<NumBoxesPerSide; j++) {
        for (k=0; k<NumBoxesPerSide; k++) {
          MVertexList    *L;
          int            n1,n2;
          int            v1,v2;
        
          L = BoxVerts[i][j][k];
        
          for (n1 = 0; n1 < L->m_NumVerts ; n1++) {
            v1 = L->m_Verts[n1];
            for (n2 = n1+1; n2 < L->m_NumVerts ; n2++)
            {
              v2 = L->m_Verts[n2];
            
              if (v1 == v2)
                continue;
            
              float    Len;
              MVector3 DiffVec;
            
              DiffVec = m_Verts[v2]->m_Pos - m_Verts[v1]->m_Pos;
              Len = DiffVec.length2();
              if (Len <= Threshold2) {
                if (TargetVerts[v2] == -1) {
                  int      Targ;
                
                  Targ = v1;
                
                  while (TargetVerts[Targ] != -1) {
                    Targ = TargetVerts[Targ];
                  }
                  TargetVerts[v2] = Targ;
                  m_Verts[v1]->setFlag(VERTEX_SELECTED);
                  m_Verts[v1]->setFlag(VERTEX_SELECTED);
                }
              }
            }
          }
        
        
        }
      }
    }
  
    // go through and move all the triangles' vertices to the correct place.
    unsetVertexFlags(~VERTEX_VISIBLE);
    for (i=0; i<getNumVerts(); i++) {
      if (TargetVerts[i] != -1) {
        m_Verts[i]->setFlag(VERTEX_FLAGFORCHANGE);
      }
    }
  
    for (i = m_NumTris - 1; i >= 0; i--) {
      for (k=0; k<3; k++) {
        if ( TargetVerts[m_Tris[i]->getVertex(k)] != -1) {
          m_Verts[m_Tris[i]->getVertex(k)]->setFlag(VERTEX_FLAGFORCHANGE);
          m_Tris[i]->setVertex(k, TargetVerts[m_Tris[i]->getVertex(k)]);
        }
      }
      if (m_Tris[i]->getVertex(0) == m_Tris[i]->getVertex(1) ||
        m_Tris[i]->getVertex(0) == m_Tris[i]->getVertex(2) ||
        m_Tris[i]->getVertex(1) == m_Tris[i]->getVertex(2)) {
        deleteTriangle(i);
      }
    }
    for (i=0; i<m_NumTris; i++) {
      for (k=0; k<3; k++) {
        m_Verts[m_Tris[i]->getVertex(k)]->unsetFlag(VERTEX_FLAGFORCHANGE);
      }
    }
  
    deleteVertexFlag(VERTEX_FLAGFORCHANGE);
  
    // Clean up the 3d box array mess.
    for (i=0; i<NumBoxesPerSide; i++) {
      for (j=0; j<NumBoxesPerSide; j++) {
        for (k=0; k<NumBoxesPerSide; k++) {
          if (BoxVerts[i][j][k])
            delete BoxVerts[i][j][k];
        }
        delete[] BoxVerts[i][j];
      }
      delete[] BoxVerts[i];
    }
    delete[] BoxVerts;

    delete[] TargetVerts;

    return 1;
  }

  bool MEditableMesh::divideEdgeWithVertex(int vertFrom, int vertTo, int newVertex) {
    ensureUndo();
    // find the triangle and edge we are on.
    std::vector<int> tris;
    getTrianglesWithEdge(vertFrom, vertTo, tris);

    if (tris.size() > 0) {
      for (unsigned i = 0; i < tris.size() ; ++i) {

        int edgeIndex = getTriangleEdge(tris[i], vertFrom, vertTo);

        // if our edge goes from vertFrom->vertTo, then we change it from vertFrom->newVertex
        // and the new triangle becomes newVertes->vertTo
        if (getTriVert(tris[i], edgeIndex) != vertFrom) {
          std::swap(vertFrom, vertTo);
        }

        int newTri = getNumTris();
        addTriangleArray(getTriangle(tris[i]));

        setTriangleVertex(tris[i], (edgeIndex+1)%3, newVertex);
        setTriangleVertex(newTri, edgeIndex, newVertex);

        unsetTriangleEdgeFlag(tris[i], (edgeIndex+1)%3, EDGE_VISIBLE | EDGE_SELECTED);
        unsetTriangleEdgeFlag(newTri, (edgeIndex+2)%3, EDGE_VISIBLE | EDGE_SELECTED);

        // check to see if an edge turn will help things out.
        if (!isTriangleEdgeFlagged(tris[i], (edgeIndex+2)%3, EDGE_VISIBLE)) {
          turnEdge(tris[i] * 3 + (edgeIndex+2)%3); 
        } else if (!isTriangleEdgeFlagged(newTri, (edgeIndex+1)%3, EDGE_VISIBLE)) {
          turnEdge(newTri * 3 + (edgeIndex+1)%3); 
        }

      }
    }

    return tris.size() > 0;
  }

  int MEditableMesh::divideEdge(int vertFrom, int vertTo, const MVector3 &pos) {
    ensureUndo();
    // find the triangle and edge we are on.
    std::vector<int> tris;
    getTrianglesWithEdge(vertFrom, vertTo, tris);

    if (tris.size() > 0) {
      int newVertex = addVertex(pos.x, pos.y, pos.z);

      divideEdgeWithVertex(vertFrom, vertTo, newVertex);
      return newVertex;
    } else {
      return -1;
    }

  }



  int MEditableMesh::addVertex(float x, float y, float z) {
    ensureUndo();
    return addVertexArray(&MVector3(x,y,z), 1);
  }

  int MEditableMesh::addVertex(const MVector3 &pos) {
    ensureUndo();
    return addVertexArray(&pos, 1);
  }

  int MEditableMesh::addVertexArray(const MVector3 Vert[], int Num) {
    ensureUndo();
    int i;

    int offset = m_Verts.size();
    m_Verts.resize(m_Verts.size() + Num);
  
    for (i = 0; i < Num; i++) {
      m_Verts[offset+i] = new MInternalVertex();
      *m_Verts[offset+i] = Vert[i];
    }
  
    invalidateNormals();

    return offset;
  }

  void MEditableMesh::deleteVertex(int Num)
  {
    ensureUndo();
    deleteVertexArray(&Num,1);
  }

  void MEditableMesh::deleteVertexArray(const int *VertNums, int Num ) {
    ensureUndo();
    if (Num > getNumVerts()) {
      return;
    }
  
    int      n, NumNewVerts, *NewVertID, *TriDelete, TriDelCount = 0;
  
    NewVertID = new int[getNumVerts()];
    TriDelete = new int[m_NumTris];
  
    NumNewVerts = 0;
    for (n=0;n<Num;n++)
    {
      if (m_Verts[VertNums[n]])
      {
        delete m_Verts[VertNums[n]];
        m_Verts[VertNums[n]] = NULL;
        NewVertID[VertNums[n]] = -1;
      }
    }
  
    for (n=0;n<getNumVerts();n++)
    {
      if (m_Verts[n] != NULL)
      {
        m_Verts[NumNewVerts] = m_Verts[n];
        NewVertID[n] = NumNewVerts;
        NumNewVerts++;
      }
    }
  
    // Go through and adjust the vertices of the triangles, and delete any triangles in use
    for (n=0; n<m_NumTris; n++)
    {
      MInternalTriangle      *Tri;
      Tri = m_Tris[n];
    
      Tri->setVertex(0, NewVertID[Tri->getVertex(0)]);
      Tri->setVertex(1, NewVertID[Tri->getVertex(1)]);
      Tri->setVertex(2, NewVertID[Tri->getVertex(2)]);
    
      if (Tri->getVertex(0) == -1 || Tri->getVertex(1) == -1 || Tri->getVertex(2) == -1)
      {
        TriDelete[TriDelCount] = n;
        TriDelCount++;
      }
    }
  
    deleteTriangleArray(TriDelete, TriDelCount);
  
    delete[] TriDelete;
    delete[] NewVertID;
  
    m_Verts.resize(NumNewVerts);

    invalidateNormals();
  }


  void MEditableMesh::addTriangle(int A, int B, int C)
  {
    ensureUndo();
    MInternalTriangle  Tri;
  
    Tri.setVertex(0, A);
    Tri.setVertex(1, B);
    Tri.setVertex(2, C);
  
    addTriangleArray(&Tri,1);
  }

  void MEditableMesh::addTriangle(const MVector3 &A, const MVector3 &B, const MVector3 &C)
  {
    ensureUndo();
    MVector3 Verts[3];
    MInternalTriangle  Tri;
  
    Verts[0] = A;
    Verts[1] = B;
    Verts[2] = C;
  
    addVertexArray(Verts,3);
  
    Tri.setVertex(0, getNumVerts()-3);
    Tri.setVertex(1, getNumVerts()-2);
    Tri.setVertex(2, getNumVerts()-1);
  
    addTriangleArray(&Tri,1);
  }

  void MEditableMesh::addTriangleArray(const MInternalTriangle *Tri, int Num)
  {
    ensureUndo();
    MInternalTriangle     **NewTris;
    int            i;
  
    NewTris = new MInternalTriangle*[m_NumTris+Num];
  
    for (i=0;i<m_NumTris;i++) {
      NewTris[i] = m_Tris[i];
    }
  
    for (i=0;i<Num;i++) {
      NewTris[m_NumTris+i] = new MInternalTriangle;
      *NewTris[m_NumTris+i] = Tri[i];

      if (Tri[i].getVertex(0) > -1) {
        m_Verts[Tri[i].getVertex(0)]->invalidateNormal();
      }
      if (Tri[i].getVertex(1) > -1) {
        m_Verts[Tri[i].getVertex(1)]->invalidateNormal();
      }
      if (Tri[i].getVertex(2) > -1) {
        m_Verts[Tri[i].getVertex(2)]->invalidateNormal();
      }
    }
  
    if (m_Tris) {
      delete[] m_Tris;
    }
    m_Tris = NewTris;
  
    m_NumTris += Num;

    invalidateNormals();
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MEditableMesh::deleteVertexFlag(AztecFlags Flag)
  {
    ensureUndo();
    int   *VertsToDel, n, NumToDel = 0;
  
    VertsToDel = new int[getNumVerts()];
  
    for (n = 0; n<getNumVerts(); n++)
    {
      if (m_Verts[n]->isFlagged(Flag))
      {
        VertsToDel[NumToDel] = n;
        NumToDel++;
      }
    }
    deleteVertexArray(VertsToDel, NumToDel);
    delete[] VertsToDel;
  }


  void MEditableMesh::deleteTriangleFlag(AztecFlags Flag) {
    ensureUndo();
    int   *TrisToDel, n, NumToDel = 0;
  
    TrisToDel = new int[getNumVerts()];
  
    for (n = 0; n<m_NumTris; n++) {
      if (m_Tris[n]->isFlagged(Flag)) {
        TrisToDel[NumToDel] = n;
        NumToDel++;
      }
    }
    deleteTriangleArray(TrisToDel, NumToDel);
    delete[] TrisToDel;
  }

  void MEditableMesh::addVertsAndTriangles(MVector3 *Verts, MInternalTriangle *Tris, int NumVerts, int NumTris) {
    ensureUndo();
    int         n, Adjust;
    MInternalTriangle   *Tri;

    Adjust = getNumVerts();
  
    Tri = Tris;
    for (n=0; n<NumTris; n++) {
      Tri->setVertex(0, Tri->getVertex(0) + Adjust);
      Tri->setVertex(1, Tri->getVertex(1) + Adjust);
      Tri->setVertex(2, Tri->getVertex(2) + Adjust);
      Tri++;
    }
  
    addVertexArray(Verts, NumVerts);
    addTriangleArray(Tris, NumTris);
  
    Tri = Tris;
    for (n=0; n<NumTris; n++) {
      Tri->setVertex(0, Tri->getVertex(0) - Adjust);
      Tri->setVertex(1, Tri->getVertex(1) - Adjust);
      Tri->setVertex(2, Tri->getVertex(2) - Adjust);
      Tri++;
    }

  }

  void MEditableMesh::setVertexPosition(int index, const MVector3 &pos) {
    ensureUndo();
    // TODO: perform some notification here
    *m_Verts[index] = pos;
    m_Verts[index]->invalidateNormal();
    invalidateNormals();
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MEditableMesh::setVertexNormal(int index, const MVector3 &normal) {
    ensureUndo();
    // TODO: perform some notification here
    m_Verts[index]->m_Normal = normal;
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MEditableMesh::setTriangleVertex(int triIndex, int vertexIndex, int newMeshVertex) {
    ensureUndo();
    // TODO: perform some notification here
    m_Tris[triIndex]->setVertex(vertexIndex, newMeshVertex);
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }


  void MEditableMesh::deleteTriangle(int Num) {
    deleteTriangleArray(&Num, 1);
  }

  void MEditableMesh::deleteTriangleArray(const int *TriNums, int Num) {
    ensureUndo();
    if (Num > m_NumTris) {
      return;
    }
  
    MInternalTriangle  **NewTris;
    int      n, NumNewTris;
  
    NewTris = new MInternalTriangle*[m_NumTris-Num];
  
    NumNewTris = 0;
    for (n=0;n<Num;n++) {
      if (m_Tris[TriNums[n]]) {
        delete m_Tris[TriNums[n]];
        m_Tris[TriNums[n]] = NULL;
      }
    }
  
    for (n=0;n<m_NumTris;n++) {
      if (m_Tris[n]) {
        NewTris[NumNewTris] = m_Tris[n];
        NumNewTris++;
      }
    }
  
    if (m_Tris) {
      delete[] m_Tris;
    }
    m_Tris = NewTris;
  
    m_NumTris = NumNewTris;

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MEditableMesh::centerMesh() {
    ensureUndo();
    MVector3   Centre;
    MInternalVertex *V;
    int         cnt;
  
    for (cnt=0;cnt<getNumVerts();cnt++) {
      V = m_Verts[cnt];
      Centre.x += V->m_Pos.x;
      Centre.y += V->m_Pos.y;
      Centre.z += V->m_Pos.z;
    }
    Centre.x /= cnt;
    Centre.y /= cnt;
    Centre.z /= cnt;
  
    for (cnt=0;cnt<getNumVerts();cnt++) {
      V = m_Verts[cnt];
      V->m_Pos.x -= Centre.x;
      V->m_Pos.y -= Centre.y;
      V->m_Pos.z -= Centre.z;
    }

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MEditableMesh::scaleMesh(float Factor) {
    ensureUndo();
    MInternalVertex *V;
    int         cnt;
    for (cnt=0;cnt<getNumVerts();cnt++)
    {
      V = m_Verts[cnt];
      V->m_Pos.x *= Factor;
      V->m_Pos.y *= Factor;
      V->m_Pos.z *= Factor;
    }
  }

  int MEditableMesh::flipNormals(TRIFLAGS Flags) {
    ensureUndo();
    MInternalTriangle      **ppTri;
    int            Num;
  
    Num = 0;
    ppTri = m_Tris;
  
    for (int n = 0; n < m_NumTris; n++) {
      MInternalTriangle      *Tri;
    
      Tri = *ppTri;
      ppTri++;
    
      if (!(Tri->isFlagged(Flags) || (Tri->getFlags() == 0 && Flags == 255))) {
        continue;
      }
    
      int Temp;
      Temp = Tri->getVertex(2);
      Tri->setVertex(2, Tri->getVertex(0));
      Tri->setVertex(0, Temp);
    
      Tri->m_Normal *= -1;
    }
  
    calculateNormals();
    return Num;
  }

  class Tess {
  public:
    Tess() 
    {
      currentVisible = true;
    }

    std::vector<int> verts;
    std::vector<bool> visible;
    bool currentVisible;

  };

  void __stdcall vertex (void * vertex_data, void *poly_data) {
    Tess *t = reinterpret_cast<Tess*>(poly_data);

    t->verts.push_back((int)vertex_data);
    t->visible.push_back(t->currentVisible);
  }

  void __stdcall edge (GLboolean flag, void *poly_data) {
    Tess *t = reinterpret_cast<Tess*>(poly_data);

    t->currentVisible = (flag == GL_TRUE);
  }


#ifdef _WIN32
  typedef void __stdcall tess_callback(void);
#else
  typedef void tess_callback(void);
#endif

  typedef std::pair<int,int> Edge;
  typedef std::map<Edge, int> EdgeMap;
  Edge makeEdge(int a, int b) {
    if (b < a) std::swap(a,b);
    return Edge(a,b);
  }

#ifdef _DEBUG
  static void checkMesh(MEditableMesh *mesh) {
    EdgeMap edges;

    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int e = 0; e < 3; ++e) {
        int a, b;
        mesh->getVertsFromEdge(i, e, &a, &b);
        EdgeMap::iterator it = edges.insert(std::make_pair(makeEdge(a, b), 0)).first;
        ++it->second;

        // ensure we have no more than two triangles per edge;
        assert(it->second <= 2);
      }
    }

  }
#else
#define checkMesh(mesh)
#endif

  void MEditableMesh::triangulatePolygon(const std::vector<int> &vertices) {
    ensureUndo();
    if (vertices.size() < 3) {
      return;
    }

    Tess data;

    GLUtesselator *tess = gluNewTess();
    assert(tess != 0);

    gluTessCallback(tess, GLU_TESS_VERTEX_DATA, (tess_callback*)vertex);
    gluTessBeginPolygon(tess, &data);

    gluTessCallback(tess, GLU_TESS_EDGE_FLAG_DATA, (tess_callback*)edge);


    for (unsigned i = 0; i < vertices.size(); ++i) {
      MVector3 pos = getVertexPosition(vertices[i]);
      GLdouble coords[3];
      coords[0] = pos.x;
      coords[1] = pos.y;
      coords[2] = pos.z;

      gluTessVertex(tess, coords, (void*)vertices[i]);
    }

    gluTessEndPolygon(tess);

    gluDeleteTess(tess);

    checkMesh(this);


    // make sure we got enough data to fill the input polygon
    // if we haven't, just create a fan
    if (data.verts.size() == 3 * (vertices.size() - 2)) {
      int tricount = 0;

      // now that we have all our triangles just add them in.
      for (unsigned i = 0; i < data.verts.size();) {
        int tri = getNumTris();
        bool edgeFlag[3] = { true, true, true };
        addTriangle(data.verts[i], data.verts[i+1], data.verts[i+2]);
        tricount++;

        checkMesh(this);

        for (int j = 0; j < 3; ++j) {
          if (!data.visible[i + j]) {
            edgeFlag[j] = false;
          }
        }

        for (int j = 0; j < 3; ++j) {
          if (!edgeFlag[j]) {
            int a, b;
            getVertsFromEdge(tri, j, &a, &b);
            m_Tris[tri]->edgeFlags[j].unsetFlag(EDGE_VISIBLE);
          }
        }

        i += 3;

      }
    } else {
      for (unsigned i = 2; i < vertices.size(); ++i) {
        addTriangle(vertices[0], vertices[i-1], vertices[i]);
      }
    }

  }

  void MEditableMesh::storeComponentPositions(AztecFlags type, AztecFlags flag) {
    storedVertices.resize(m_Verts.size());
    for (unsigned int i = 0; i < m_Verts.size(); ++i) {
      storedVertices[i] = *m_Verts[i];
    }
  }

  void MEditableMesh::restoreComponentPositions(AztecFlags type, AztecFlags flag) {
    ensureUndo();
    // we can only do this if we have the same number of points
    if (storedVertices.size() != m_Verts.size()) {
      return;
    }

    switch (type) {
    case POINT_TYPE:
      {
        for (unsigned int i = 0; i < m_Verts.size(); ++i) {
          if (flag == 0 || m_Verts[i]->isFlagged(flag)) {
            *m_Verts[i] = storedVertices[i];
          }
        }
        break;
      }
    case FACET_TYPE:
      {
        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            if (flag == 0 || m_Tris[i]->isFlagged(flag)) {
              *m_Verts[m_Tris[i]->getVertex(0)] = storedVertices[m_Tris[i]->getVertex(0)];
              *m_Verts[m_Tris[i]->getVertex(1)] = storedVertices[m_Tris[i]->getVertex(1)];
              *m_Verts[m_Tris[i]->getVertex(2)] = storedVertices[m_Tris[i]->getVertex(2)];
            }
          }
        }
        break;
      }
    case EDGE_TYPE:
      {
        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            for (int edge = 0; edge < 3; ++edge) {
              if (flag == 0 || m_Tris[i]->isEdgeFlagged(edge, flag)) {
                *m_Verts[m_Tris[i]->getVertex(edge)] = storedVertices[m_Tris[i]->getVertex(edge)];
                *m_Verts[m_Tris[i]->getVertex((edge+1)%3)] = storedVertices[m_Tris[i]->getVertex((edge+1)%3)];
              }
            }
          }
        }
        break;
      }
    }
  }

  void MEditableMesh::transformComponents(AztecFlags type, 
                                          const MMatrix4 &transform, 
                                          AztecFlags flag, 
                                          bool localTransform,
                                          TransformModifier *modifier) 
  {
    ensureUndo();
    MMatrix4 localMat;
    bool *doVertex = NULL;

    switch (type) {
    case POINT_TYPE:
      break;
    case FACET_TYPE:
      {
        doVertex = new bool[m_Verts.size()];
        memset(doVertex, 0, sizeof(bool) * m_Verts.size());

        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            if (flag == 0 || m_Tris[i]->isFlagged(flag)) {
              doVertex[m_Tris[i]->getVertex(0)] = true;
              doVertex[m_Tris[i]->getVertex(1)] = true;
              doVertex[m_Tris[i]->getVertex(2)] = true;
            }
          }
        }

        break;
      }
    case EDGE_TYPE:
      {
        doVertex = new bool[m_Verts.size()];
        memset(doVertex, 0, sizeof(bool) * m_Verts.size());

        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            for (int edge = 0; edge < 3; ++edge) {
              if (flag == 0 || m_Tris[i]->isEdgeFlagged(edge, flag)) {
                doVertex[m_Tris[i]->getVertex(edge)] = true;
                doVertex[m_Tris[i]->getVertex((edge+1)%3)] = true;
              }
            }
          }
        }

        break;
      }
    }

    if (type == POINT_TYPE || doVertex != NULL) {
      for (unsigned int i = 0; i < m_Verts.size(); ++i) {
        if ((doVertex != NULL && doVertex[i]) || 
            (doVertex == NULL && (flag == 0 || m_Verts[i]->isFlagged(flag))) ) {
          if (localTransform) {
            localMat.createRotationMatrix(MVector3(0,0,1), getVertexNormal(i));
            
            localMat = transform * localMat;
            MVector3 newPos = m_Verts[i]->m_Pos + localMat * MVector3(0,0,0);
            if (modifier != NULL) {
              modifier->adjustTransform(m_Verts[i]->m_Pos, newPos);
            }
            m_Verts[i]->m_Pos = newPos;
            
          } else {
            MVector3 newPos = transform * (m_Verts[i]->m_Pos);
            if (modifier != NULL) {
              modifier->adjustTransform(m_Verts[i]->m_Pos, newPos);
            }
            m_Verts[i]->m_Pos = newPos;
          }

          m_Verts[i]->invalidateNormal();
        }
      }
      invalidateNormals();
    }
    if (doVertex != NULL) {
      delete[] doVertex;
    }
  }

  void MEditableMesh::updateComponentAnimation(AztecFlags type, AztecFlags flag, long time, bool createKey) {
  }

  void MEditableMesh::updateSingleComponentAnimation(AztecFlags type, int index, long time, bool createKey) {
  }

  int MEditableMesh::extrudeFaces(AztecFlags Flags, float Amount, const MVector3 &NormVec) {
    ensureUndo();
    MSystemManagerPtr SysMan;
  
    SysMan = Aztec::getSystemManager();
  
    {
      MMeshEdgeData  Edges;
    
      Edges.setVertexNum(getNumVerts());
      for (int n=0; n<m_NumTris; n++) {
        Edges.addFace(n, m_Tris[n]);
      }
    }
  
  
    // What we have to do:
    //
    // Find all the vertices that are on the edge of the selection. These are the ones that 
    // Create new faces.
    //
  
    int                  n, NumNewVerts;
    std::vector<int> newVertsArray; // This is an array of what the corresponding new vertices are.
    int                  *EdgeArray;       // Counter for edge flags as to how many times it is in use.
    MVector3             OffsetVec;
  
    MSortedNumberList    *EdgeList;
    int                  EdgeCount;
  
    OffsetVec = Amount * NormVec;
  
    EdgeList = new MSortedNumberList[getNumVerts()];
    newVertsArray.resize(getNumVerts(), -1);
  
    EdgeArray = new int[m_NumTris*3];
    memset(EdgeArray, 0, m_NumTris*3*sizeof(int));
  
    // Mark all the vertices of the extrusion
    {
      char  *VertArray;
    
      VertArray = new char[getNumVerts()];
    
      memset(VertArray, 0, getNumVerts());
    
      {
        unsigned int n;
		int k;
      
        EdgeCount = 0;
      
        // go through the selected faces, and add the edges in.
      
        for (n=0; n<(unsigned int)m_NumTris; n++)
        {
          if (m_Tris[n]->isFlagged(Flags))
          {
            // If it is selected, try and add the edges
            for(k=0; k<3; k++)
            {
              MInternalTriangle   *Tri;
              int         V1, V2;
            
              Tri = m_Tris[n];
              V1 = Tri->getVertex(k);
              V2 = Tri->getVertex((k+1)%3);
            
              // See if the edge already exists in the list
              if (EdgeList[V1].find(V2) != -1)
              {
                EdgeList[V1].remove(V2);
                EdgeCount--;
                continue;
              }
              if (EdgeList[V2].find(V1) != -1)
              {
                EdgeList[V2].remove(V1);
                EdgeCount--;
                continue;
              }
            
              // Add it to the list
              EdgeList[V2].add(V1);
              EdgeCount++;
            }
          }
        }
      
        // Iterate through the edge list and mark all the vertices with -1.
        for (n=0; n<(unsigned int)getNumVerts(); n++)
        {
          VertArray[n] = 1;
        }
        for (n=0; n<(unsigned int)getNumVerts(); n++)
        {
          if (EdgeList[n].getNumItems() > 0)
          {
            // iterate through the items
            VertArray[n] = -1;
            for (k=0; k<EdgeList[n].getNumItems(); k++)
            {
              VertArray[EdgeList[n].getItem(k)] = -1;
            }
          }
        }
      
      }
    
      // Now all the vertices with values of -1 are on the edge of the extrusion.
      // Makr them all accordingly
      NumNewVerts = 0;
      for (n = 0; n<getNumVerts(); n++)
      {
        if (VertArray[n] < 0)
        {
          m_Verts[n]->m_Pos = m_Verts[n]->m_Pos + OffsetVec;
        }
      
        if (VertArray[n] == -1)
        {
          newVertsArray[n] = NumNewVerts + getNumVerts();
          NumNewVerts++;
        }
        else
        {
          newVertsArray[n] = -1;
        }
        m_Verts[n]->unsetFlag(VERTEX_FLAGFORCHANGE);
        m_Verts[n]->unsetFlag(VERTEX_SELECTED);
      }
    
      delete[] VertArray;
    }
  
    // Now we know which verts are changed and which stay the same.
  
    // Create the new vertex array.
    MVector3 *NewVerts;
    MInternalTriangle   *NewTris, *CurTri;
  
    NewVerts = new MVector3[NumNewVerts];
  
    for (n = 0; n<getNumVerts(); n++)
    {
      if (newVertsArray[n] != -1)
      {
        NewVerts[newVertsArray[n] - getNumVerts()] = m_Verts[n]->m_Pos + OffsetVec;
      }
    }
  
    // Find out how many edges are in the extrusion, acreate the extra faces, and change the old verts
  
    {
	     int      k;
     
       NewTris = new MInternalTriangle[EdgeCount * 2];    // We can at most have double the tris as verts
       // We now have a list of edges that will make up the extruded edged faces.
       CurTri = NewTris;
       for (n = 0; n<getNumVerts(); n++)
       {
         int      V1, V2;
       
         if (EdgeList[n].getNumItems() == 0)
           continue;
       
         for (k = EdgeList[n].getNumItems() - 1; k>=0; k--)
         {
           V1 = n;
           V2 = EdgeList[n].getItem(k);
         
           CurTri->setVertex(0, V1);
           CurTri->setVertex(1, newVertsArray[V1]);
           CurTri->setVertex(2, newVertsArray[V2]);
           CurTri->unsetEdgeFlag(2 ,EDGE_VISIBLE);

           CurTri++;
           CurTri->setVertex(0, V1);
           CurTri->setVertex(1, newVertsArray[V2]);
           CurTri->setVertex(2, V2);
           CurTri->unsetEdgeFlag(0, EDGE_VISIBLE);
           CurTri++;
         }
       }
     
    }
  
    // Update the falgged tirangles to use the new vertices
    for (n=0; n<m_NumTris; n++)
    {
      if (!m_Tris[n]->isFlagged(Flags))
        continue;
    
      for (int k=0; k<3; k++)
      {
        if (m_Tris[n]->getVertex(k) >= getNumVerts())
          continue;
      
        if (newVertsArray[ m_Tris[n]->getVertex(k) ] == -1)
          continue;
      
        m_Tris[n]->setVertex(k, newVertsArray[ m_Tris[n]->getVertex(k) ] );
      }
    }

    // Add the new arrays
    addVertexArray(NewVerts, NumNewVerts);
    addTriangleArray(NewTris, EdgeCount * 2);
  
    for (unsigned int index = 0; index<newVertsArray.size(); index++) {
      if (newVertsArray[index] != -1) {
        setVertexFlag(newVertsArray[index], VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);
      }
    }

    // go through, and zero all the triangle vertex numbers, and add them to the texture mesh.
    {
      MInternalTriangle      *Tri;
      MEditableMeshPtr textureMesh;
    
      Tri = NewTris;
      for (n=0; n<EdgeCount; n++)
      {
        Tri->setVertex(0, 0);
        Tri->setVertex(1, 0);
        Tri->setVertex(2, 0);
        Tri++;
        Tri->setVertex(0, 0);
        Tri->setVertex(1, 0);
        Tri->setVertex(2, 0);
        Tri++;
      }
    
      textureMesh = AZTEC_CAST(MEditableMesh, getTextureMesh());
      if (textureMesh != NULL) {
        textureMesh->addTriangleArray(NewTris, EdgeCount*2);
      }
    }
  
    delete[] EdgeList;
    delete[] EdgeArray;
    delete[] NewVerts;
    delete[] NewTris;

    // Set the flag for Change flag on all the vertices
    // that sit on flagged triangles
    for (n=0; n<m_NumTris; n++) {
      if (!m_Tris[n]->isFlagged(Flags)) {
        continue;
      }

      m_Verts[m_Tris[n]->getVertex(0)]->setFlag(VERTEX_FLAGFORCHANGE);
      m_Verts[m_Tris[n]->getVertex(1)]->setFlag(VERTEX_FLAGFORCHANGE);
      m_Verts[m_Tris[n]->getVertex(2)]->setFlag(VERTEX_FLAGFORCHANGE);
    }

    calculateNormals();
  
    return 1;
  
  }

  int MEditableMesh::divideEdge(int EdgeIndex, float dist) {
    ensureUndo();

    // do this divide edge on the texture mesh as well.
    MEditableMeshPtr textureMesh = AZTEC_CAST(MEditableMesh, getTextureMesh());
    if (textureMesh != NULL) {
      textureMesh->divideEdge(EdgeIndex, dist);
    }

    int a, b;
    getVertsFromEdge(EdgeIndex, &a, &b);
    if (dist < 0.001) {
      return a;
    } else if (dist >= 0.999) {
      return b;
    } else {
      return divideEdge(a, b, dist * getVertexPosition(b) + (1.0 - dist) * getVertexPosition(a));
    }
  }

  int MEditableMesh::turnEdge (int EdgeNum) {
    ensureUndo();
    std::vector<int> tris;
    int a, b;

    getVertsFromEdge(EdgeNum, &a, &b);

    getTrianglesWithEdge(a, b, tris);

    // if we do not have two triangles, we acn't really do an edge flip now can we?
    if (tris.size() != 2) {
      return 0;
    }

    MInternalTriangle old1(*m_Tris[tris[0]]);
    MInternalTriangle old2(*m_Tris[tris[1]]);

    int edgeA, edgeB;

    edgeA = getTriangleEdge(tris[0], a, b);
    edgeB = getTriangleEdge(tris[1], a, b);

    int otherPointA = getTriangleVertex(tris[0], (edgeA + 2) % 3);
    int otherPointB = getTriangleVertex(tris[1], (edgeB + 2) % 3);

    setTriangleVertex(tris[0], edgeA, otherPointB);
    setTriangleVertex(tris[1], edgeB, otherPointA);

    assignTriangleEdgeFlag(tris[0], edgeA, old2.getEdgeFlag((edgeB + 2) % 3));
    assignTriangleEdgeFlag(tris[0], (edgeA + 2) % 3, old1.getEdgeFlag(edgeA));

    assignTriangleEdgeFlag(tris[1], edgeB, old1.getEdgeFlag((edgeA + 2) % 3));
    assignTriangleEdgeFlag(tris[1], (edgeB + 2) % 3, old2.getEdgeFlag(edgeB));

    calculateNormals();

    return 1;
  }

  //------------------
  //  MMeshEdgeData
  //------------------
  MMeshEdgeData::MMeshEdgeData() {
  }
  
  MMeshEdgeData::~MMeshEdgeData() {
    m_List.clear();
  }
  
  void MMeshEdgeData::setVertexNum(int NumVerts) {
    m_NumVerts = NumVerts;
    
    m_List.resize(m_NumVerts);
    for (int n = 0; n < m_NumVerts; n++) {
      m_List[n].clear();
    }
  }
  
  void MMeshEdgeData::addFace(int TriNum, MInternalTriangle *Tri) {
    if (!Tri) {
      return;
    }
    
    m_List[Tri->getVertex(0)].insert(TriNum);
    m_List[Tri->getVertex(1)].insert(TriNum);
    m_List[Tri->getVertex(2)].insert(TriNum);
  }
  
  void MMeshEdgeData::addFace(int TriNum, int VertA, int VertB, int VertC) {
    m_List[VertA].insert(TriNum);
    m_List[VertB].insert(TriNum);
    m_List[VertC].insert(TriNum);
  }
  
  bool MMeshEdgeData::edgeOnFace(int VertA, int VertB, int TriA) {
    std::set<int>::iterator it;

    if (m_List[VertA].find(TriA) != m_List[VertA].end()) {
      return true;
    }
    if (m_List[VertB].find(TriA) != m_List[VertB].end()) {
      return true;
    }

    return false;
  }

}


