#include "StdAfx.h"

#include <MMesh.h>

#include <MMath.h>
#include <MSystemManager.h>
#include <MScene.h>

#include <params/MArrayParameter.h>

#include "ModelGLConst.h"

#include <GL/gl.h>
#include <GL/glu.h>

#include <assert.h>
#include <math.h>
#include <algorithm>

#include <mesh/MVertexArrayParam.h>
#include <mesh/MTriangleArrayParam.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
  //-----------
  //  MMesh
  //-----------
  
  MMesh::MMesh() {
    componentMode = MComponentisedObject::OBJECT_TYPE;
    m_Tris = NULL;
    
    m_NumTris = 0;
    
    m_TextureMesh = NULL;

    normalsInvalid = false;


    MParameterObjectPtr meshParam;
    MParameterObjectPtr triParam;
    addParameter(meshParam = new MeshVertexArrayParam(this));
    addParameter(triParam = new MTriangleArrayParam(this));

    meshParam->setVisible(false);
    triParam->setVisible(false);
  }

  MMesh::MMesh(MMeshPtr src) {
    componentMode = src->componentMode;
    m_Tris = NULL;
    
    m_NumTris = 0;
    
    m_TextureMesh = NULL;

    MParameterObjectPtr meshParam;
    MParameterObjectPtr triParam;
    addParameter(meshParam = new MeshVertexArrayParam(this));
    addParameter(triParam = new MTriangleArrayParam(this));

    meshParam->setVisible(false);
    triParam->setVisible(false);

    setFrom(src);
  }

  MMesh::~MMesh() {
    cleanUpMesh();
    
    setTextureMesh(NULL);
  }
  
  MBaseObjectPtr MMesh::createNew() {
    MMeshPtr NewMesh;
    
    NewMesh = new MMesh;
    NewMesh->setFrom(this);
    
    return NewMesh;
  }
  
  void MMesh::setFrom(MBaseObjectPtr SrcObj) {
    if (SrcObj == NULL) {
      return;
    }
    
    MNamedObject::setFrom(SrcObj);
    
    cleanUpMesh();

    MMeshPtr Obj;
    
    Obj = AZTEC_CAST(MMesh, SrcObj);
    
    if (Obj == NULL) {
      return;
    }
    setComponentMode(Obj->getComponentMode());
    
    // Create the vertices
    if (Obj->getNumVerts() > 0) {
      m_Verts.resize(Obj->getNumVerts());
      for (int i = Obj->getNumVerts() - 1; i >= 0; i--) {
        m_Verts[i] = new MInternalVertex;
        *m_Verts[i] = *Obj->m_Verts[i];
      }
    }

    // Create the triangles
    if (Obj->m_NumTris > 0) {
      m_Tris = new MInternalTriangle*[Obj->m_NumTris];
      for (int i = Obj->m_NumTris - 1; i >= 0; --i) {
        m_Tris[i] = new MInternalTriangle(*Obj->m_Tris[i]);
      }
    } else {
      m_Tris = NULL;
    }
    
    m_NumTris = Obj->m_NumTris;
    
    if (Obj->m_TextureMesh != NULL) {
      setTextureMesh( AZTEC_CAST( MMesh, Obj->m_TextureMesh->createNew() ) );
    }
    
    invalidateNormals();
    calculateNormals();
  }
  
  int MMesh::flagVerticesOnTris(TRIFLAGS TriFlags, AztecFlags VertFlags)
  {
    int         n;
    MInternalTriangle   **T, *Tri;
  
    InternalVertexArray::iterator it, end = m_Verts.end();
    // Go through the vertices and unflag them all
    for (it = m_Verts.begin(); it != end; ++it) {
      (*it)->unsetFlag(VertFlags);
    }
  
    // go through the triangles, and if flagged set the flag on the vertices
    T = m_Tris;
    for (n=0; n<m_NumTris; n++)
    {
      Tri = *T;
      if (Tri->isFlagged(TriFlags))
      {
        m_Verts[Tri->getVertex(0)]->setFlag(VertFlags);
        m_Verts[Tri->getVertex(1)]->setFlag(VertFlags);
        m_Verts[Tri->getVertex(2)]->setFlag(VertFlags);
      }
      T++;
    }
    return 1;
  }

  int MMesh::getMeshExtents(MVector3 &Min, MVector3 &Max) {
    if (m_Verts.size() == 0) {
      return 0;
    }
  
    Min.set(*m_Verts[0]);
    Max.set(*m_Verts[0]);
  
    for (int n = 1; n < getNumVerts(); n++) {
      MInternalVertex *V;
      V = m_Verts[n];
    
      if (V->m_Pos.x < Min.x)
        Min.x = V->m_Pos.x;
      if (V->m_Pos.y < Min.y)
        Min.y = V->m_Pos.y;
      if (V->m_Pos.z < Min.z)
        Min.z = V->m_Pos.z;
    
      if (V->m_Pos.x > Max.x)
        Max.x = V->m_Pos.x;
      if (V->m_Pos.y > Max.y)
        Max.y = V->m_Pos.y;
      if (V->m_Pos.z > Max.z)
        Max.z = V->m_Pos.z;
    }
  
    return 1;
  }

  int MMesh::getMeshExtents(MVector3 &Min, MVector3 &Max, const MMatrix4 &XForm) {
    if (m_Verts.size() == 0) {
      return 0;
    }
  
    Min.set(XForm * (*m_Verts[0]));
    Max.set(XForm * (*m_Verts[0]));
  
    for (int n = 1; n < getNumVerts(); n++)
    {
      MVector3    V;
    
      V = XForm * (*m_Verts[n]);
    
      if (V.x < Min.x)
        Min.x = V.x;
      if (V.y < Min.y)
        Min.y = V.y;
      if (V.z < Min.z)
        Min.z = V.z;
    
      if (V.x > Max.x)
        Max.x = V.x;
      if (V.y > Max.y)
        Max.y = V.y;
      if (V.z > Max.z)
        Max.z = V.z;
    }
  
    return 1;
  }

  int MMesh::getNumVerts() const { 
    return m_Verts.size();
  }

  int MMesh::getNumTris() const { 
    return m_NumTris;
  }

  MVector3 MMesh::getVertexPosition(int index) const {
    return MVector3(*m_Verts[index]);
  }

  void MMesh::setVertexPosition(int index, MVector3 pos){
		m_Verts[index]->m_Pos = pos;
  }

  MVector3 MMesh::getVertexNormal(int index) const {
    if (!m_Verts[index]->isFlagged(VERTEX_NORMAL_OKAY)) {
      const_cast<MMesh*>(this)->calculateNormals();
    }
    return m_Verts[index]->m_Normal;
  }

  void MMesh::setVertexFlag(int index, AztecFlags flag) {
    ensureUndo();
    m_Verts[index]->setFlag(flag);
  }

  void MMesh::unsetVertexFlag(int index, AztecFlags flag) {
    ensureUndo();
    m_Verts[index]->unsetFlag(flag);
  }

  bool MMesh::isVertexFlagged(int index, AztecFlags flag) const {
    return m_Verts[index]->isFlagged(flag);
  }

  ///////////////
  const MInternalTriangle* MMesh::getTriangle(int triIndex) const {
    return m_Tris[triIndex];
  }

  int MMesh::getTriangleVertex(int triIndex, int cornerIndex) const {
    return m_Tris[triIndex]->m_Vertices[cornerIndex];
  }

  const MVector3& MMesh::getTriangleNormal(int triIndex) const {
    return m_Tris[triIndex]->m_Normal;
  }

  const MVector3& MMesh::getTriangleCentre(int triIndex) const {
    return m_Tris[triIndex]->m_Centre;
  }

  const MVector3& MMesh::getTriangleColour(int triIndex) const {
    return m_Tris[triIndex]->colour;
  }


  AztecFlags MMesh::getTriangleFlags(int triIndex) const {
    return m_Tris[triIndex]->triangleFlag.getFlags();
  }

  void MMesh::setTriangleFlag(int triIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->triangleFlag.setFlag(flag);
  }

  void MMesh::unsetTriangleFlag(int triIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->triangleFlag.unsetFlag(flag);
  }

  void MMesh::assignTriangleFlag(int triIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->triangleFlag.assignFlags(flag);
  }

  bool MMesh::isTriangleFlagged(int triIndex, AztecFlags flag) const {
    return m_Tris[triIndex]->triangleFlag.isFlagged(flag);
  }

  void MMesh::toggleTriangleFlag(int triIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->triangleFlag.toggleFlag(flag);
  }


  AztecFlags MMesh::getTriangleEdgeFlag(int triIndex, int edgeIndex) const {
    return m_Tris[triIndex]->edgeFlags[edgeIndex].getFlags();
  }

  void MMesh::setTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->edgeFlags[edgeIndex].setFlag(flag);
    int neigh = getTriangleNeighbour(triIndex, edgeIndex);

    if (neigh != -1) {
      int a, b;
      getVertsFromEdge(triIndex, edgeIndex, &a, &b);
      edgeIndex = getTriangleEdge(neigh, a, b);
      assert(edgeIndex != -1);
      m_Tris[neigh]->edgeFlags[edgeIndex].setFlag(flag);
    }
  }

  void MMesh::unsetTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->edgeFlags[edgeIndex].unsetFlag(flag);
  }

  void MMesh::assignTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag) {
    ensureUndo();
    m_Tris[triIndex]->edgeFlags[edgeIndex].assignFlags(flag);
  }

  bool MMesh::isTriangleEdgeFlagged(int triIndex, int edgeIndex, AztecFlags flag) const {
    return m_Tris[triIndex]->edgeFlags[edgeIndex].isFlagged(flag);
  }

  void MMesh::toggleTriangleEdgeFlag(int triIndex, int edgeIndex, AztecFlags flag) {
    m_Tris[triIndex]->edgeFlags[edgeIndex].toggleFlag(flag);
  }

  bool MMesh::isEdgeFlagged(int from, int to, Aztec::AztecFlags flag) {
    std::vector<int> tris;
    getTrianglesWithEdge(from, to, tris);

    for (int i = 0; i < tris.size(); ++i) {
      int e = getTriangleEdge(tris[i], from, to);
      if (isTriangleEdgeFlagged(tris[i], e, flag)) {
        return true;
      }
    }

    return false;
  }

  Aztec::AztecFlags MMesh::getEdgeFlag(int from, int to) {
    std::vector<int> tris;
    getTrianglesWithEdge(from, to, tris);

    for (int i = 0; i < tris.size(); ++i) {
      int e = getTriangleEdge(tris[i], from, to);
      return getTriangleEdgeFlag(tris[i], e);
    }

    return 0;
  }

  void MMesh::setEdgeFlag(int from, int to, Aztec::AztecFlags flag) {
    ensureUndo();
    std::vector<int> tris;
    getTrianglesWithEdge(from, to, tris);

    for (int i = 0; i < tris.size(); ++i) {
      int e = getTriangleEdge(tris[i], from, to);
      setTriangleEdgeFlag(tris[i], e, flag);
    }

  }
  void MMesh::unsetEdgeFlag(int from, int to, Aztec::AztecFlags flag) {
    ensureUndo();
    std::vector<int> tris;
    getTrianglesWithEdge(from, to, tris);

    for (int i = 0; i < tris.size(); ++i) {
      int e = getTriangleEdge(tris[i], from, to);
      unsetTriangleEdgeFlag(tris[i], e, flag);
    }

  }

  void MMesh::assignEdgeFlag(int from, int to, Aztec::AztecFlags flag) {
    ensureUndo();
    std::vector<int> tris;
    getTrianglesWithEdge(from, to, tris);

    for (int i = 0; i < tris.size(); ++i) {
      int e = getTriangleEdge(tris[i], from, to);
      assignTriangleEdgeFlag(tris[i], e, flag);
    }
  }



  int MMesh::getTriangleEdge(int triIndex, int VertA, int VertB) const {
    if (VertA == m_Tris[triIndex]->m_Vertices[0] && VertB == m_Tris[triIndex]->m_Vertices[1])
      return 0;
    if (VertB == m_Tris[triIndex]->m_Vertices[0] && VertA == m_Tris[triIndex]->m_Vertices[1])
      return 0;
  
    if (VertA == m_Tris[triIndex]->m_Vertices[1] && VertB == m_Tris[triIndex]->m_Vertices[2])
      return 1;
    if (VertB == m_Tris[triIndex]->m_Vertices[1] && VertA == m_Tris[triIndex]->m_Vertices[2])
      return 1;
  
    if (VertA == m_Tris[triIndex]->m_Vertices[2] && VertB == m_Tris[triIndex]->m_Vertices[0])
      return 2;
    if (VertB == m_Tris[triIndex]->m_Vertices[2] && VertA == m_Tris[triIndex]->m_Vertices[0])
      return 2;
  
    return -1;
  }


  int MMesh::getTriVert(int TriNum, int VertNum) const {
    if (TriNum < 0 || TriNum >= m_NumTris)
      return NULL;
  
    return m_Tris[TriNum]->getVertex(VertNum);
  }

  int MMesh::getTriFromEdge(int EdgeNum, int &triIndex, int &edgeIndex) const {
    triIndex = EdgeNum / 3;
    edgeIndex = EdgeNum % 3;
  
    return 1;
  }

  int MMesh::getVertsFromEdge(int EdgeNum, int *VertA, int *VertB) {
    int      TriNum, TriEdge;
  
    TriNum = EdgeNum / 3;
    TriEdge = EdgeNum % 3;
  
    if (TriNum >= m_NumTris || TriNum < 0) {
      return 0;
    }

    return getVertsFromEdge(TriNum, TriEdge, VertA, VertB);
  }

  int MMesh::getVertsFromEdge(int triangle, int edge, int *VertA, int *VertB) {
    *VertA = m_Tris[triangle]->getVertex(edge);
    *VertB = m_Tris[triangle]->getVertex((edge + 1) % 3);
    return 1;
  }

  int MMesh::getTriangleNeighbour(int triIndex, int edgeIndex) {
    int vertA = m_Tris[triIndex]->getVertex(edgeIndex);
    int vertB = m_Tris[triIndex]->getVertex((edgeIndex+1)%3);

    for (int triNum = 0; triNum < m_NumTris; ++triNum) {
      if (triNum != triIndex && m_Tris[triNum]->getEdge(vertA, vertB) != -1) {
        return triNum;
      }
    }
    return -1;
  }

  inline void addTo(std::vector<int> &v, int i) {
    std::vector<int>::iterator it = std::lower_bound(v.begin(), v.end(), i);
    if (it == v.end() || *it != i) {
      v.insert(it, i);
    }
  }

  inline bool sameEdge(int a1, int b1, int a2, int b2) {
    return ( (a1 == a2 && b1 == b2) || (a2 == b2 && b1 == a2) );
  }

  inline int getVertexNot(int a, int b, int v) {
    if (a == v) return b;
    if (b == v) return a;
    return -1;
  }

  int MMesh::getNextEdgeAroundVertex(int triangle, int edge, int vertex) {
    for (int i = 0; i < 3; ++i) {
      if (i == edge) continue;

      int a, b;
      getVertsFromEdge(triangle, i, &a, &b);

      if (a == vertex || b == vertex) {
        return i;
      }
    }

    // This should never happen on a well formed triangle
    assert(0);
    return -1;
  }

  void MMesh::getVerticesConnecting(int vertex, std::vector<int> &vertices, bool includeHidden) {
    // TODO: Make this faster. Possibly cache the values, or change the 
    // underlying data structure completely.
    int baseTri = -1, baseEdge = -1;

    vertices.clear();
    for (int tri = 0; tri < getNumTris(); ++tri) {
      for (int e = 0; e < 3; ++e) {
        if (getTriangleVertex(tri, e) == vertex &&
            isTriangleEdgeFlagged(tri, e, EDGE_VISIBLE)) 
        {
          baseTri = tri;
          baseEdge = e;
          break;
        }
      }
    }

    int numIterations = 0;

    if (baseTri != -1 && baseEdge != -1) {
      int tri = baseTri;
      int edge = baseEdge;

      std::vector<int> trisOnEdge;


      do {
        int a, b;
        getVertsFromEdge(tri, edge, &a, &b);

        if (includeHidden || isTriangleEdgeFlagged(tri, edge, EDGE_VISIBLE)) {
          vertices.push_back(getVertexNot(a, b, vertex));
        }

        int nextEdge = getNextEdgeAroundVertex(tri, edge, vertex);
        int nextTri = -1;

        getVertsFromEdge(tri, nextEdge, &a, &b);

        getTrianglesWithEdge(a, b, trisOnEdge);

        // Make sure our mesh is manifold, ie. only two triangles per edge.
#ifdef _DEBUG
        assert(trisOnEdge.size() <= 2);
#else 
        // just bail out, as there is nothing we can do.
        if (trisOnEdge.size() > 2) {
          MSystemManager::getInstance()->logOutput("ERROR: MMesh::getVerticesConnecting() - Mesh has more than two polygons connected to one edge. Danger Will Robinson! Danger!");
          vertices.clear();
          return;
        }
#endif
        for (int i = 0; i < trisOnEdge.size(); ++i) {
          if (trisOnEdge[i] != tri) {
            nextTri = trisOnEdge[i];
            nextEdge = getTriangleEdge(nextTri, a, b);
            break;
          }
        }

        // if we have hit a border, and we haven't turned around once already,
        // just start again in the other direction
        if (nextTri == -1) {
          if (numIterations > 0) {
            nextEdge = getNextEdgeAroundVertex(tri, edge, vertex);
            if (includeHidden || isTriangleEdgeFlagged(tri, nextEdge, EDGE_VISIBLE)) {
              getVertsFromEdge(tri, nextEdge, &a, &b);
              vertices.push_back(getVertexNot(a, b, vertex));
            }
            // since the vertices were retrieved in a backwards order, we have to flip them around.
            std::reverse(vertices.begin(), vertices.end());
            break;
          } else {
            numIterations++;

            nextTri = tri;
            nextEdge = getTriangleEdge(nextTri, a, b);
            vertices.clear();
          }
        }

        // otherwise we just keep going.
        edge = nextEdge;
        tri = nextTri;
        
      } while (tri != baseTri || edge != baseEdge);
    }

  }

  void MMesh::getTrianglesWithVertex(int vertex, std::vector<int> &triangles) {
    // TODO: Make this faster. Possibly cache the values, or change the 
    // underlying data structure completely.
    triangles.clear();
    for (int i = 0; i < getNumTris(); ++i) {
      const MInternalTriangle *tri = getTriangle(i);
      if (tri->getVertex(0) == vertex ||
          tri->getVertex(1) == vertex ||
          tri->getVertex(2) == vertex)
      {
        triangles.push_back(i);
      }
    }
  }

  void MMesh::getTrianglesWithEdge(int vertexFrom, int vertexTo, std::vector<int> &triangles) {
    // TODO: Make this faster. Possibly cache the values, or change the 
    // underlying data structure completely.
    triangles.clear();
    for (int i = 0; i < getNumTris(); ++i) {
      const MInternalTriangle *tri = getTriangle(i);
      for (int e = 0; e < 3; ++e) {
        if (tri->getVertex(e) == vertexFrom && 
          (tri->getVertex((e+1)%3) == vertexTo || tri->getVertex((e+2)%3) == vertexTo)) 
        {
          triangles.push_back(i);
          break;
        }
      }
    }
  }

  class CannotFindEdge {
  };

  bool MMesh::getPolygonVerticesFromEdge(int vertexFrom, int vertexTo, bool left, std::vector<int> &points) {
    points.clear();

    std::vector<int> p;
    int nextVertex = 0;

    int a = vertexFrom, b = vertexTo;
    // just walk around the connected vertices and collect up all the points.

    int count = 0;
    do {

      points.push_back(a);
      getVerticesConnecting(b, p);
      assert(p.size() > 0);

      int olda = a;
      int oldb = b;

      for (int i = 0; i < p.size(); ++i) {
        if (p[i] == a) {
          std::vector<int> edgeTris;
          getTrianglesWithEdge(a, b, edgeTris);

          if (edgeTris.size() == 0 || edgeTris.size() > 2) {
            points.clear();
            return false;
          }

          // tri and find an edge on a triangle that is in the correct 
          // order to ensure that the left hand side of the edge HAS a polygon.
          bool triFound = false;
          for (int tri = 0; tri < edgeTris.size(); ++tri) {
            int edgeNum = getTriangleEdge(edgeTris[tri], a, b);
            assert(edgeNum != -1);
            // now make sure it is in the correct order
            if (getTriVert(edgeTris[tri], edgeNum) == (left ? b : a)) {
              triFound = true;
              break;
            }
          }

          if (!triFound) {
            points.clear();
            return false;
          }

          i =  ( i + (left ? 1 : p.size() - 1) ) % p.size(); 
          a = b;
          b = p[i];
          if (a == b) {
            return false;
          }
          break;
        }
      }

      assert(olda != a || oldb != b);

#ifdef _DEBUG
      {
        std::vector<int> newp;
        getVerticesConnecting(b, newp);
        bool found = false;
        for (int i = 0; i < newp.size(); ++i) {
          if (newp[i] == a) {
            found = true;
            break;
          }
        }
        assert(found);
      }
#endif

      ++count;
      if (count > 250) {
        throw CannotFindEdge();
      }
    } while (a != b && a != vertexFrom);

    return (a != b && a == vertexFrom);
  }

  int MMesh::getSideOfEdgeForTriangle(int vertexFrom, int vertexTo, int triangleIndex) {
    int edgeIndex = getTriangleEdge(triangleIndex, vertexFrom, vertexTo);
    if (edgeIndex == -1) return 0;

    if (getTriangleVertex(triangleIndex, edgeIndex) == vertexFrom) {
      return 1;
    } else {
      return -1;
    }
  }

  void MMesh::getPolygonVerticesFromTriangle(int triangle, std::vector<int> &points) {

    // find a visible edge
    int startEdge = -1;
    for (int i = 0; i < 3; ++i) {
      if (isTriangleEdgeFlagged(triangle, i, EDGE_VISIBLE)) {
        startEdge = i;
        break;
      }
    }

    if (startEdge == -1) {
      points.clear();
      return;
    }

    // pick a visible edge
    int from = getTriangleVertex(triangle, startEdge);
    int to = getTriangleVertex(triangle, (startEdge+1)%3);

    int dir = getSideOfEdgeForTriangle(from,to,triangle);
    if (dir == 0) {
      return;
    }

    getPolygonVerticesFromEdge(from, to, dir == -1, points);
  }


  int MMesh::getNumVertsFlagged(AztecFlags Flags) {
    int      Count = 0, n;
    for (n=0; n < getNumVerts(); n++) {
      if (m_Verts[n]->isFlagged(Flags))
        Count++;
    }
    return Count;
  }

  int MMesh::getNumTrisFlagged(AztecFlags Flags) {
    int      Count = 0, n;
    for (n=0; n<m_NumTris; n++) {
      if (m_Tris[n]->isFlagged(Flags))
        Count++;
    }
    return Count;
  }


  int MMesh::createGLVertexArray(float *&Array)
  {
    float    *Val;
  
    Array = new float[m_NumTris*3];
    Val = Array;
  
    for (int t=0; t<m_NumTris; t++)
    {
      for (int vn=2;vn>=0;vn--)
      {
        *Val = m_Verts[m_Tris[t]->getVertex(vn)]->m_Pos.x; Val++;
        *Val = m_Verts[m_Tris[t]->getVertex(vn)]->m_Pos.y; Val++;
        *Val = m_Verts[m_Tris[t]->getVertex(vn)]->m_Pos.z; Val++;
      }
    }
  
    return 0;
  }

  void MMesh::setTextureMesh(MMeshPtr Mesh) {
    ensureUndo();
    m_TextureMesh = Mesh;
  }

  void MMesh::calculateNormals() {
    if (!normalsInvalid) {
      return;
    }

    if (m_Verts.size() == 0) {
      return;
    }

    MInternalTriangle   *T;
    int         n;
  
    for (n = 0; n < getNumVerts(); n++) {
      if (!m_Verts[n]->isFlagged(VERTEX_NORMAL_OKAY)) {
        m_Verts[n]->m_Normal.set(0,0,0);
      } else {
        m_Verts[n]->unsetFlag(VERTEX_NORMAL_NEEDS_UPDATE);
      }
    }

    for (n = 0; n < getNumTris(); n++) {
      for (int i = 0; i < 3; ++i) {
        if (!m_Verts[m_Tris[n]->m_Vertices[i]]->isFlagged(VERTEX_NORMAL_OKAY)) {
          m_Tris[n]->unsetFlag(TRIANGLE_NORMAL_VALID);
          break;
        }
      }
    }
  
    MVector3	va,vb,vc,vd,ve,vn;
    for (n = 0;n < m_NumTris; n++) {
      T = m_Tris[n];

      T->m_Centre = (m_Verts[T->m_Vertices[0]]->m_Pos + 
                     m_Verts[T->m_Vertices[1]]->m_Pos + 
                     m_Verts[T->m_Vertices[2]]->m_Pos)*(float)(1.0/3.0);
      
      if (T->isFlagged(TRIANGLE_NORMAL_VALID)) {
        continue;
      }

      std::set<int> poly;

      getPolygonFromTriangle(n, poly);

      std::set<int> points;

      int a_index = -1;
      int b_index = -1;
      for (std::set<int>::iterator i = poly.begin(); i != poly.end(); ++i) {
        for (int j = 0; j < 3; ++j) {
          if (a_index == -1 && b_index == -1 && m_Tris[*i]->isEdgeFlagged(j, EDGE_VISIBLE)) {
            a_index = m_Tris[*i]->m_Vertices[j];
            b_index = m_Tris[*i]->m_Vertices[(j+1)%3];
          }
        }
        points.insert(m_Tris[*i]->getVertex(0));
        points.insert(m_Tris[*i]->getVertex(1));
        points.insert(m_Tris[*i]->getVertex(2));
      }

      MVector3 a = m_Verts[a_index]->m_Pos; 
      MVector3 b = m_Verts[b_index]->m_Pos;
      MVector3 ab = b - a;
      MVector3 c, norm;

      for (std::set<int>::iterator point = points.begin(); point != points.end(); ++point) {
        if (*point != a_index && *point != b_index) {
          norm += ab / (m_Verts[*point]->m_Pos - a);

        }
      }

      norm.normalize();

      for (std::set<int>::iterator i = poly.begin(); i != poly.end(); ++i) {
        m_Tris[*i]->m_Normal = norm;
        m_Tris[*i]->setFlag(TRIANGLE_NORMAL_VALID);
      }

      MInternalVertex *v[3];

      v[0] = m_Verts[T->getVertex(0)];
      v[1] = m_Verts[T->getVertex(1)];
      v[2] = m_Verts[T->getVertex(2)];

      if (v[0]->isFlagged(VERTEX_NORMAL_OKAY) &&
          v[1]->isFlagged(VERTEX_NORMAL_OKAY) &&
          v[2]->isFlagged(VERTEX_NORMAL_OKAY)) {
        continue;
      }

      va = v[0]->m_Pos;
      vb = v[1]->m_Pos;
      vc = v[2]->m_Pos;
    
      vd = vb-va;
      ve = vc-vb;
    
      v[0]->setFlag(VERTEX_NORMAL_NEEDS_UPDATE);
      v[1]->setFlag(VERTEX_NORMAL_NEEDS_UPDATE);
      v[2]->setFlag(VERTEX_NORMAL_NEEDS_UPDATE);
      v[0]->m_Normal.set(0,0,0);
      v[1]->m_Normal.set(0,0,0);
      v[2]->m_Normal.set(0,0,0);

    }

    for (n = 0;n < m_NumTris; n++) {
      T = m_Tris[n];
      MInternalVertex *v[3];

      v[0] = m_Verts[T->getVertex(0)];
      v[1] = m_Verts[T->getVertex(1)];
      v[2] = m_Verts[T->getVertex(2)];

      if (v[0]->isFlagged(VERTEX_NORMAL_NEEDS_UPDATE)) {
        v[0]->m_Normal += T->m_Normal;
      }
      if (v[1]->isFlagged(VERTEX_NORMAL_NEEDS_UPDATE)) {
        v[1]->m_Normal += T->m_Normal;
      }
      if (v[2]->isFlagged(VERTEX_NORMAL_NEEDS_UPDATE)) {
        v[2]->m_Normal += T->m_Normal;
      }
      
      // do the triangle normalise *after* altering the normals of 
      // the vertices, so the area of the triangle effects the outcome.
      T->m_Normal.normalize();
    
    }
    
    for (n = 0; n < getNumVerts(); n++) {
      if (m_Verts[n]->isFlagged(VERTEX_NORMAL_NEEDS_UPDATE)) {
        m_Verts[n]->m_Normal.normalize();
        m_Verts[n]->setFlag(VERTEX_NORMAL_OKAY);
      }
    }

    normalsInvalid = false;
  }

  MMeshPtr MMesh::convertToMesh() {
    return this;
  }

  bool MMesh::drawObject(const MBaseObjectPtr &baseObj, MSceneViewFlags ViewFlags) {
    calculateNormals();
    MSystemManagerPtr SysMan;
    MScenePtr Scene;
  
    SysMan = Aztec::getSystemManager();
  
    assert(SysMan != NULL);
    Scene = SysMan->getScene();
  
  
    MMeshPtr TextureMesh, ActualMesh;
    int i;
  
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  
    if (ViewFlags.m_TextureMode != glNone) {
      glEnable(GL_TEXTURE_2D);
    } else {
      glDisable(GL_TEXTURE_2D);
    }

    if (ViewFlags.m_GLLighting) {
      glEnable(GL_LIGHTING);
//      glEnable(GL_COLOR_MATERIAL);
//      glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    } else {
      glDisable(GL_LIGHTING);
//      glDisable(GL_COLOR_MATERIAL);
    }
    
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, ViewFlags.m_TextureMode);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, ViewFlags.m_TextureMode);

    if (ViewFlags.m_ShadingMode == glWireframe) {
      glColor4fv((float*)&ViewFlags.m_WireColor);
    } else {
      glColor4fv((float*)&ViewFlags.m_FlatColor);
    }
  
    TextureMesh = getTextureMesh();
  
  
    bool     DoTriangleColor;
    float    *RegularColor;
  
    DoTriangleColor = isInComponentMode(FACET_TYPE);
  
    if (ViewFlags.m_ShadingMode == glWireframe) {
      RegularColor = (float*)&ViewFlags.m_WireColor;
    } else {
      RegularColor = (float*)&ViewFlags.m_FlatColor; 
    }
  
    ActualMesh = AZTEC_CAST( MMesh, convertToMesh() );

#ifdef DRAW_NORMALS
    // draw the normals of the vertices
    glBegin(GL_LINES);
    for (int i = 0; i < getNumVerts(); ++i) {
      MInternalVertex *vert = m_Verts[i];
      glColor3f(1,0,0);
      glVertex3f(vert->m_Pos.x, vert->m_Pos.y, vert->m_Pos.z);
      glColor3f(1,1,0);
      glVertex3f(vert->m_Pos.x + vert->m_Normal.x * 3, 
        vert->m_Pos.y + vert->m_Normal.y * 3, 
        vert->m_Pos.z + vert->m_Normal.z * 3);
    }
#endif

    if (ViewFlags.m_SelectMethod != smNone) {
      ActualMesh->drawSelectableMesh(ViewFlags);
      return true;
    }

    //   if (ViewFlags.m_ShadeMode != glWire && (ViewFlags.m_SelectMethod == smNone || !isFlagged(OBJECTFLAG_COMPONENTMODE)))     // If we are not selecting anything, draw as 
    if (ViewFlags.m_SelectMethod == smNone || !isInComponentMode())
    {
      if (ViewFlags.m_ShadingMode != glWireframe)
      {
        MInternalVertex *V, *SkinVert;
        MInternalTriangle      *T, *SkinT;
        float triangleColour[4];
      
        SkinT = NULL;
        SkinVert = NULL;
      
        glColor4fv(RegularColor);
      
        glBegin(GL_TRIANGLES);
      
        for (i=0;i<getNumTris();i++)
        {
          T = m_Tris[i];

          if (T->isFlagged(TRIANGLE_HAS_COLOUR)) {
            triangleColour[0] = T->colour.x;
            triangleColour[1] = T->colour.y;
            triangleColour[2] = T->colour.z;
            triangleColour[3] = 1;

          } else {
            triangleColour[0] = RegularColor[0];
            triangleColour[1] = RegularColor[1];
            triangleColour[2] = RegularColor[2];
            triangleColour[3] = RegularColor[3];
          }
        
          if (TextureMesh != NULL) {
            if (TextureMesh->m_Tris != NULL) {
              SkinT = TextureMesh->m_Tris[i];
            } else {
              SkinT = NULL;
            }

          }
          if (DoTriangleColor) {
            if (T->isFlagged(TRIANGLE_SELECTED)) {
              glColor4f(1,1,0,1);
            } else {
              glColor4fv(triangleColour);
            }
          }

          if (ViewFlags.m_ShadingMode == glFlat) {
            glNormal3fv((float*)&T->m_Normal);
          }
        
          for (int vn = 0; vn<3; vn++) {
            V = ActualMesh->m_Verts[T->getVertex(vn)];
            ActualMesh->m_Verts;
            
            if (!V) {
              continue;
            }
            
            if (TextureMesh != NULL && SkinT) {
              SkinVert = TextureMesh->m_Verts[SkinT->getVertex(vn)];
            } else {
              SkinVert = NULL;
            }
            
            if (ViewFlags.m_ShadingMode != glFlat) {
              glNormal3fv((float*)&V->m_Normal);
            }
            
            if (SkinVert) {
              glTexCoord2fv((float*)&SkinVert->m_Pos);
            }
            
            if (V->isFlagged(VERTEX_HAS_COLOUR)) {
              glColor3fv((float*)&V->m_VertexColour);
            } else {
              glColor4fv(triangleColour);
            }
            
            glVertex3fv((float*)&V->m_Pos);
          }
        }
        glEnd();
      }
    }
  
    glDepthFunc(GL_LEQUAL);
    //   if (ViewFlags.m_DrawWire && ViewFlags.m_SelectMethod == smNone) 
    if (ViewFlags.m_DrawWire) {
      GLint    OldLighting, OldTexture;
    
      glGetIntegerv(GL_LIGHTING, &OldLighting);
      glGetIntegerv(GL_TEXTURE_2D, &OldTexture);
      glDisable(GL_LIGHTING);
      glDisable(GL_TEXTURE_2D);
    
      glEnable(GL_POLYGON_OFFSET_LINE);
      glEnable(GL_POLYGON_OFFSET_FILL);
      glPolygonOffset(-1.0, -2.0);
    
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glEdgeFlag(GL_TRUE);
    
      // Now draw those triangles that do have edges selected
      if (ViewFlags.m_ComponentMode == MComponentisedObject::EDGE_TYPE) {
        glLineWidth(3);
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        glBegin(GL_LINES);
        for (i=0;i<getNumTris();i++) {
          MInternalVertex *V[3];
          MInternalTriangle      *T;
        
          T = m_Tris[i];
        
          if (!T) {
            continue;
          }
        
          V[0] = ActualMesh->m_Verts[T->getVertex(0)];
          V[1] = ActualMesh->m_Verts[T->getVertex(1)];
          V[2] = ActualMesh->m_Verts[T->getVertex(2)];
        
          for (int edgeNum = 0; edgeNum < 3; ++edgeNum) {
            if (T->edgeFlags[edgeNum].isFlagged(EDGE_SELECTED)) {
              glColor3f(1,0,0);
              glVertex3fv((float*)&V[edgeNum]->m_Pos);
              glVertex3fv((float*)&V[(edgeNum+1)%3]->m_Pos);
            }
          }
        }
        glEnd();
        glLineWidth(1);
      }
    
      glDepthFunc(GL_LESS);


      if (ViewFlags.m_ComponentMode == MComponentisedObject::EDGE_TYPE) {
        glLineWidth(2);
      }

      glColor4fv((float*)&ViewFlags.m_WireColor);
    

      // Draw the triangles that have not been selected with edges
      glBegin(GL_TRIANGLES);
      for (i=0;i<getNumTris();i++) {
        MInternalVertex *V;
        MInternalTriangle      *T;
      
        T = m_Tris[i];
      
        if (!T) {
          continue;
        }
      
        if (DoTriangleColor) {
          if (T->isFlagged(TRIANGLE_SELECTED)) {
            glEnd();
            glColor4f(1,1,0,0.5);
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glBegin(GL_TRIANGLES);
          } else {
            glColor4fv((float*)&ViewFlags.m_WireColor);
          }
        }
      
        for (int vn = 0;vn<3;vn++) {
          V = ActualMesh->m_Verts[T->getVertex(vn)];
          if (!V) {
            continue;
          }
        
		  // ARE: The logical or of EDGE_VISIBLE and EDGE_SELECTED produces
		  // an "int" by default.  This causes a type ambiguity when looking
		  // for a version of the member function isFlagged(...).  Therefore
		  // the type cast - the largest size chosen in case the definitions
		  // chage at some point.
          bool vis = T->edgeFlags[vn].isFlagged(EDGE_VISIBLE);
          bool sel = ViewFlags.m_ComponentMode == MComponentisedObject::EDGE_TYPE && 
                     T->edgeFlags[vn].isFlagged(EDGE_SELECTED);

          glEdgeFlag(vis && !sel);
          glVertex3fv((float*)&V->m_Pos);
        }
      
        if (DoTriangleColor) {
          if (T->isFlagged(TRIANGLE_SELECTED)) {
            glEnd();
            glColor4fv((float*)&ViewFlags.m_WireColor);
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            glBegin(GL_TRIANGLES);
          }
        }
      
      }
      glEnd();

      glLineWidth(1);

      glEdgeFlag(GL_TRUE);
    
      glDisable(GL_POLYGON_OFFSET_LINE);
      glDisable(GL_POLYGON_OFFSET_FILL);
    
      if (OldLighting) {
        glEnable(GL_LIGHTING);
      }
      if (OldLighting) {
        glEnable(GL_TEXTURE_2D);
      }
      if (ViewFlags.m_ShadingMode == glWireframe) {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      } else {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      }
    }
   
    glDisable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
  
    // Now the object is drawn using the correct shading method and textureing method
    // We now do the selection and component related parts of the model.
  
    // In component mode, we hilight all vertices with big dots. And selected vertices are in yellow
    // In component mode, all faces are drawn in blue outline, and green solid if it is selected
    // do the selection and component stuff.
  
#ifdef _DEBUG  
    if (isInComponentMode() && isInComponentMode(MComponentisedObject::POINT_TYPE)) {
#else
    if (isInComponentMode() && isInComponentMode(MComponentisedObject::POINT_TYPE)) {
#endif
      int      i;
      // Draw components.
    
      // Draw vertices first.
      glPointSize(6);
    
      glDisable(GL_TEXTURE_2D);
    
      glBegin(GL_POINTS);
    
      const bool inPointMode = isInComponentMode(MComponentisedObject::POINT_TYPE);
      int Num = getNumVerts();
      for (i=0;i<Num;i++)
      {
        MInternalVertex *V;
        MRGBAFloat feedbackColour;
      
        V = ActualMesh->m_Verts[i];
      
        if (!V || !V->isFlagged(VERTEX_VISIBLE)) {
          continue;
        }
      
        feedbackColour = ActualMesh->m_Verts[i]->m_FeedbackColor;
      
        // Feedback colors are more important (?)
        if (V->isFlagged(VERTEX_DRAW_FEEDBACK)) {
          glColor4fv((float*)&feedbackColour);
        } else if (inPointMode && V->isFlagged(VERTEX_SELECTED)) 
        {
          glColor3f(1.0f, 0.1f, 0.1f);
        } else {
          glColor3f(0.1f, 0.8f, 0.1f);
        }
      
        glVertex3fv((float*)&V->m_Pos);
      
      }
    
      glEnd();
    }
  
    ActualMesh = NULL;

    return 1;
  }

  void MMesh::drawSelectableMesh(MSceneViewFlags ViewFlags) {
    if (ViewFlags.m_ComponentMode == MComponentisedObject::OBJECT_TYPE) {
      glBegin(GL_TRIANGLES);
      for (int i = 0; i < getNumTris(); i++) {
        MInternalTriangle      *T;
      
        // Place the object on the selection stack if necessary
        T = m_Tris[i];
      
        for (int vn = 0;vn<3;vn++) {
          glVertex3fv((float*)&(m_Verts[T->getVertex(vn)])->m_Pos);
        }
      
      }
      glEnd();

    } if (ViewFlags.m_ComponentMode == MComponentisedObject::FACET_TYPE)
    {
      glPushName(SELECTITEM_FACE);
      glPushName(-1);
      for (int i = 0; i < getNumTris(); i++)
      {
        MInternalVertex *V;
        MInternalTriangle      *T;
      
        // Place the object on the selection stack if necessary
        T = m_Tris[i];
      
        glLoadName(i);
      
        glBegin(GL_TRIANGLES);
      
        for (int vn = 0;vn<3;vn++)
        {
          V = m_Verts[T->getVertex(vn)];
        
          glVertex3fv((float*)&V->m_Pos);
        }
      
        glEnd();
      }
      glPopName();
      glPopName();
    }
    else if (ViewFlags.m_ComponentMode == MComponentisedObject::EDGE_TYPE)
    {
      int      CurName;
    
      glPushName(SELECTITEM_EDGE);
      glPushName(-1);
    
      CurName = 0;
      for (int i = 0; i < getNumTris(); i++)
      {
        MInternalVertex *V1, *V2;
        MInternalTriangle      *T;
      
        // Place the object on the selection stack if necessary
        T = m_Tris[i];
      
        for (int vn = 0;vn<3;vn++)
        {
        
          if (T->isEdgeFlagged(vn, EDGE_VISIBLE)) {
            glLoadName(CurName);
            V1 = m_Verts[T->getVertex(vn)];
            V2 = m_Verts[T->getVertex((vn+1)%3)];
        
            glBegin(GL_LINES);
            glVertex3fv((float*)&V1->m_Pos);
            glVertex3fv((float*)&V2->m_Pos);
            glEnd();
          }
        
          CurName++;
        }
      
      }
      glPopName();
      glPopName();
    } else if (ViewFlags.m_ComponentMode == MComponentisedObject::POINT_TYPE) {
      glPushName(SELECTITEM_VERTEX);
      glPushName(-1);
    
      int Num = getNumVerts();
      for (int i=0;i<Num;i++)
      {
        MInternalVertex *V = m_Verts[i];
      
        if (V == NULL || !V->isFlagged(VERTEX_VISIBLE)) {
          continue;
        }
      
        glLoadName(i);
        glBegin(GL_POINTS);
        glVertex3fv((float*)&V->m_Pos);
        glEnd();
      }
    
      glPopName();
      glPopName();
    }
  }

  void MMesh::getPolygonFromTriangle(int triIndex, std::set<int> &trisInPoly) {
    MInternalTriangle *tri = m_Tris[triIndex];

    trisInPoly.insert(triIndex);

    // traverse over the edges
    for (int edge = 0; edge < 3; ++edge) {
      // skip over any edge we've just been to, or if
      // the edge is solid.
      if (tri->edgeFlags[edge].isFlagged(EDGE_VISIBLE)) {
        continue;
      }

      int neighTri = getTriangleNeighbour(triIndex, edge);

      if (neighTri != -1 && trisInPoly.find(neighTri) == trisInPoly.end()) {
        getPolygonFromTriangle(neighTri, trisInPoly);
      }
    }
  }

  int MMesh::selectComponent(AztecFlags Mode, int CompNum, bool TargetSel)
  {
    ensureUndo();
    if (TargetSel) {
      return flagComponent(Mode, CompNum, COMPONENT_SELECTED, atSet);
    } else {
      return flagComponent(Mode, CompNum, COMPONENT_SELECTED, atUnset);
    }
    MSystemManagerPtr SysMan;
    MScenePtr Scene;
    MMeshPtr Mesh;
  
    // Convert this object to a mesh so we use the mesh at the end of the modifier list
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
  
    if (Mesh == NULL)
      return 0;
  
    SysMan = Aztec::getSystemManager();
  
    assert(SysMan != NULL);
    Scene = SysMan->getScene();
  
    int      Result;
  
    Result = 0;
  
    if (Mode & MComponentisedObject::POINT_TYPE)
    {
      MInternalVertex *Vert;
      Vert = Mesh->m_Verts[CompNum];
    
      if (Vert)
      {
        // Check to see if the selection will change
        if (Vert->isFlagged(VERTEX_SELECTED) != TargetSel)
          Result = 1;
      
        if (TargetSel)
          Vert->setFlag(VERTEX_SELECTED);
        else
          Vert->unsetFlag(VERTEX_SELECTED);
      }
    }
    if (Mode & MComponentisedObject::FACET_TYPE)
    {
      MInternalTriangle  *Tri;
      Tri = Mesh->m_Tris[CompNum];
    
      if (Tri)
      {
        if (Tri->isFlagged(TRIANGLE_SELECTED) != TargetSel)
          Result = 1;
      
        if (TargetSel)
          Tri->setFlag(TRIANGLE_SELECTED);
        else
          Tri->unsetFlag(TRIANGLE_SELECTED);
      }
    }
    if (Mode & MComponentisedObject::EDGE_TYPE)
    {
      int      TriNum, EdgeNum;
    
      TriNum = CompNum / 3;
      EdgeNum = CompNum % 3;
    
      MInternalTriangle  *Tri;
      Tri = Mesh->m_Tris[TriNum];
    
      if (Tri) {
        if (Tri->edgeFlags[EdgeNum].isFlagged(EDGE_SELECTED) != TargetSel) {
          Result = 1;
        }
        Tri->edgeFlags[EdgeNum].setFlag(EDGE_SELECTED);
      
        if (TargetSel) {
          Tri->edgeFlags[EdgeNum].setFlag(EDGE_SELECTED);
        } else {
          Tri->edgeFlags[EdgeNum].unsetFlag(EDGE_SELECTED);
        }
      }
    }

    Mesh = NULL;
  
    return Result;
  }

  int MMesh::isComponentSelected(AztecFlags Mode, int CompNum)
  {
    MSystemManagerPtr SysMan;
    MScenePtr Scene;
    MMeshPtr Mesh;
  
    // Convert this object to a mesh so we use the mesh at the end of the modifier list
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
  
    if (Mesh == NULL)
      return 0;
  
    SysMan = Aztec::getSystemManager();
  
    assert(SysMan != NULL);
    Scene = SysMan->getScene();
  
    int      Result;
  
    Result = 0;
  
    if (Mode & MComponentisedObject::POINT_TYPE) {
      MInternalVertex *Vert;
      Vert = Mesh->m_Verts[CompNum];
    
      if (Vert) {
        Result = Vert->isFlagged(VERTEX_SELECTED);
      }
    }
    if (Mode & MComponentisedObject::FACET_TYPE) {
      MInternalTriangle  *Tri;
      Tri = Mesh->m_Tris[CompNum];
    
      if (Tri) {
        Result = Tri->isFlagged(TRIANGLE_SELECTED);
      }
    }
    if (Mode & MComponentisedObject::EDGE_TYPE) {
      int      TriNum, EdgeNum;

      TriNum = CompNum / 3;
      EdgeNum = CompNum % 3;
    
      MInternalTriangle  *Tri;
      Tri = Mesh->m_Tris[TriNum];
    
      if (Tri) {
        Result = Tri->edgeFlags[EdgeNum].isFlagged(EDGE_SELECTED);
      }
    }
  
    Mesh = NULL;
  
    return Result;
  
  }

  void MMesh::selectComponentToggle(AztecFlags Mode, int CompNum) {
    if (isComponentSelected(Mode, CompNum)) {
      selectComponent(Mode, CompNum, false);
    } else {
      selectComponent(Mode, CompNum, true);
    }
  }

  void MMesh::flagPolygon(int triIndex, AztecFlags flag, MAction action) {
    unsetTriangleFlags(TRIANGLE_FLAGFORCHANGE2);
    doFlagPolygon(triIndex, flag, action, -1);
  }

  void MMesh::setVertexFlags(AztecFlags Flags) {
    ensureUndo();
    for (int i=0; i < getNumVerts(); i++) {
      m_Verts[i]->setFlag(Flags);
    }
  }

  void MMesh::unsetVertexFlags(AztecFlags Flags) {
    ensureUndo();
    for (int i=0; i < getNumVerts(); i++) {
      m_Verts[i]->unsetFlag(Flags);
    }
  }

  void MMesh::setTriangleFlags(AztecFlags Flags) {
    ensureUndo();
    for (int i=0; i<m_NumTris; i++) {
      m_Tris[i]->setFlag(Flags);
    }
  }

  void MMesh::unsetTriangleFlags(AztecFlags Flags) {
    ensureUndo();
    for (int i=0; i<m_NumTris; i++) {
      m_Tris[i]->unsetFlag(Flags);
    }
  }

  // MComponentisedObject methods
  bool MMesh::isInComponentMode(AztecFlags type) {
    return (componentMode & type) != 0;
  }

  void MMesh::setComponentMode(AztecFlags type) {
    if (componentMode != type) {
      ensureUndo();
      componentMode = type;
    }
  }

  AztecFlags MMesh::getComponentMode() {
    return componentMode;
  }

  bool MMesh::hasComponentType(AztecFlags type) {
    return (type | (POINT_TYPE & EDGE_TYPE & FACET_TYPE)) == 0;

  }

  int MMesh::getComponentCount(AztecFlags type) {
    if (type & POINT_TYPE) {
      return getNumVerts();
    } else if (type & EDGE_TYPE) {
      return getNumTris()*3;
    } else if (type & FACET_TYPE) {
      return getNumTris();
    }
    return 0;
  }

  
  MFlagObject* MMesh::getFlagObjectForComponent(AztecFlags type, int index) {
    if (type & POINT_TYPE) {
      return dynamic_cast<MFlagObject *>(m_Verts[index]);
    } else if (type & EDGE_TYPE) {
      int      triNum, edgeNum;
    
      triNum = index / 3;
      edgeNum = index % 3;

      return  &m_Tris[triNum]->edgeFlags[edgeNum];
    } else if (type & FACET_TYPE) {
      return &m_Tris[index]->triangleFlag;
    }

    return NULL;
  }

  bool MMesh::flagComponent(AztecFlags type, int index, AztecFlags flag, MAction action) {
    ensureUndo();
    if (type == FACET_TYPE) {
      flagPolygon(index, flag, action);
      return true;
    }

    MFlagObject *flagObj = getFlagObjectForComponent(type, index);
    
    if (flagObj != NULL) {
      if (action == atSet) {
        if (!flagObj->isFlagged(flag)) {
          flagObj->setFlag(flag);
          return true;
        }
      } else if (action == atUnset) {
        if (flagObj->isFlagged(flag)) {
          flagObj->unsetFlag(flag);
          return true;
        }
      } else {
        flagObj->toggleFlag(flag);
        return true;
      }
    }
    return false;
  }

  bool MMesh::isComponentFlagged(AztecFlags type, int index, AztecFlags flag) {
    MFlagObject *flagObj = getFlagObjectForComponent(type, index);

    if (flagObj != NULL) {
      return flagObj->isFlagged(flag);
    }
    return false;
  }

  bool MMesh::hasComponentsFlagged(AztecFlags type, AztecFlags flag) {
    if (type & POINT_TYPE) {
      for (int n = getNumVerts() - 1; n >= 0; --n) {
        if (m_Verts[n]->isFlagged(flag)) {
          return true;
        }
      }
    }
    if (type & FACET_TYPE) {
      for (int n = m_NumTris - 1; n >= 0; --n) {
        if (m_Tris[n]->isFlagged(flag)) {
          return true;
        }
      }
    }
    if (type & EDGE_TYPE) {
      for (int n = m_NumTris - 1; n >= 0; --n) {
        if (m_Tris[n]->edgeFlags[0].isFlagged(flag) ||
            m_Tris[n]->edgeFlags[1].isFlagged(flag) || 
            m_Tris[n]->edgeFlags[2].isFlagged(flag) ) {
          return true;
        }
      }
    }
    return false;
  }

  bool MMesh::setComponentFeedbackColour(AztecFlags type, int index, const MRGBAFloat &colour) {
    ensureUndo();
    switch (type) {
    case POINT_TYPE:
      {
        MMeshPtr Mesh;
        MInternalVertex* vertex;
        
        // Use the mesh at the end of the modifier list
        Mesh = AZTEC_CAST(MMesh, convertToMesh());
        if (Mesh == NULL) return false;
        
        vertex = Mesh->m_Verts[index];
        if (vertex != NULL) {
          vertex->m_FeedbackColor = colour;
          vertex->setFlag(VERTEX_DRAW_FEEDBACK);
          // TODO: notify the mesh that something has changed.
        }
        return true;
      }
    default:
      return false;
    }
  }

  bool MMesh::unsetComponentFeedbackColour(AztecFlags type, int index) {
    ensureUndo();
    switch (type) {
    case POINT_TYPE:
      {
        MMeshPtr Mesh;
        MInternalVertex* vertex;
        
        // Use the mesh at the end of the modifier list
        Mesh = AZTEC_CAST(MMesh, convertToMesh());
        if (Mesh == NULL) return false;
        
        vertex = Mesh->m_Verts[index];
        if (vertex != NULL) {
          vertex->m_FeedbackColor.set(0.0f, 0.0f, 0.0f, 0.f);
          vertex->unsetFlag(VERTEX_DRAW_FEEDBACK);
        }
        return true;
      }

    default:
      return false;
    }
  }

  MRGBAFloat MMesh::getComponentFeedbackColour(AztecFlags type, int index) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE:
      return m_Verts[index]->m_FeedbackColor;
    default:
      return MRGBAFloat(0,0,0,1);
    }
  }

  bool MMesh::flagConnected(ComponentType type, AztecFlags flag) {
    ensureUndo();
    if (type == POINT_TYPE) {
      // loop over all the vertices, connecting all the selected ones together.
      std::vector<bool> visited;
      visited.resize(getNumVerts(), false);

      for (int i = 0; i < getNumVerts(); ++i) {
        if (isVertexFlagged(i, flag)) {
          flagConnectedVertices(i, flag, visited);
        }
      }
    }
    return false;
  }

  bool MMesh::growFlagged(ComponentType type, AztecFlags flag) {
    ensureUndo();
    return false;
  }

  inline void checkVertex(MMesh *mesh, int vertex, AztecFlags flag, std::set< std::pair<int, int> > &edges) {
    std::vector<int> others;
    mesh->getVerticesConnecting(vertex, others);
    int count = 0;
    for (int i = 0; i < others.size(); ++i) {
      if (mesh->isEdgeFlagged(vertex, others[i], flag)) {
        ++count;
      }
    }

    // if there is only one edge, go and mark it for deflagging
    if (count == 1) {
      for (int i = 0; i < others.size(); ++i) {
        if (mesh->isEdgeFlagged(vertex, others[i], flag)) {
          edges.insert( std::pair<int, int>(vertex, others[i]) );
        }
      }
    }
  }

  bool MMesh::shrinkFlagged(ComponentType type, AztecFlags flag) {
    ensureUndo();
    if (type == EDGE_TYPE) {
      typedef std::pair<int, int> Edge;
      std::set<Edge> edges;

      for (int i = 0; i < getNumVerts(); ++i) {
        checkVertex(this, i, flag, edges);
      }

      // now go and unflag the edges
      for (std::set<Edge>::iterator it = edges.begin(); it != edges.end(); ++it) {
        this->unsetEdgeFlag(it->first, it->second, flag);
      }

      return true;
    } else {
      return false;
    }
  }

  MVector3 MMesh::getComponentCentre(AztecFlags type, int index) {
    if (type & POINT_TYPE) {
      return MVector3(*m_Verts[index]);
    } else if (type & EDGE_TYPE) {
      MVector3 centre;
      int triNum = index / 3;
      int edgeNum = index % 3;

      int *triVerts = m_Tris[triNum]->m_Vertices;
      centre = *m_Verts[triVerts[edgeNum]];
      centre += *m_Verts[triVerts[(edgeNum + 1) % 3]];
      centre *= 0.5;

      return centre;
    } else if (type & FACET_TYPE) {
      return m_Tris[index]->m_Centre;
    }

    return MVector3();
  }

  MVector3 MMesh::getFlaggedCentre(AztecFlags type, AztecFlags flags) {
    int count;
    MVector3 Centre;
    MMeshPtr Mesh;
  
    Centre.set(0,0,0);
    count = 0;
  
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
  
    if (Mesh == NULL) {
      return Centre;
    }

    if (type == MComponentisedObject::OBJECT_TYPE || type == MComponentisedObject::POINT_TYPE) {
      InternalVertexArray::iterator vertIt, end = m_Verts.end();
      for (vertIt = m_Verts.begin(); vertIt != end; ++vertIt) {
        MInternalVertex *Vert = *vertIt;
    
        if (type == MComponentisedObject::OBJECT_TYPE ||
            flags == 0 || Vert->isFlagged(flags)) {
          Centre.x += Vert->m_Pos.x;
          Centre.y += Vert->m_Pos.y;
          Centre.z += Vert->m_Pos.z;
          ++count;
        }
      }
    } else if (type == MComponentisedObject::FACET_TYPE) {
      for (int n = 0; n < m_NumTris; ++n) {
        if (flags == 0 || m_Tris[n]->isFlagged(flags)) {
          Centre += m_Tris[n]->m_Centre;
          ++count;
        }
      }

    } else if (type == MComponentisedObject::EDGE_TYPE) {
      for (int n = 0; n < m_NumTris; ++n) {
        MInternalTriangle *tri = m_Tris[n];
        if (flags == 0 || tri->edgeFlags[0].isFlagged(flags)) {
          Centre += *m_Verts[tri->getVertex(0)];
          Centre += *m_Verts[tri->getVertex(1)];
          count += 2;
        }
        if (flags == 0 || tri->edgeFlags[1].isFlagged(flags)) {
          Centre += *m_Verts[tri->getVertex(1)];
          Centre += *m_Verts[tri->getVertex(2)];
          count += 2;
        }
        if (flags == 0 || tri->edgeFlags[2].isFlagged(flags)) {
          Centre += *m_Verts[tri->getVertex(0)];
          Centre += *m_Verts[tri->getVertex(2)];
          count += 2;
        }
      }

    } else {
      assert(0);
    }
        

    if (count > 0) {
      Centre.x /= count;
      Centre.y /= count;
      Centre.z /= count;
    }
  
    Mesh = NULL;
  
    return Centre;
  }


  MVector3 MMesh::getComponentNormal(AztecFlags type, int index) {
    calculateNormals();
    if (type & POINT_TYPE) {
      return m_Verts[index]->m_Normal;
    } else if (type & EDGE_TYPE) {
      MVector3 normal;
      int triNum = index / 3;
      int edgeNum = index % 3;

      int *triVerts = m_Tris[triNum]->m_Vertices;
      normal = m_Verts[triVerts[edgeNum]]->m_Normal;
      normal += m_Verts[triVerts[(edgeNum + 1) % 3]]->m_Normal;
      normal.normalize();

      return normal;
    } else if (type & FACET_TYPE) {
      return m_Tris[index]->m_Normal;
    }

    return MVector3();
  }

  void MMesh::finishWithUndoNode() {
    undoNode = NULL;
    MNamedObject::finishWithUndoNode();
  }


  void MMesh::invalidateNormals() {
    normalsInvalid = true;
  }

  void MMesh::cleanUpMesh() {
    for (unsigned int n = 0; n < m_Verts.size(); ++n) {
      if (m_Verts[n] != NULL) {
        delete m_Verts[n];
      }
    }
    m_Verts.clear();
    if (m_Tris) {
      for (int n=0;n<m_NumTris;n++) {
        if (m_Tris[n]) {
          delete m_Tris[n];
        }
      }
      delete[] m_Tris;
      m_Tris = NULL;
    }
  }

  void MMesh::doFlagPolygon(int triIndex, AztecFlags flag, MAction action, int ignoreEdge) {
    ensureUndo();
    MInternalTriangle *tri = m_Tris[triIndex];

    // if we have already done this triangle, skip it.
    if (tri->isFlagged(TRIANGLE_FLAGFORCHANGE2)) {
      return;
    }

    if (action == atSet) {
      tri->setFlag(flag);
    } else if (action == atUnset) {
      tri->unsetFlag(flag);
    } else {
      tri->toggleFlag(flag);
    }

    tri->setFlag(TRIANGLE_FLAGFORCHANGE2);

    // traverse over the edges
    for (int edge = 0; edge < 3; ++edge) {
      // skip over any edge we've just been to, or if
      // the edge is solid.
      if (edge == ignoreEdge || tri->edgeFlags[edge].isFlagged(EDGE_VISIBLE)) {
        continue;
      }

      int neighTri = getTriangleNeighbour(triIndex, edge);

      if (neighTri != -1) {
        doFlagPolygon(neighTri, flag, action, -1);
      }
    }
  }

  void MMesh::setVertexColour(int index, const MVector3 &colour)
  {
    ensureUndo();
    MMeshPtr Mesh;
    MInternalVertex* vertex;
        
    // Use the mesh at the end of the modifier list
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
    if (Mesh == NULL) 
		return;
        
    vertex = Mesh->m_Verts[index];
    if (vertex != NULL) {
      vertex->m_VertexColour = colour;
	  vertex->setFlag(VERTEX_HAS_COLOUR);
    }
  }

  MVector3 MMesh::getVertexColour(int index)
  {
	MMeshPtr Mesh;
    MInternalVertex* vertex;

    MVector3 nullMVect3;
    
    // Use the mesh at the end of the modifier list
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
    if (Mesh == NULL) 
		return nullMVect3;
        
    vertex = Mesh->m_Verts[index];
    if (vertex != NULL) {
      return vertex->m_VertexColour;
    }

	return nullMVect3;
  }

  void MMesh::unsetVertexColour(int index)
  {
    ensureUndo();
    MMeshPtr Mesh;
    MInternalVertex* vertex;

    MVector3 nullMVect3;
        
    // Use the mesh at the end of the modifier list
    Mesh = AZTEC_CAST(MMesh, convertToMesh());
    if (Mesh == NULL) 
		return;
        
    vertex = Mesh->m_Verts[index];
    if (vertex != NULL) {
      vertex->m_VertexColour = nullMVect3;
	  vertex->unsetFlag(VERTEX_HAS_COLOUR);
    }
  }

  void MMesh::setTriangleColour(int index, const MVector3 &colour) {
    ensureUndo();
    m_Tris[index]->colour = colour;
    m_Tris[index]->setFlag(TRIANGLE_HAS_COLOUR);
  }

  MVector3 MMesh::getTriangleColour(int index) {
    return m_Tris[index]->colour;
  }

  void MMesh::unsetTriagleColour(int index) {
    ensureUndo();
    m_Tris[index]->unsetFlag(TRIANGLE_HAS_COLOUR);
  }

  void MMesh::flagConnectedVertices(int index, AztecFlags flag, std::vector<bool> &visited) {
    ensureUndo();
    if (visited[index]) {
      return;
    }

    visited[index] = true;
    setVertexFlag(index, flag);

    // get the connected vertices
    std::vector<int> c;

    getVerticesConnecting(index, c);

    for (int i = 0; i < c.size(); ++i) {
      if (!isVertexFlagged(c[i], flag) && !visited[c[i]]) {
        flagConnectedVertices(c[i], flag, visited);
      }
    }
    
  }


  void MMesh::ensureUndo() {
    if (undoNode == NULL && MUndoManager::isEnabled()) {
      undoNode = new MeshUndoNode(this);
      MUndoManager::getInstance()->addUndoNode(undoNode);
    }
  }

  void copy(MInternalVertex &lhs, const MInternalVertex &rhs) {
    lhs.m_Pos = rhs.m_Pos;
    lhs.m_Normal = rhs.m_Normal;
    lhs.m_VertexColour = rhs.m_VertexColour;
    lhs.assignFlags(rhs.getFlags());
  }

  void copy(MInternalTriangle &lhs, const MInternalTriangle &rhs) {
    lhs = rhs;
  }

  void MMesh::copyToUndo(MeshUndoNode *node) {
    node->storedComponentMode = componentMode;

    node->verts.resize(m_Verts.size());
    for (unsigned i = 0; i < m_Verts.size(); ++i) {
      copy(node->verts[i], *m_Verts[i]);
    }

    node->tris.resize(m_NumTris);
    for (int i = 0; i < m_NumTris; ++i) {
      copy(node->tris[i], *m_Tris[i]);
    }

    node->textureMesh = m_TextureMesh;

  }

  void MMesh::swapWithUndo(MeshUndoNode *node) {
    std::swap(node->storedComponentMode,componentMode);

    std::vector<MInternalVertex> tempVerts = node->verts;
    std::vector<MInternalTriangle> tempTris = node->tris;

    // copy from the mesh -> node
    node->verts.resize(m_Verts.size());
    for (unsigned i = 0; i < m_Verts.size(); ++i) {
      copy(node->verts[i], *m_Verts[i]);
    }

    node->tris.resize(m_NumTris);
    for (int i = 0; i < m_NumTris; ++i) {
      copy(node->tris[i], *m_Tris[i]);
    }

    // copy from our temporary -> mesh
    while (tempVerts.size() > m_Verts.size()) {
      m_Verts.push_back(new MInternalVertex());
    }
    while (tempVerts.size() < m_Verts.size()) {
      delete m_Verts.back();
      m_Verts.pop_back();
    }

    for (unsigned i = 0; i < tempVerts.size(); ++i) {
      copy(*m_Verts[i], tempVerts[i]);
    }

    MInternalTriangle **newTris = NULL;
    if (tempTris.size() != m_NumTris) {
      if (tempTris.size() > 0) {
        newTris = new MInternalTriangle*[tempTris.size()];
        if (tempTris.size() > m_NumTris) {
          for (int i = 0; i < m_NumTris; ++i) {
            newTris[i] = m_Tris[i];
          }
          while (tempTris.size() > m_NumTris) {
            newTris[m_NumTris] = new MInternalTriangle();
            ++m_NumTris;
          }
        } else {
          while (tempTris.size() < m_NumTris) {
            --m_NumTris;
            delete m_Tris[m_NumTris];
          }
          for (int i = 0; i < m_NumTris; ++i) {
            newTris[i] = m_Tris[i];
          }
        }
        delete[] m_Tris;
        m_Tris = newTris;
      } else {
        for (int i = 0; i < m_NumTris; ++i) {
          delete m_Tris[i];
        }
        delete[] m_Tris;
        m_Tris = NULL;
        m_NumTris = 0;
      }

    }

    assert(m_NumTris == tempTris.size());

    for (unsigned i = 0; i < tempTris.size(); ++i) {
      copy(*m_Tris[i], tempTris[i]);
    }

    std::swap(node->textureMesh, m_TextureMesh);
  }

  MMesh::MeshUndoNode::MeshUndoNode(const MMeshPtr &m) {
    mesh = m;
    mesh->copyToUndo(this);
  }

  // MBaseUndoNode Methods
  MUndoableObject* MMesh::MeshUndoNode::getObject() {
    return &*mesh;
  }

  void MMesh::MeshUndoNode::undo() {
    mesh->swapWithUndo(this);
  }

  void MMesh::MeshUndoNode::redo() {
    mesh->swapWithUndo(this);
  }




  //----------------------------------------------------------------------------------------
  //  MInternalVertex
  //----------------------------------------------------------------------------------------

  MInternalVertex::MInternalVertex(const MInternalVertex &Src) 
    : MFlagObject(Src) 
  {
    m_Pos = Src.m_Pos;
    m_Normal = Src.m_Normal;
    m_Flags = Src.m_Flags;
  }

  MInternalVertex::MInternalVertex(const MVector3 &Src) 
    : m_Pos(Src) 
  {
    setFlag(VERTEX_VISIBLE);
  }

  MInternalVertex::MInternalVertex(float nx, float ny, float nz) 
    : m_Pos(nx,ny,nz) 
  {

    setFlag(VERTEX_VISIBLE);
  }

  MMatrix4 MInternalVertex::getMatrix() {
    return MMatrix4(MVector3(0,0,1), m_Normal);
  }

  MInternalVertex& MInternalVertex::operator=(const MVector3 &RHS) {
    m_Pos = RHS;
  
    return *this;
  }

  //----------------------------------------------------------------------------------------
  //  MInternalTriangle
  //----------------------------------------------------------------------------------------
  MInternalTriangle::MInternalTriangle()
  {
    m_Vertices[0] = 0;
    m_Vertices[1] = 0;
    m_Vertices[2] = 0;
    edgeFlags[0].setFlag(EDGE_VISIBLE);
    edgeFlags[1].setFlag(EDGE_VISIBLE);
    edgeFlags[2].setFlag(EDGE_VISIBLE);
  }

  MInternalTriangle::MInternalTriangle(int a, int b, int c) 
  {
    m_Vertices[0] = a;
    m_Vertices[1] = b;
    m_Vertices[2] = c;
    edgeFlags[0].setFlag(EDGE_VISIBLE);
    edgeFlags[1].setFlag(EDGE_VISIBLE);
    edgeFlags[2].setFlag(EDGE_VISIBLE);
  }

  MInternalTriangle::MInternalTriangle(const MInternalTriangle &src)
    : triangleFlag(src.triangleFlag),
      m_Normal(src.m_Normal),
      m_Centre(src.m_Centre),
      colour(src.colour)
      
  {
    m_Vertices[0] = src.m_Vertices[0];
    m_Vertices[1] = src.m_Vertices[1];
    m_Vertices[2] = src.m_Vertices[2];
    edgeFlags[0] = src.edgeFlags[0];
    edgeFlags[1] = src.edgeFlags[1];
    edgeFlags[2] = src.edgeFlags[2];
  }

  MInternalTriangle& MInternalTriangle::operator=(const MInternalTriangle &src)
  {
    m_Vertices[0] = src.m_Vertices[0];
    m_Vertices[1] = src.m_Vertices[1];
    m_Vertices[2] = src.m_Vertices[2];
    triangleFlag = src.triangleFlag;
    edgeFlags[0] = src.edgeFlags[0];
    edgeFlags[1] = src.edgeFlags[1];
    edgeFlags[2] = src.edgeFlags[2];
    m_Normal = src.m_Normal;
    m_Centre = src.m_Centre;
    colour = src.colour;
    return *this;
  }

  /**
   * Gets the index into the mesh's vertex array for the given triangle vertex. 
   */
  int MInternalTriangle::getVertex(int index) const {
    return m_Vertices[index];
  }

  void MInternalTriangle::setVertex(int index, int vertexIndex) {
    m_Vertices[index] = vertexIndex;
  }


  const MVector3& MInternalTriangle::getNormal() const {
    return m_Normal;
  }

  const MVector3& MInternalTriangle::getCentre() const {
    return m_Centre;
  }

  const MVector3& MInternalTriangle::getColour() const {
    return colour;
  }


  AztecFlags MInternalTriangle::getFlags() const {
    return triangleFlag.getFlags();
  }

  void MInternalTriangle::setFlag(AztecFlags flag) {
    triangleFlag.setFlag(flag);
  }

  void MInternalTriangle::unsetFlag(AztecFlags flag) {
    triangleFlag.unsetFlag(flag);
  }

  void MInternalTriangle::assignFlag(AztecFlags flag) {
    triangleFlag.assignFlags(flag);
  }

  bool MInternalTriangle::isFlagged(AztecFlags flag) const {
    return triangleFlag.isFlagged(flag);
  }

  void MInternalTriangle::toggleFlag(AztecFlags flag) {
    triangleFlag.toggleFlag(flag);
  }


  AztecFlags MInternalTriangle::getEdgeFlag(int index) const {
    return edgeFlags[index].getFlags();
  }

  void MInternalTriangle::setEdgeFlag(int index, AztecFlags flag) {
    edgeFlags[index].setFlag(flag);
  }

  void MInternalTriangle::unsetEdgeFlag(int index, AztecFlags flag) {
    edgeFlags[index].unsetFlag(flag);
  }

  void MInternalTriangle::assignEdgeFlag(int index, AztecFlags flag) {
    edgeFlags[index].assignFlags(flag);
  }

  bool MInternalTriangle::isEdgeFlagged(int index, AztecFlags flag) const {
    return edgeFlags[index].isFlagged(flag);
  }

  void MInternalTriangle::toggleEdgeFlag(int index, AztecFlags flag) {
    edgeFlags[index].toggleFlag(flag);
  }

  int MInternalTriangle::getEdge(int VertA, int VertB) const {
    if (VertA == m_Vertices[0] && VertB == m_Vertices[1])
      return 0;
    if (VertB == m_Vertices[0] && VertA == m_Vertices[1])
      return 0;
  
    if (VertA == m_Vertices[1] && VertB == m_Vertices[2])
      return 1;
    if (VertB == m_Vertices[1] && VertA == m_Vertices[2])
      return 1;
  
    if (VertA == m_Vertices[2] && VertB == m_Vertices[0])
      return 2;
    if (VertB == m_Vertices[2] && VertA == m_Vertices[0])
      return 2;
  
    return -1;
  }

}



