#include "StdAfx.h"
#include "MParameterObjectList.h"

#include "MSystemManager.h"
#include <params/MVector3KeyParameter.h>

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
  std::vector<std::string> MParameterObjectList::tempMapping;

  MParameterObjectList::MParameterObjectList(MBaseObject *owner) {
    m_Owner = owner;
  }
  
  MParameterObjectList::~MParameterObjectList() {
    deleteAllParams();
  }
  
  MBaseObject* MParameterObjectList::getOwner() const {
    assert(m_Owner == NULL || AZTEC_CAST(MNamedObject, m_Owner) != NULL);

    return m_Owner;
  }

  int MParameterObjectList::addParam(MParameterObjectPtr ParamObj) {
    if (ParamObj == NULL) {
      return 0;
    }
    
    // Add it to the end of the param list.
    m_ParamList.push_back(ParamObj);
    ParamObj->setOwner(m_Owner);

    makeConnection(ParamObj);

#ifdef ADD_X_Y_Z_PARAMS
    // if we have a keyable vector3 parameter, add in the xyz components as well
    MVector3KeyParameter *vector3Param = AZTEC_CAST(MVector3KeyParameter, ParamObj);
    if (vector3Param != NULL) {
      MParameterObjectPtr param;

      bool doStorage = false;

      // if the parameter we are adding should be stored, then store the, xyz params instead.
      if (vector3Param->isStorable()) {
        doStorage = true;
        vector3Param->setStorable(false);
      }

      param = vector3Param->getXParameter();
      param->setVisible(false);
      param->setStorable(doStorage);
      addParam(param);

      param = vector3Param->getYParameter();
      param->setVisible(false);
      param->setStorable(doStorage);
      addParam(param);

      param = vector3Param->getZParameter();
      param->setVisible(false);
      param->setStorable(doStorage);
      addParam(param);
    }
#endif

    return m_ParamList.size();
  }
  
  int MParameterObjectList::replaceParamObject(const MStr &Name, MParameterObjectPtr Param) {
    unsigned int n;
    
    if (Param == NULL) {
      return 0;
    }
    
    for (n=0; n<m_ParamList.size(); n++) {
      if (m_ParamList[n]->isNamed(Name)) {
        break;
      }
    }
    if (n == m_ParamList.size()) {
      return 0;
    }
    
    breakConnection(m_ParamList[n]);
    m_ParamList[n] = Param;
    makeConnection(Param);
    
    return 1;
  }
  
  
  void MParameterObjectList::deleteAllParams() {
    for (unsigned int i = 0; i < m_ParamList.size(); i++) {
      breakConnection(m_ParamList[i]);
      m_ParamList[i] = NULL;
    }
    m_ParamList.clear();
  }
  
  void MParameterObjectList::removeParameter(int index) {
    if (index < 0 || index >= getNumParams()) {
      return;
    }

    breakConnection(m_ParamList[index]);
    m_ParamList.erase(m_ParamList.begin() + index);
  }

  MParameterObjectPtr MParameterObjectList::getParameter(const MStr &name) {
    return getParameter( getParameterIndex( name ) );
  }

  MParameterObjectPtr MParameterObjectList::getParameter(int index) { 
    if (index < 0 || index >= getNumParams()) {
      return NULL;
    }

    return m_ParamList[index];
  }

  int MParameterObjectList::getParameterIndex(const MStr &name) {
    unsigned int n;

    // first search the short names
    for (n = 0; n < m_ParamList.size(); n++) {
      if (m_ParamList[n]->getShortName() == name) {
        return n;
      }
    }

    // then search the long names
    for (n = 0; n < m_ParamList.size(); n++) {
      if (m_ParamList[n]->getLongName() == name) {
        return n;
      }
    }

    // then seatch the friendly names
    for (n = 0; n < m_ParamList.size(); n++) {
      if (m_ParamList[n]->getFriendlyName() == name) {
        return n;
      }
    }

    // if we have gotten to here, there is nothing that has
    // the name we are after.
    return -1;
  }
 
  int MParameterObjectList::getNumParams() { 
    return m_ParamList.size(); 
  }

  int MParameterObjectList::setFromList(MParameterObjectListPtr SrcList) {
    if (SrcList == NULL) {
      return 0;
    }
    
    int      n, NumSet = 0;
    MParameterObjectPtr SrcParam, DestParam;
    
    for (n=0; n<SrcList->getNumParams(); n++){
      SrcParam = SrcList->m_ParamList[n];
      DestParam = getParameter( SrcParam->getShortName() );
      
      if (DestParam == NULL)
        continue;
      
      NumSet += DestParam->setFromParameter(SrcParam);
    }
    return NumSet;
  }
  
  bool MParameterObjectList::doUpdateObject() {
    MDAGNode::doUpdateObject();

    // we manually unset the flag here, otherwise
    // we would end up in an infite loop.
    unsetFlag(OBJECTFLAG_NEEDS_UPDATE);
    if (m_Owner != NULL) {
      m_Owner->updateObject();
    }
    return true;
  }

  void MParameterObjectList::makeConnection(MParameterObjectPtr parameter) {
    MDAGNode *ownerNode = AZTEC_CAST(MDAGNode, m_Owner);
    if (ownerNode != NULL) {
      if (parameter->isInputParameter()) {
        ownerNode->addInput(&*parameter);      
      } else {
        ownerNode->addOutput(&*parameter);      
      }
    }
  }

  void MParameterObjectList::breakConnection(MParameterObjectPtr parameter) {
    MDAGNode *ownerNode = AZTEC_CAST(MDAGNode, m_Owner);
    if (ownerNode != NULL) {
      // Object Parameters need to be handled differently
      MObjectParameter *objParam = AZTEC_CAST(MObjectParameter, parameter);
      if (parameter->isInputParameter()) {
        ownerNode->removeInput(&*parameter);      
      } else {
        ownerNode->removeOutput(&*parameter);      
      }
    }
  }


}