#include "StdAfx.h"

#include "MUIManager.h"
#include "MComponentisedObject.h"

#include <math.h>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

namespace Aztec {
  // this is added in to provide initialisation of the UI manager.
  MUIManager static_UIManager;

  const int MUIManager::COMPONENT_MODE_LISTENER = 1;
  const int MUIManager::AXIS_MODE_LISTENER = 2;
  const int MUIManager::SEPARATE_AXIS_LISTENER = 3;
  const int MUIManager::SNAP_MODE_LISTENER = 3;
  const int MUIManager::COORDINATE_FEEDBACK_LISTENER = 4;

  MUIManager::ComponentType m_CompMode = MComponentisedObject::OBJECT_TYPE;
  int m_AxisMode = MUIManager::WORLD_AXIS;
  bool m_SeparateAxis = false;
  MUIManager::SnapModeType snapMode = MUIManager::SNAP_NONE;
  MVector3 worldCoords;
  MVector3 diffCoords;
  bool worldCoordsEnabled = false;
  bool diffCoordsEnabled = false;

  std::multimap<int, MListenerPtr> listeners;

  static void fireListeners(int action) {
    std::multimap<int, MListenerPtr>::iterator it;

    for (it = listeners.begin(); it != listeners.end(); ++it) {
      if (it->first == action) {
        it->second->onListen();
      }
    }
  }

  class MUIManagerUndoableObject : public MUndoableObject {
  public:
    // MUndoableObject methods
    void finishWithUndoNode();
  };

  static MUIManagerUndoableObject undoableUIManager;

  class MUIManagerUndoNode : public MBaseUndoNode {
  public:
    MUIManagerUndoNode() {
      storedCompMode = m_CompMode;
      storedAxisMode = m_AxisMode;
      storedSeparateAxis = m_SeparateAxis;
      storedSnapMode = snapMode;
      storedWorldCoords = worldCoords;
      storedDiffCoords = diffCoords;
      storedWorldCoordsEnabled = worldCoordsEnabled;
      storedWiffCoordsEnabled = diffCoordsEnabled;
      storedListeners = listeners;
    }

    // MBaseUndoNode Methods
    MUndoableObject* getObject() {
      return &undoableUIManager;
    }

    void undo() {
      std::swap(storedCompMode, m_CompMode);
      std::swap(storedAxisMode, m_AxisMode);
      std::swap(storedSeparateAxis, m_SeparateAxis);
      std::swap(storedSnapMode, snapMode);
//      std::swap(storedWorldCoords, worldCoords);
//      std::swap(storedDiffCoords, diffCoords);
//      std::swap(storedWorldCoordsEnabled, worldCoordsEnabled);
//      std::swap(storedWiffCoordsEnabled, diffCoordsEnabled);
//      std::swap(storedListeners, listeners);

      for (std::multimap<int, MListenerPtr>::iterator it = listeners.begin(); it != listeners.end(); ++it) {
        it->second->onListen();
      }
    }

    void redo() {
      undo();
    }

    MUIManager::ComponentType storedCompMode;
    int storedAxisMode;
    bool storedSeparateAxis;
    MUIManager::SnapModeType storedSnapMode;
    MVector3 storedWorldCoords;
    MVector3 storedDiffCoords;
    bool storedWorldCoordsEnabled;
    bool storedWiffCoordsEnabled;

    std::multimap<int, MListenerPtr> storedListeners;
  };

  typedef MRefCountedPtr<MUIManagerUndoNode> MUIManagerUndoNodePtr;

  static MUIManagerUndoNodePtr uiManagerUndoNode;
  static void ensureUndo() {
    if (uiManagerUndoNode == NULL && MUndoManager::isEnabled()) {
      uiManagerUndoNode = new MUIManagerUndoNode();
      MUndoManager::getInstance()->addUndoNode(uiManagerUndoNode);
    }
  }

  void MUIManagerUndoableObject::finishWithUndoNode() {
    uiManagerUndoNode = NULL;
  }


  MUIManager::MUIManager() {
  }

  MUIManager::~MUIManager() {
  }

  MUIManager::ComponentType MUIManager::getComponentMode() {
    return m_CompMode;
  }

  void MUIManager::setComponentMode(ComponentType mode) {
    ensureUndo();
    m_CompMode = mode;
    fireListeners(COMPONENT_MODE_LISTENER);
  }

  int MUIManager::getAxisMode() {
    return m_AxisMode;
  }

  void MUIManager::setAxisMode(int mode) {
    ensureUndo();
    m_AxisMode = mode;
    fireListeners(AXIS_MODE_LISTENER);
  }

  bool MUIManager::getSeparateAxisMode() {
    return m_SeparateAxis;
  }

  void MUIManager::setSeparateAxisMode(bool mode) {
    ensureUndo();
    bool oldMode = m_SeparateAxis;

    m_SeparateAxis = mode;

    if (oldMode != mode) {
      fireListeners(SEPARATE_AXIS_LISTENER);
    }
  }

  MUIManager::SnapModeType MUIManager::getSnapMode() {
    return snapMode;
  }

  void MUIManager::setSnapMode(SnapModeType mode) {
    ensureUndo();
    if (snapMode != mode) {
      snapMode = mode;
      fireListeners(SNAP_MODE_LISTENER);
    }
  }

  void MUIManager::setWorldCoordinates(const MVector3 &coords) {
    bool changed = false;

    if (!worldCoordsEnabled) {
      worldCoordsEnabled = true;
      changed = true;
    }

    MVector3 diff = coords - worldCoords;
    if (diff.length2() != 0) {
      worldCoords = coords;
      changed = true;
    }

    if (changed) {
      fireListeners(MUIManager::COORDINATE_FEEDBACK_LISTENER);
    }
  }

  void MUIManager::setDiffCoordinates(const MVector3 &coords) {
    bool changed = false;

    if (!diffCoordsEnabled) {
      diffCoordsEnabled = true;
      changed = true;
    }

    MVector3 diff = coords - diffCoords;
    if (diff.length2() != 0) {
      diffCoords = coords;
      changed = true;
    }

    if (changed) {
      fireListeners(MUIManager::COORDINATE_FEEDBACK_LISTENER);
    }
  }

  void MUIManager::unsetWorldCoordinates() {
    bool changed = false;

    if (worldCoordsEnabled) {
      changed = true;
      worldCoordsEnabled = false;
    }
    if (changed) {
      fireListeners(MUIManager::COORDINATE_FEEDBACK_LISTENER);
    }
  }


  void MUIManager::unsetDiffCoordinates() {
    bool changed = false;

    if (diffCoordsEnabled) {
      changed = true;
      diffCoordsEnabled = false;
    }

    if (changed) {
      fireListeners(MUIManager::COORDINATE_FEEDBACK_LISTENER);
    }
  }

  void MUIManager::unsetCoordinates() {
    unsetDiffCoordinates();
    unsetWorldCoordinates();
  }

  bool MUIManager::getWorldCoordinates(MVector3 &coords) {
    coords = worldCoords;
    return worldCoordsEnabled;
  }

  bool MUIManager::getDiffCoordinates(MVector3 &coords) {
    coords = diffCoords;
    return diffCoordsEnabled;
  }



  void MUIManager::addListener(int action, MListenerPtr listener) {
    listeners.insert( std::multimap<int, MListenerPtr>::value_type(action, listener) );
    listener->onListen();
  }

  void MUIManager::removeListener(int action, MListenerPtr listener) {
    std::multimap<int, MListenerPtr>::iterator it;

    for (it = listeners.begin(); it != listeners.end(); ++it) {
      if (it->first == action  && it->second == listener) {
        listeners.erase(it);
        break;
      }
    }
  }

  void MUIManager::removeAllListeners() {
    listeners.clear();
  }

  static MUIManager::TimeStampEnum timeStampSetting = MUIManager::Frames;

  MUIManager::TimeStampEnum MUIManager::getUserTimeStamp() {
    return timeStampSetting;
  }

  void MUIManager::setUserTimeStamp(MUIManager::TimeStampEnum stamp) {
    timeStampSetting = stamp;
  }

  std::string MUIManager::tickCountToTimeStamp(int tick) {
    if (timeStampSetting == Ticks) {
      // Ticks is jsut a straight int to string conversion.
      char buf[16];
      sprintf(buf, "%i", tick);
      return buf;
    } else if (timeStampSetting == MinutesSecondsMilliseconds) {
      // this is of the form:
      //    minute:seconds.millisconds
      
      double realSeconds = MScene::getGlobalScene()->tickToSeconds(tick);

      // if we have a negative, compensate for it
      bool negative = realSeconds < 0;
      realSeconds = fabs(realSeconds);

      // workout the number of minutes
      int minutes = (int)(realSeconds / 60.0);
      realSeconds -= minutes * 60;

      // truncate off the real part.
      int seconds = (int)realSeconds;

      // now get the remaining fraction
      double realMilli = realSeconds - seconds;
      int milli = (int)(1000.0 * realMilli);

      char buf[16];
      sprintf(buf, "%i:%i.%i", minutes, seconds, milli);
      return buf;

    } else {
      // Do Frames if we can't find anything else to do.
      char buf[16];
      sprintf(buf, "%i", (int)MScene::getGlobalScene()->tickToFrame(tick));
      return buf;
    } 
  }


}


