// Registry.cpp: implementation of the MRegistry class.
//
//////////////////////////////////////////////////////////////////////
#include "StdAfx.h"

#include "RegClass.h"

#include <assert.h>

#ifndef WIN32
#include <stdlib.h>
#endif

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

namespace Aztec {

  const std::string MRegistry::CURRENT_USER = "HKCU";

  MRegistry::MRegistry()
  {
  #ifdef REGCLASS_USE_REGISTRY
	  m_hRootKey = HKEY_CURRENT_USER;
	  m_hCurrentKey = NULL;
  #else
    m_RootKey = "CurrentUser";
  #endif
  }

  MRegistry::~MRegistry()
  {
  #ifdef REGCLASS_USE_REGISTRY
	  if (m_hCurrentKey)
	  {
		  RegCloseKey(m_hCurrentKey);
		  if (m_hRootKey)
			  RegCloseKey(m_hRootKey);
	  }
  #else
    closeKey();
  #endif
  }


  void MRegistry::setRootKey(const std::string &key) {
  #ifdef REGCLASS_USE_REGISTRY
    if (key == CURRENT_USER) {
  	  m_hRootKey = HKEY_CURRENT_USER;
    } else {
      assert(0);
    }
  #else
    if (key == CURRENT_USER) {
      m_RootKey = "CurrentUser";
    } else {
      assert(0);
    }
  #endif
  }

  void MRegistry::setKeyPrefix(char *Prefix) {
    m_KeyPrefix = Prefix;
  }

  void MRegistry::setKeySuffix(char *Suffix) {
    m_KeySuffix = Suffix;
  }


  // Returns 0 on failure, 1 on successful opening, 2 on successful open and create.
  int MRegistry::openKey(char *Key, REGSAM Access, bool AutoCreate) {
    std::string keyStr;

    keyStr = "";

    keyStr += m_KeyPrefix;
    keyStr += Key;
    keyStr += m_KeySuffix;


  #ifdef REGCLASS_USE_REGISTRY

	  if (!m_hRootKey)
		  return 0;

	  if (!Key)	
		  return 0;

    if (m_hCurrentKey) {
		  RegCloseKey(m_hCurrentKey);
    }
    m_CurrentKeyStr = keyStr;

	  int		Result;
	  DWORD	Disposition;

	  if (AutoCreate)		// Use createkey, since it will open it if it is already there
	  {
		  Result = RegCreateKeyEx(m_hRootKey, m_CurrentKeyStr.c_str(), 0, REG_NONE, REG_OPTION_NON_VOLATILE, Access, NULL, &m_hCurrentKey, &Disposition);
	  }
	  else
	  {
		  Result = RegOpenKeyEx(m_hRootKey, m_CurrentKeyStr.c_str(), 0, Access, &m_hCurrentKey);
		  Disposition = REG_OPENED_EXISTING_KEY;
	  }

    if (Result != ERROR_SUCCESS) {
		  return 0;
    }

    if (Disposition == REG_CREATED_NEW_KEY) {
		  return 2;
    }

	  return 1;

  #else

    closeKey();

    std::string filename;

    for (int i = 0; i < keyStr.length(); ++i) {
      if (keyStr[i] == '\\') {
        keyStr[i] = '-';
      }
    }

  #ifdef WIN32
    char buf[1024];
    filename = "%USERPROFILE%\\.aztecrc";
    ExpandEnvironmentStrings(filename.c_str(), buf, 1024);
    filename = buf;
  #else
    filename = getenv("HOME");
    filename += "/.aztecrc";
  #endif

    m_CurrentKeyStr = keyStr;

    iniFile.SetPath(filename);
    iniFile.ReadFile();
    changesMade = false;

    return 1;
  #endif

  }

  int MRegistry::closeKey() {
  #ifdef REGCLASS_USE_REGISTRY
	  if (!m_hCurrentKey)
		  return 0;

	  RegCloseKey(m_hCurrentKey);
	  m_hCurrentKey = NULL;
    m_CurrentKeyStr = "";

	  return 1;
  #else
    if (m_CurrentKeyStr.length() == 0) {
      return 0;
    }

    m_CurrentKeyStr = "";

    if (changesMade) {
      iniFile.WriteFile();
      changesMade = false;
    }
    iniFile.Reset();

    return 1;
  #endif
  }

  int MRegistry::writeString(const char *Name, const char *Data) {
  #ifdef REGCLASS_USE_REGISTRY

	  int Result;
	  
	  Result = RegSetValueEx(m_hCurrentKey, Name, 0, REG_SZ, (BYTE*)Data, strlen(Data)+1);
	  if (Result == ERROR_SUCCESS)
		  return 1;
	  return 0;

  #else

    changesMade = true;
    return iniFile.SetValue(m_CurrentKeyStr, Name, Data);

  #endif
  }

  int MRegistry::writeDWORD(const char *Name, DWORD Data) {
  #ifdef REGCLASS_USE_REGISTRY
	  int Result;
	  
	  Result = RegSetValueEx(m_hCurrentKey, Name, 0, REG_DWORD, (BYTE*)&Data, sizeof(DWORD));
	  if (Result == ERROR_SUCCESS)
		  return 1;
	  return 0;

  #else

    changesMade = true;
    return iniFile.SetValueI(m_CurrentKeyStr, Name, Data);

  #endif

  }

  int MRegistry::writeBinary(const char *Name, BYTE *Data, int Size) {
  #ifdef REGCLASS_USE_REGISTRY
	  int Result;
	  
	  Result = RegSetValueEx(m_hCurrentKey, Name, 0, REG_BINARY, (BYTE*)Data, Size);
	  if (Result == ERROR_SUCCESS)
		  return 1;
	  return 0;

  #else

    // TODO:: NOT Implemented for INI Files!!
    assert(0);
    return 0;

  #endif

  }

  int MRegistry::readString(const char *Name, char *Data) {
  #ifdef REGCLASS_USE_REGISTRY
	  int		Result;
	  DWORD	Size, Type;
	  BYTE	*Buffer;

	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, NULL, &Size);
	  if (Result != ERROR_SUCCESS)
		  return 0;

	  if (Type != REG_SZ)
		  return 0;

	  Buffer = new BYTE[Size];
	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, Buffer, &Size);

	  if (Result == ERROR_SUCCESS)
	  {
		  strcpy(Data, (char*)Buffer);
		  delete[] Buffer;
		  return 1;
	  }

	  delete[] Buffer;
	  return 0;

  #else

    if (!iniFile.ValueExists(m_CurrentKeyStr, Name)) {
      return 0;
    }

    CIniString value = iniFile.GetValue(m_CurrentKeyStr, Name);

    strcpy(Data, value.c_str());

    return 1;

  #endif
  }

  int MRegistry::readString(const char *Name, MStr &Data) {
  #ifdef REGCLASS_USE_REGISTRY
    char  Buf[1024];
  
    if (readString(Name, Buf) == 0)
      return 0;
  
    Data = Buf;
  
    return 1;
  #else

    if (!iniFile.ValueExists(m_CurrentKeyStr, Name)) {
      return 0;
    }

    CIniString value = iniFile.GetValue(m_CurrentKeyStr, Name);

    Data = value.c_str();

    return 1;

  #endif
  }

  MStr MRegistry::readString(const char *Name) {
  #ifdef REGCLASS_USE_REGISTRY
    char  Buf[1024];
  
    if (readString(Name, Buf) == 0)
      return MStr();
  
    return MStr(Buf);
  #else

    if (!iniFile.ValueExists(m_CurrentKeyStr, Name)) {
      return "";
    }

    CIniString value = iniFile.GetValue(m_CurrentKeyStr, Name);

    return value.c_str();

  #endif
  }

  int MRegistry::readDWORD(const char *Name, DWORD &Data) {
  #ifdef REGCLASS_USE_REGISTRY
	  int		Result;
	  DWORD	Size, Type;
	  BYTE	*Buffer;

	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, NULL, &Size);
	  if (Result != ERROR_SUCCESS)
		  return 0;

	  if (Type != REG_DWORD)
		  return 0;

	  Buffer = new BYTE[Size];
	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, Buffer, &Size);

	  if (Result == ERROR_SUCCESS) {
		  memcpy(&Data, Buffer, Size);
		  delete[] Buffer;
		  return 1;
	  }

	  delete[] Buffer;
	  return 0;
  #else

    if (!iniFile.ValueExists(m_CurrentKeyStr, Name)) {
      return 0;
    }

    Data = iniFile.GetValueI(m_CurrentKeyStr, Name);

    return 1;

  #endif
  }

  int MRegistry::readBinary(const char *Name, BYTE *Data, int MaxSize) {
  #ifdef REGCLASS_USE_REGISTRY
	  int		Result;
	  DWORD	Size, Type;
	  BYTE	*Buffer;

	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, NULL, &Size);
	  if (Result != ERROR_SUCCESS)
		  return 0;

	  if (Type != REG_BINARY)
		  return 0;

	  if (Size>(DWORD)MaxSize)
		  return 0;

	  Buffer = new BYTE[Size];
	  Result = RegQueryValueEx(m_hCurrentKey, Name, 0, &Type, Buffer, &Size);

	  if (Result == ERROR_SUCCESS) {
		  memcpy(Data, Buffer, Size);
		  delete[] Buffer;
		  return 1;
	  }

	  delete[] Buffer;
	  return 0;
  #else

    // TODO: Binary data not implemented for INI files.
    assert(0);
    return 0;

  #endif
  }

}