#include "StdAfx.h"

#include "MMaterial.h"
#include "MSystemManager.h"

#include <params/MParameterFactory.h>

#include <assert.h>
#include <stdlib.h>
#include <GL/gl.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  MMaterial::MMaterial() {
    unsetFlag(OBJECTFLAG_VISIBLE);
    setObjectType(OBJECTFLAG_TYPE_MATERIAL);
    m_DiffuseMap = NULL;
    
    // Set up the parameters
    addParameter(MParameterFactory::createColour("Amb", "AmbientColor", "Ambient Color"));
    addParameter(MParameterFactory::createColour("Diff", "DiffuseColor", "Diffuse Color"));
    addParameter(MParameterFactory::createColour("Spec", "SpecularColor", "Specular Color"));
    addParameter(MParameterFactory::createColour("Emm", "EmissionColor", "Emission Color"));
    addParameter(MParameterFactory::createFloat("Shin", "Shininess", "Shininess"));
    addParameter(MParameterFactory::createFloat("Opac", "Opacity", "Opacity"));
    addParameter(MParameterFactory::createImageFilename("DiffFile", "DiffuseMapName", "Diffuse Filename"));
    addParameter(MParameterFactory::createInteger("Cull", "CullMode", "Cull (f/b/n)"));
    
    setParamByName("Cull", "1");
    // Set up the key lists
    {
      setParamByName("Amb", "0.0 0.0 0.0");
      setParamByName("Diff", "0.8 0.8 0.8");
      setParamByName("Spec", "0.8 0.8 0.8");
      setParamByName("Emm", "0.0 0.0 0.0");
      setParamByName("Shin", "12.0");
      setParamByName("Opac", "1.0");
    }
    
  }
  
  MMaterial::~MMaterial() {
    m_DiffuseMap = NULL;
    
  }
  
  
  // Class related
  MBaseObjectPtr MMaterial::createNew() {
    MMaterial* Material;
    
    Material = new MMaterial;
    
    Material->setFrom(this);
    
    return Material;
  }
  
  void MMaterial::setFrom(MBaseObjectPtr SrcObj)
  {
    if (SrcObj == NULL) {
      return;
    }
    
    MNamedObject::setFrom(SrcObj);

    MNamedObjectPtr namedObj;
    MMaterialPtr Obj;
    
    namedObj = AZTEC_CAST(MNamedObject, SrcObj);
    Obj = AZTEC_CAST(MMaterial, SrcObj);
    
    if (Obj == NULL) {
      return;
    }
    
  }
  
  
  void MMaterial::clearImages() {
    setDiffuseMapImage(NULL);
  }
  
  void MMaterial::loadImagesFromMaps() {
    if (m_DiffuseMap == NULL || (m_DiffuseMapFile != getParamByName("DiffFile"))) {
      MSystemManagerPtr SysMan;
      MImageTranslatorPtr Trans;
      MImagePtr Image;
      
      SysMan = Aztec::getSystemManager();
      assert(SysMan != NULL);
      
      m_DiffuseMapFile = getParamByName("DiffFile");
      Image = SysMan->loadImage(m_DiffuseMapFile);
      
      if (Image != NULL) {
        setDiffuseMapImage(Image);
        SysMan->logOutput("Successfully loaded '%s'", (LPCTSTR)m_DiffuseMapFile);            
        m_DiffuseMapFileError = "";
      } else {
        if (!(m_DiffuseMapFileError == m_DiffuseMapFile))
          SysMan->logOutput("Translator could not load '%s'", (LPCTSTR)m_DiffuseMapFile);
        m_DiffuseMapFileError = m_DiffuseMapFile;
      }
      
      Trans = SysMan->getPluginManager()->getImageTranslatorThatImports(m_DiffuseMapFile);
    }
  }
  
  void MMaterial::setDiffuseMapFilename(const MStr &Filename) {
    // Check to see if the filename is different
    if (m_DiffuseMapFile.compareNoCase(Filename) != 0) {
      // Get rid of the old image. The new one will be loaded on demand.
      setDiffuseMapImage(NULL);
      
      m_DiffuseMapFile = Filename;
    }
  }
  
  void MMaterial::setDiffuseMapImage(MImagePtr Image) {
    m_DiffuseMap = Image;
  }
  
  
  void MMaterial::setOpenGLMaterialAndTextures() {
    // loadImagesFromMaps(); // Seems to appear below as well...
    
    float       MatInfo[4];
    MVector3    Vec;
    
    // Set the material parameters
    MatInfo[3] = 1.0f;
    Vec = getParamVec("Amb");
    memcpy(MatInfo, &Vec, sizeof(float)*3);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, MatInfo);
    
    
    MatInfo[3] = getParamFloat("Opacity");
    Vec = getParamVec("Diff");
    memcpy(MatInfo, &Vec, sizeof(float)*3);
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, MatInfo);
    
    MatInfo[3] = 1.0f;
    Vec = getParamVec("Spec");
    memcpy(MatInfo, &Vec, sizeof(float)*3);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, MatInfo);
    Vec = getParamVec("Emm");
    memcpy(MatInfo, &Vec, sizeof(float)*3);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, MatInfo);
    MatInfo[0] = getParamFloat("Shin");
    glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, MatInfo);
    
    // Set the GL_TEXTURE_2D information
    loadImagesFromMaps();
    if (m_DiffuseMap != NULL) {
      m_DiffuseMap->setGLImageAsCurrent(GL_TEXTURE_2D);
    } else {
      glDisable(GL_TEXTURE_2D);
    }
  }

}
