#include "StdAfx.h"

#include "scripting/ScriptScene.h"
#include "scripting/ScriptSceneObject.h"
#include "scripting/ScriptVectorParam.h"
#include "scripting/MScriptInterp.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptUtils.h"
#include "scripting/ScriptFuncs.h"
#include "scripting/MJavaScriptSupport.h"

#include "jsstddef.h"
#include "jscntxt.h"

#include "MAnimMesh.h"
#include "MMeshShape.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  

#if 0
// currently unused - intended for passing information about an Aztec
// function from where it is resolved to where it ends up getting called.
class AztecFuncInfo {
public:
	AztecFuncInfo(CScriptCallbacks::actionFunc *act, std::string &fn) {
		action = act;
		jnam = fn;
	}

	CScriptCallbacks::actionFunc *action;
	std::string jnam;
};

static void
	aztecAction_finalize(JSContext *cx, JSObject *obj)
{
	AztecFuncInfo *info = (AztecFuncInfo *)JS_GetPrivate(cx, obj);
	if (info != NULL)
		delete info;
}


JSClass aztecAction_class = {
	"AztecAction", JSCLASS_HAS_PRIVATE,
		JS_PropertyStub, JS_PropertyStub,
		JS_PropertyStub, JS_PropertyStub, 
		JS_EnumerateStub, JS_ResolveStub,
		JS_ConvertStub,   aztecAction_finalize
};
#endif

// Execute an action that we pulled from Aztec
JSBool JS_DLL_CALLBACK performAction(JSContext *cx, JSObject *obj, uintN argc,
	jsval *argv, jsval *rval)
{
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (interp == NULL) {
		*rval = JSVAL_VOID;
		return JS_FALSE;
	}

	// Hack - taken from netscape.public.mozilla.jseng newsgroup.  The
	// function object is at -2 on the argument stack.  By grabbing it,
	// turning it back into a JSFunction, and then querying the name,
	// we can figure out what needs to get called.
	JSObject *jobj =  JSVAL_TO_OBJECT(argv[-2]);
	JSFunction *fun = (JSFunction *)JS_GetPrivate(cx, jobj);
	if (fun == NULL) {
		*rval = JSVAL_VOID;
		return JS_FALSE;
	}

	// TBD: pass private data that has the action we want to call - that
	// way we only ask Aztec to look up the action name one time.
	const char *funcName = JS_GetFunctionName(fun);

	CScriptCallbacks::actionFuncWithArgs *actionArgs = interp->getActionWithArgs(funcName);

	if (actionArgs != NULL) {
		// construct the arguments from the jsvalues that we have
		std::vector<std::string> args;

		for (uintN i = 0; i < argc; ++i) {
			args.push_back(JS_GetStringBytes(JS_ValueToString(cx, argv[i])));
		}

		// now call our function to see what happens
		std::string result;
		int returnValue = (*actionArgs)(args, result);

		// if we have a textual return value, use that, otherwise just use the integer returned.
		if (result.length() > 0) {
			*rval = STRING_TO_JSVAL(JS_NewStringCopyZ(cx, result.c_str()));
		} else {
			*rval = INT_TO_JSVAL(returnValue);
		}
	} else {

		CScriptCallbacks::actionFunc *action = interp->GetAction(funcName);
		if (action == NULL) {
			JS_ReportError(cx, "Can't find action function in callback");
			return JS_FALSE;
		}

		(*action)();
  	*rval = JSVAL_VOID;
	}


	return JS_TRUE;
}

class AztecEnumSceneObjects : public std::vector< std::string > {
public:
	AztecEnumSceneObjects() {
		index = 0;
		numObjs = 0;
	}

	int index;
	int numObjs;
};

static AztecEnumSceneObjects *collectSceneNames()
{
	AztecEnumSceneObjects *objEnum = new AztecEnumSceneObjects;
	MBaseObjectPtr BaseObj;
	MNamedObjectPtr Obj;
	MBaseObjectTreePtr sceneObjs = getAllSceneObjects();

	if (sceneObjs == NULL) {
		// Empty list of scene objects.
		return NULL;
	}

	sceneObjs->beginIteration();
	while ((BaseObj = sceneObjs->getNext()) != NULL) {
		Obj = AZTEC_CAST(MNamedObject, BaseObj);
		if (Obj == NULL) {
			continue;
		}
		MStr objName = Obj->getName();
		objEnum->push_back(objName.c_str());
		objEnum->numObjs++;
	}
	sceneObjs->endIteration();

	return objEnum;
}

// See if a particular property exists in the Aztec object.  
static JSBool
aztecScene_lookupProperty(JSContext *cx, JSObject *obj, jsid id,
	JSObject **objp, JSProperty **propp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		// Failed to resolve the object name
		*objp = NULL;
		*propp = NULL;
		return JS_TRUE;
	}

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (interp == NULL) {
		*objp = NULL;
		*propp = NULL;
		return JS_TRUE;
	}

	if (JSVAL_IS_STRING(idval)) {
		char *objStr = JS_GetStringBytes(JSVAL_TO_STRING(idval));
		MBaseObjectPtr Obj = findNamedObject(objStr);
		*objp  = obj;
		if (Obj == NULL) {
			// No object by that name in the scene
			*propp = NULL;
		} else {
			// Found the object
			*propp = (JSProperty *)1;
		}
	} else if (JSVAL_IS_INT(idval)) {
		// Look for a property by index
		int i = JSVAL_TO_INT(idval);
		AztecEnumSceneObjects *sceneObjs = collectSceneNames();
		*objp = obj;
		if (i >= 0 && i < sceneObjs->numObjs) {
			*propp = (JSProperty *)1;
		} else {
			*propp = NULL;
		}
	} else {
		// Failed to resolve the property name
		*objp = NULL;
		*propp = NULL;
	}

	return JS_TRUE;
}

static JSBool
aztecScene_defineProperty(JSContext *cx, JSObject *obj, jsid id, jsval value,
	JSPropertyOp getter, JSPropertyOp setter,
	uintN attrs, JSProperty **propp)
{
	// Invoke the generic property definition within the JavaScript engine.
	// TBD: check to ensure this is a valid property/method name?
	JSBool result =
		js_ObjectOps.defineProperty(cx, obj, id, value, getter, setter, attrs, propp);
	return result;
}

static JSObject *
getScriptObject(JSContext *cx, JSObject *obj, MNamedObjectPtr Obj)
{
	// Get the generic scripting support wrapper.
	MScriptingSupportPtr supportPtr = Obj->getScriptObject();

	// Attempt to cast to Javascript support wrapper
	MJavaScriptSupport *jscriptPtr = dynamic_cast<MJavaScriptSupport *>(supportPtr.m_Ptr);

	// jobj will hold the Javascript unique object that reflects an Aztec objec.
	JSObject *jobj = NULL;

	// If the cast is unsuccessful, then drop down to the lowest level of support
	// (Reflection of a base MNamedObject).
	if (jscriptPtr == NULL) {
		MJavaScriptSupport jSupport;
		jobj = jSupport.getJavaScriptObject(cx, obj, Obj);
	} else {
		// If the cast is successful, then there is direct support for scripting.
		// Ask the object how it wants to reflect itself.
		jobj = jscriptPtr->getJavaScriptObject(cx, obj, Obj);
	}

	return jobj;
}

static JSBool
aztecScene_getPropertyById(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(idval)) {
		char *objStr = JS_GetStringBytes(JSVAL_TO_STRING(idval));
		MBaseObjectPtr BaseObj = findNamedObject(objStr);
		if (BaseObj == NULL) {
			// Not a named object - perhaps it is an action of some sort.
			MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
			if (interp == NULL) {
				JS_ReportError(cx, "Can't find script interpreter");
				return JS_FALSE;
			}

			CScriptCallbacks::actionFuncWithArgs* actionWithArgs = interp->getActionWithArgs(objStr);
			CScriptCallbacks::actionFunc *action = NULL;
			if (actionWithArgs == NULL) {
				action = interp->GetAction(objStr);
			}
			if (action == NULL && actionWithArgs == NULL) {
				// May be a built-in function.
				for (int m=0;aztecScene_methods[m].name != NULL;m++) {
					if (!strcmp(objStr, aztecScene_methods[m].name)) {
						// Yup - is a built-in function.
						JSFunction *jfunc =
							JS_NewFunction(cx, aztecScene_methods[m].call,
								aztecScene_methods[m].nargs,
								0, obj, objStr);
						JSObject *jfobj = JS_GetFunctionObject(jfunc);
						*vp = OBJECT_TO_JSVAL(jfobj);
						return JS_TRUE;
					}
				}


				JS_ReportError(cx, "%s is not defined", objStr);
				return JS_TRUE;
				// return JS_FALSE;
			}

			// Yup it is an action - now make a function that can get called
			// by JavaScript.
			JSFunction *jfunc = JS_NewFunction(cx, performAction, 0, 0, obj, objStr);
			JSObject *jfobj = JS_GetFunctionObject(jfunc);
			// AztecFuncInfo *info = new AztecFuncInfo(jfunc, std::string(objStr));
			// JSObject *jobj = JS_NewObject(cx, &aztecAction_class, NULL, NULL);
			// JS_SetPrivate(cx, jobj, info);

			*vp = OBJECT_TO_JSVAL(jfobj);
			return JS_TRUE;
		}
		MNamedObjectPtr Obj = AZTEC_CAST(MNamedObject, BaseObj);
		if (Obj == NULL) {
			JS_ReportError(cx, "%s is not a valid scene object", objStr);
			return JS_FALSE;
		}

#if 1
		// jobj will hold the Javascript unique object that reflects an Aztec objec.
		JSObject *jobj = getScriptObject(cx, obj, Obj);
#else
		// TBD: If a scene object has a more specialized representation for scripting
		// than a SceneObject, we need a way to find it.  
		JSObject *jobj = newSceneObject(cx, obj, Obj.m_Ptr);
#endif
		if (jobj == NULL) {
			JS_ReportError(cx, "unable to create reflection object for %s", objStr);
			return JS_FALSE;
		}
		*vp = OBJECT_TO_JSVAL(jobj);
		return JS_TRUE;
	} else if (JSVAL_IS_INT(idval)) {
		int index = JSVAL_TO_INT(idval);
		MBaseObjectTreePtr sceneObjs = getAllSceneObjects();

		if (sceneObjs == NULL) {
			// Empty list of scene objects.
			JS_ReportError(cx, "%s is not a valid scene object",
				JS_GetStringBytes(JSVAL_TO_STRING(idval)));
			return JS_FALSE;
		}

		MBaseObjectPtr BaseObj;
		MNamedObjectPtr Obj;
		int count = 0;
		sceneObjs->beginIteration();
		while ((count < index) && ((BaseObj = sceneObjs->getNext()) != NULL)) {
			Obj = AZTEC_CAST(MNamedObject, BaseObj);
			if (Obj == NULL) {
				continue;
			} else {
				count++;
			}
		}
		sceneObjs->endIteration();
		if (count != index) {
			JS_ReportError(cx, "%s is not a valid scene object",
				JS_GetStringBytes(JSVAL_TO_STRING(idval)));
			return JS_FALSE;
		}
		// Ask the object how it wants to reflect itself to scripting.  The default
		// for Scene objects is the SceneObject class.
		JSObject *jobj = getScriptObject(cx, obj, Obj);
		if (jobj == NULL) {
			JS_ReportError(cx, "unable to create reflection object for %s",
				JS_GetStringBytes(JSVAL_TO_STRING(idval)));
			return JS_FALSE;
		}
		*vp = OBJECT_TO_JSVAL(jobj);
		return JS_TRUE;
	}

	JS_ReportError(cx, "%s is not defined",
		JS_GetStringBytes(JSVAL_TO_STRING(idval)));
	return JS_FALSE;
}

// Currently not allowed to set properties in a scene.  Not sure
// what it would mean, perhaps creating a new object?
static JSBool
aztecScene_setPropertyById(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	return JS_FALSE;
}

static JSBool
aztecScene_getAttributes(JSContext *cx, JSObject *obj, jsid id,
	JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec Scene getAttributes\n");

	// We don't maintain JS property attributes for the Aztec Scene
	*attrsp = JSPROP_PERMANENT | JSPROP_READONLY | JSPROP_ENUMERATE;
	return JS_FALSE;
}

static JSBool
aztecScene_setAttributes(JSContext *cx, JSObject *obj, jsid id,
	JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec Scene setAttributes\n");

	// We don't maintain JS property attributes for the Aztec Scene
	if (*attrsp != (JSPROP_PERMANENT | JSPROP_READONLY | JSPROP_ENUMERATE)) {
		return JS_FALSE;
	}

	// Silently ignore all setAttribute attempts
	return JS_TRUE;
}

// Currently not allowed to delete properties from a scene.  Not sure
// what it would mean, perhaps delete the named object?
static JSBool
aztecScene_deleteProperty(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	OutputDebugString("Scene deleteProperty\n");

	return JS_FALSE;
}

static JSBool
aztecScene_defaultValue(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	OutputDebugString("Aztec Scene defaultValue\n");
	switch (typ) {
	case JSTYPE_OBJECT:
		*vp = OBJECT_TO_JSVAL(obj);
		return JS_TRUE;

	case JSTYPE_VOID:
	case JSTYPE_STRING:
	{
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Scene Object]");
		*vp = STRING_TO_JSVAL(str);
		return JS_TRUE;
	}

	case JSTYPE_FUNCTION:
	case JSTYPE_NUMBER:
	case JSTYPE_BOOLEAN:
	default:
		return JS_FALSE;
	}
}

static JSBool
	aztecScene_newEnumerate(JSContext *cx, JSObject *obj, JSIterateOp enum_op,
	jsval *statep, jsid *idp)
{
	switch(enum_op) {
	case JSENUMERATE_INIT:
	{
		// When iteration of properties begins, we collect the name of
		// all scene objects.  Each iteration after this will return
		// the name of each object in the order we collected them.
		MBaseObjectTreePtr sceneObjs = getAllSceneObjects();
		if (sceneObjs == NULL) {
			*statep = JSVAL_NULL;
			return JS_TRUE;
		}
		AztecEnumSceneObjects *objEnum = collectSceneNames();
		*statep = PRIVATE_TO_JSVAL(objEnum);
		if (idp != NULL) {
			*idp = INT_TO_JSVAL(objEnum->numObjs);
		}

		return JS_TRUE;
	}

	case JSENUMERATE_NEXT:
	{
		AztecEnumSceneObjects *objEnum =
			(AztecEnumSceneObjects *)JSVAL_TO_PRIVATE(*statep);
		if (objEnum->index < objEnum->numObjs) {
			JSString *str = JS_NewStringCopyZ(cx, (*objEnum)[objEnum->index].c_str());
			JS_ValueToId(cx, STRING_TO_JSVAL(str), idp);
			objEnum->index++;
			return JS_TRUE;
		}
		// Drop through to destroy
	}

	case JSENUMERATE_DESTROY:
	{
		AztecEnumSceneObjects *objEnum =
			(AztecEnumSceneObjects *)JSVAL_TO_PRIVATE(*statep);
		delete objEnum;
		*statep = JSVAL_NULL;
		return JS_TRUE;
	}

	default:
		OutputDebugString("Bad enumeration state\n");
		return JS_FALSE;
	}
}

static JSBool
aztecScene_checkAccess(JSContext *cx, JSObject *obj, jsid id,
	JSAccessMode mode, jsval *vp, uintN *attrsp)
{
	OutputDebugString("Aztec Scene object checkAccess\n");

	switch (mode) {
	case JSACC_WATCH:
		return JS_FALSE;

	case JSACC_IMPORT:
		return JS_FALSE;

	default:
		return JS_TRUE;
	}
}

// Specialized object operations - these allow us to dynamically reflect
// Aztec properties into the script interpreter.
JSObjectOps aztecScene_ops = {
	// Mandatory non-null function pointer members.
	js_ObjectOps.newObjectMap,
	js_ObjectOps.destroyObjectMap,
	aztecScene_lookupProperty,
	aztecScene_defineProperty,
	aztecScene_getPropertyById,  // getProperty
	aztecScene_setPropertyById,  // setProperty
	aztecScene_getAttributes,
	aztecScene_setAttributes,
	aztecScene_deleteProperty,
	aztecScene_defaultValue,
	aztecScene_newEnumerate,
	aztecScene_checkAccess,

	// Optionally non-null members start here.
	NULL,                       /* thisObject */
	NULL,                       /* dropProperty */
	NULL,                       /* call */
	NULL,                       /* construct */
	NULL,                       /* xdrObject */
	NULL,                       /* hasInstance */
	NULL,                       /* setProto */
	NULL,                       /* setParent */
	NULL,                       /* mark */
	NULL,                       /* clear */
	NULL,                       /* getRequiredSlot */
	NULL                        /* setRequiredSlot */
};

static JSObjectOps *
aztecScene_getObjectOps(JSContext *cx, JSClass *clazz)
{
	return &aztecScene_ops;
}

static JSBool
createObject(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MShapeObjectPtr shapeObj;

	// we can accept one or two parameters, the type followed by the name
	if (argc != 1 && argc != 2) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	JSString *str = JS_ValueToString(cx, argv[0]);
	if (!str) {
		return JS_FALSE;
	}

	const char *primName = JS_GetStringBytes(str);

	if (primName == NULL) {
		return JS_FALSE;
	}

	const char *newName = primName;

	if (argc == 2) {
		JSString *str = JS_ValueToString(cx, argv[1]);
		if (!str) {
			return JS_FALSE;
		}
		newName = JS_GetStringBytes(str);
	}

	MNamedObjectPtr newObject;
	JSObject *jobj = NULL;

	newObject = Scene->createObject(primName, newName);

	if (newObject != NULL) { 
		JSObject *jobj = getScriptObject(cx, obj, newObject);

		if (jobj == NULL) {
			JS_ReportError(cx, "unable to create reflection object for %s", 
				newObject->getName().c_str());
			return JS_FALSE;
		}
		*rval = OBJECT_TO_JSVAL(jobj);
		return JS_TRUE;
	}

	return JS_FALSE;
}

static JSBool
deleteObject(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MShapeObjectPtr shapeObj;

	// we can accept one or two parameters, the type followed by the name
	if (argc != 1 && argc != 2) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	// The first argument may be either a string (name of an object), or an
	// AztecSceneObject
	MNamedObject *AztecObj;
	if (JSVAL_IS_OBJECT(argv[0])) {
		AztecObj = (MNamedObject *)JS_GetInstancePrivate(cx, JSVAL_TO_OBJECT(argv[0]), &aztecSceneObject_class, NULL);
		if (AztecObj == NULL) {
			return JS_FALSE;
		}
	} else if (JSVAL_IS_STRING(argv[0])) {
		JSString *str = JS_ValueToString(cx, argv[0]);
		if (!str) {
			return JS_FALSE;
		}
		const char *primName = JS_GetStringBytes(str);
		if (primName == NULL) {
			return JS_FALSE;
		}
		AztecObj = AZTEC_CAST(MNamedObject, findNamedObject(primName));
		if (AztecObj == NULL) {
			return JS_FALSE;
		}
	} else {
		return JS_FALSE;
	}

	// At this point, we have a reference to the actual object in "AztecObj".
	// See if there is a second parameter that indicates whether children
	// of this object should be deleted.
	bool delChildren;
	if (argc == 2 && getBooleanVal(cx, *(argv+1), delChildren)) {
		int res = Scene->deleteObject(AztecObj, delChildren);
	} else {
		int res = Scene->deleteObject(AztecObj);
	}

	// We don't want to reuse this object - the pointer is invalid.  Therefore
	// clean out the private data that points to the thing we just deleted.
	if (JSVAL_IS_OBJECT(argv[0])) {
		JS_SetPrivate(cx, JSVAL_TO_OBJECT(argv[0]), NULL);
	}

	return JS_TRUE;
}

static JSBool
createMesh(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0 && argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	const char *primName;
	if (argc == 1) {
		JSString *str = JS_ValueToString(cx, argv[0]);
		if (!str) {
			return JS_FALSE;
		}
		primName = JS_GetStringBytes(str);
	} else {
		primName = "";
	}

	if (primName == NULL) {
		return JS_FALSE;
	}

	MSceneObjectPtr sceneObj = Scene->createMesh(primName);

	if (sceneObj == NULL) {
		return JS_FALSE;
	}

	JSObject *jobj = getScriptObject(cx, obj, sceneObj);
	if (jobj == NULL) {
		JS_ReportError(cx, "unable to create reflection object for %s",
			sceneObj->getName().c_str());
		return JS_FALSE;
	}
	*rval = OBJECT_TO_JSVAL(jobj);
	return JS_TRUE;
}

static JSBool
createLight(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0 && argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	const char *primName;
	if (argc == 1) {
		JSString *str = JS_ValueToString(cx, argv[0]);
		if (!str) {
			return JS_FALSE;
		}
		primName = JS_GetStringBytes(str);
	} else {
		primName = "";
	}

	if (primName == NULL) {
		return JS_FALSE;
	}

	MSceneObjectPtr sceneObj = Scene->createLight(primName);

	if (sceneObj == NULL) {
		return JS_FALSE;
	}

	JSObject *jobj = getScriptObject(cx, obj, sceneObj);
	if (jobj == NULL) {
		JS_ReportError(cx, "unable to create reflection object for %s",
			sceneObj->getName().c_str());
		return JS_FALSE;
	}
	*rval = OBJECT_TO_JSVAL(jobj);
	return JS_TRUE;
}

static JSBool
createSceneObject(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0 && argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	const char *primName;
	if (argc == 1) {
		JSString *str = JS_ValueToString(cx, argv[0]);
		if (!str) {
			return JS_FALSE;
		}
		primName = JS_GetStringBytes(str);
	} else {
		primName = "";
	}

	if (primName == NULL) {
		return JS_FALSE;
	}

	MSceneObjectPtr sceneObj = Scene->createSceneObject(primName);

	if (sceneObj == NULL) {
		return JS_FALSE;
	}

	JSObject *jobj = getScriptObject(cx, obj, sceneObj);
	if (jobj == NULL) {
		JS_ReportError(cx, "unable to create reflection object for %s",
			sceneObj->getName().c_str());
		return JS_FALSE;
	}
	*rval = OBJECT_TO_JSVAL(jobj);
	return JS_TRUE;
}

static JSBool
clearScene(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}


	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		// If no scene, then no objects in scene.
		return JS_FALSE;
	}

	Scene->clearScene();

	return JS_TRUE;
}

static JSBool
selectObject(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	uintN i;
	JSString *str;
	const char *objName;

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		// If no scene, then no objects in scene.
		return JS_FALSE;
	}

	for (i = 0; i < argc; i++) {
		if (JSVAL_IS_OBJECT(argv[i])) {
			// Get the name of the object
			JSObject *jobj = JSVAL_TO_OBJECT(argv[i]);
			if (jobj == NULL) {
				return JS_FALSE;
			}

			void *objPtr = JS_GetInstancePrivate(cx, jobj, &aztecSceneObject_class, NULL);
			if (objPtr == NULL) {
				return JS_FALSE;
			}

			MBaseObjectPtr Obj = (MBaseObject *)objPtr;
			Scene->selectObject(Obj);

		} else if (JSVAL_IS_STRING(argv[i])) {
			str = JS_ValueToString(cx, argv[i]);
			if (!str)
				return JS_FALSE;
			argv[i] = STRING_TO_JSVAL(str);
			objName = JS_GetStringBytes(str);

			MBaseObjectPtr Obj = findNamedObject(objName);

			Scene->selectObject(Obj);
			MNamedObjectPtr ObjParam = AZTEC_CAST(MNamedObject, Obj);
			if (ObjParam == NULL) {
				return JS_FALSE;
			}
			updateParamObject(ObjParam);
		} else {
			// Don't know what to do with non-string object names.
			return JS_FALSE;
		}
	}

	return JS_TRUE;
}

static JSBool
selectNone(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// TODO: This function should be removed entirely when Aztec2 becomes the main aztec, since Aztec2 provides a special selectNone function.
	// check to see if we have a custom selectNone function first.
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (interp == NULL) {
		*rval = JSVAL_VOID;
		return JS_FALSE;
	}
	CScriptCallbacks::actionFuncWithArgs *actionArgs = interp->getActionWithArgs("selectNone");

	if (actionArgs != NULL) {

		// now call our function to see what happens
		std::vector<std::string> args;
		std::string result;
		int returnValue = (*actionArgs)(args, result);

		// if we have a textual return value, use that, otherwise just use the integer returned.
		if (result.length() > 0) {
			*rval = STRING_TO_JSVAL(JS_NewStringCopyZ(cx, result.c_str()));
		} else {
			*rval = INT_TO_JSVAL(returnValue);
		}
		return JS_TRUE;
	} else {

		if (argc != 0) {
			return JS_FALSE;
		}

		MScenePtr Scene = getScenePointer();
		if (Scene == NULL) {
			return JS_FALSE;
		}

		Scene->selectNone();

		return JS_TRUE;
	}
}

static JSBool
getSelectedObjectList(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	MBaseObjectListPtr objList = Scene->getSelectedObjectList();
	jsint objCnt = objList->getCount();

	JSObject *jarrObj = JS_NewArrayObject(cx, 0, NULL);
	if (jarrObj == NULL) {
		return JS_FALSE;
	}

	if (JS_SetArrayLength(cx, jarrObj, objCnt) != JS_TRUE) {
		return JS_FALSE;
	}

	MBaseObjectPtr BaseObj;
	int offset = 0;
	objList->beginIteration();
	while ((BaseObj = objList->getNext()) != NULL) {
		MNamedObjectPtr Obj = AZTEC_CAST(MNamedObject, BaseObj);
		if (Obj == NULL) {
			continue;
		}

		JSObject *jobj = getScriptObject(cx, obj, Obj);
		if (jobj == NULL) {
			continue;
		}

		jsval jval = OBJECT_TO_JSVAL(jobj);

		JSBool defineOk = JS_DefineElement(cx, jarrObj, offset, jval, NULL, NULL, JSPROP_ENUMERATE);
		if (defineOk == JS_TRUE) {
			offset++;
		}
	}
	objList->endIteration();

	if (offset > 0) {
		*rval = OBJECT_TO_JSVAL(jarrObj);
		return JS_TRUE;
	} else {
		return JS_FALSE;
	}
}

static JSBool
getNumSelectedObjects(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->getNumSelectedObjects());
	return JS_TRUE;
}

static JSBool
getSelectionCenter(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	int maxCnt = -1;
	if (argc > 1) {
		return JS_FALSE;
	} else if (argc == 1) {
		if (!getIntegerVal(cx, *argv, maxCnt)) {
			return JS_FALSE;
		}
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	JSObject *vecobj = newVector3(cx);
	if (vecobj == NULL) {
		return JS_FALSE;
	}

	MVector3 vec = Scene->getSelectionCentre(maxCnt);

	setVector3Val(cx, vecobj, &vec);

	*rval = OBJECT_TO_JSVAL(vecobj);

	return JS_TRUE;
}

static JSBool
anythingSelected(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	*rval = (Scene->anythingSelected() ? JSVAL_TRUE : JSVAL_FALSE);
	return JS_TRUE;
}

static JSBool
redraw(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (interp == NULL) {
		return JS_FALSE;
	}

	interp->Redraw();

	return JS_TRUE;
}

static JSBool
setTime(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	Scene->setTime(t);

	return JS_TRUE;
}

static JSBool
setStartTime(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	Scene->setStartTime(t);

	return JS_TRUE;
}

static JSBool
getStartTime(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->getStartTime());
	return JS_TRUE;
}

static JSBool
setEndTime(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	Scene->setEndTime(t);

	return JS_TRUE;
}

static JSBool
getEndTime(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->getEndTime());
	return JS_TRUE;
}

static JSBool
setTimeRange(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 2) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t0, t1;
	if (!getIntegerVal(cx, *argv, t0) || !getIntegerVal(cx, *(argv+1), t1)) {
		return JS_FALSE;
	}

	Scene->setTimeRange(t0, t1);

	return JS_TRUE;
}

static JSBool
getFramesPerSecond(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 0) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->getFramesPerSecond());
	return JS_TRUE;
}

static JSBool
setFramesPerSecond(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	Scene->setFramesPerSecond(t);

	return JS_TRUE;
}

static JSBool
tickToFrame(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	return JS_NewDoubleValue(cx, Scene->tickToFrame(t), rval);
}

static JSBool
frameToTick(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	double t;
	if (!getDoubleVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->frameToTick(t));
	return JS_TRUE;
}

static JSBool
tickToSeconds(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	int t;
	if (!getIntegerVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	return JS_NewDoubleValue(cx, Scene->tickToSeconds(t), rval);
}


static JSBool
secondsToTick(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc != 1) {
		return JS_FALSE;
	}

	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		return JS_FALSE;
	}

	double t;
	if (!getDoubleVal(cx, *argv, t)) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(Scene->SecondsToTick(t));
	return JS_TRUE;
}

static void
aztecScene_finalize(JSContext *cx, JSObject *obj)
{
	OutputDebugString("Aztec Scene finalize\n");
}

JSFunctionSpec aztecScene_methods[] = {
	// Object/mesh creation
	{"createObject",		createObject,			1},
	{"createSceneObject",	createSceneObject,		0},
	{"createMesh",			createMesh,				0},
	{"createLight",			createLight,			0},
	{"deleteObject",		deleteObject,			1},
	{"clearScene",			clearScene,				0},

	// Selection functions
	{"getSelected",				getSelectedObjectList,	0},
	{"getSelectedObjectList",	getSelectedObjectList,	0},
	{"redraw",					redraw,					0},
	{"selectNone",				selectNone,				0},
	{"selectObject",			selectObject,			1},
	{"getNumSelectedObjects",	getNumSelectedObjects,	0},
	{"getSelectionCenter",		getSelectionCenter,		0},
	{"anythingSelected",		anythingSelected,		0},

	// Time related functions
	{"setTime",				setTime,				1},
	{"setStartTime",		setStartTime,			1},
	{"getStartTime",		getStartTime,			0},
	{"setEndTime",			setEndTime,				1},
	{"getEndTime",			getEndTime,				0},
	{"setTimeRange",		setTimeRange,			2},
	{"getFramesPerSecond",	getFramesPerSecond,		0},
	{"setFramesPerSecond",	setFramesPerSecond,		1},
	{"tickToFrame",			tickToFrame,			1},
	{"frameToTick",			frameToTick,			1},
	{"tickToSeconds",		tickToSeconds,			1},
	{"secondsToTick",		secondsToTick,			1},

	{ NULL,					NULL,					0}
};

JSClass aztecScene_class = {
	"AztecScene", JSCLASS_HAS_PRIVATE,
		JS_PropertyStub, JS_PropertyStub,
		JS_PropertyStub, JS_PropertyStub, 
		JS_EnumerateStub, JS_ResolveStub,
		JS_ConvertStub,   aztecScene_finalize,
		aztecScene_getObjectOps
};

// Add a class that reflects the Aztec scene back to the property "Scene"
// that lives in the global namespace
void addSceneClass(JSContext *cx)
{
	JSObject *jobj = JS_DefineObject(cx, JS_GetGlobalObject(cx), "Scene",
		&aztecScene_class, 0, JSPROP_ENUMERATE);
	JS_DefineFunctions(cx, jobj, aztecScene_methods);
	// JS_AddNamedRoot(cx, &jobj, "Aztec Scene");
}

}
