#include <StdAfx.h>

#include "scripting/ScriptUtils.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptSceneObject.h"
#include "scripting/ScriptAggregateParam.h"
#include "scripting/ScriptArrayParam.h"
#include "scripting/ScriptVectorParam.h"

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

MScenePtr getScenePointer()
{
	MSystemManagerPtr SysMan = Aztec::getSystemManager();

	if (SysMan == NULL) {
		// If no system manager, then no chance of finding the object
		return NULL;
	}

	return SysMan->getScene();
}

MBaseObjectTreePtr getAllSceneObjects()
{
	MScenePtr Scene = getScenePointer();
	if (Scene == NULL) {
		// If no scene, then no objects in scene.
		return NULL;
	}

	return Scene->getObjectList();
}

MBaseObjectPtr findNamedObject(const char *name)
{
	MBaseObjectTreePtr Objs = getAllSceneObjects();
	if (Objs == NULL) {
		// No objects in scene
		return NULL;
	}

	MBaseObjectPtr Obj = Objs->findObject(name);

	return Obj;
}

void
updateParamObject(const MNamedObjectPtr &obj)
{
	obj->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
	obj->updateObject();
}

// Convert a parameter object into a JavaScript value.  Primitive types (e.g.,
// int, bool, float) are translated into the corresponding JavaScript types.
// More complex types (e.g., Parameter objects) are turned into a JavaScript
// class that reflects it's basic methods/properties.
jsval convertParam(JSContext *cx, 
                   JSObject *obj,
                   const MNamedObjectPtr &aztec_obj, 
                   const MParameterObjectPtr &aztec_param)
{
	if (aztec_param == NULL) {
		return INT_TO_JSVAL(0);
	}

	int dataType = aztec_param.m_Ptr->getDataType();

	if (dataType == MParameterObject::TYPE_INTEGER) {
		int val;
		aztec_param->getValueInteger(val);
		return INT_TO_JSVAL(val);
	} else if (dataType == MParameterObject::TYPE_FLOAT) {
		float val;
		aztec_param->getValueFloat(val);
		return DOUBLE_TO_JSVAL(JS_NewDouble(cx, val));

	} else if (dataType == MParameterObject::TYPE_BOOLEAN) {
		bool val;
		aztec_param->getValueBoolean(val);
		return BOOLEAN_TO_JSVAL(val ? JS_TRUE : JS_FALSE);

	} else if (dataType == MParameterObject::TYPE_STRING) {
		MStr val;
		aztec_param->getValueString(val);
		JSString *str = JS_NewStringCopyZ(cx, val.c_str());
		return STRING_TO_JSVAL(str);

	} else if (dataType == MParameterObject::TYPE_VECTOR) {
		// Build a variable that reflects this vector to/from JavaScript

/*    JSObject *jsvec = newVector3(cx);
		MVector3 avec;
		aztec_param->getValueVector(avec);
		setVector3Val(cx, jsvec, &avec);
*/

    JSObject *jsvec = newVectorParam(cx, aztec_obj, aztec_param);

		// JSObject *jsvec = newVectorParam(cx, aztec_obj, aztec_param.m_Ptr);
		return OBJECT_TO_JSVAL(jsvec);

	} else if (dataType == MParameterObject::TYPE_MATRIX) {
		// TBD: create a matrix object and pass it back
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Matrix]");
  		return STRING_TO_JSVAL(str);

	} else if (dataType == MParameterObject::TYPE_OBJECT) {
		MObjectParameterPtr ObjParam = AZTEC_CAST(MObjectParameter, aztec_param);
		if (ObjParam == NULL) {
			return false;
		}
		// Now have a pointer to an object parameter.  Pull a reference to the
		// underlying named object.
		MNamedObjectPtr Obj = ObjParam->getValue();
		if (Obj == NULL) {
			return false;
		}

		// Got the object pointed to by the parameter.  Now build a variable that
		// reflects into/out of JavaScript.
		JSObject *jsobj = getScriptObject(cx, obj, Obj.m_Ptr);

		return OBJECT_TO_JSVAL(jsobj);

	} else if (dataType == MParameterObject::TYPE_IMAGE) {
		// TBD: create some sort of object to hold an image
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Image]");
		return STRING_TO_JSVAL(str);

	} else if (dataType == MParameterObject::TYPE_ARRAY) {
		// Build a variable that reflects this vector to/from JavaScript
    MArrayParameterPtr array = AZTEC_CAST(MArrayParameter, aztec_param);

    if (array != NULL) {
		  JSObject *jsobject = newArrayParam(cx, aztec_obj, array);

      return OBJECT_TO_JSVAL(jsobject);
    }

    // if we have failed, just return NULL.
    return NULL;
  } else if (dataType == MParameterObject::TYPE_AGGREGATE) {

		// Build a variable that reflects this vector to/from JavaScript
    MAggregateParameterPtr aggregate = AZTEC_CAST(MAggregateParameter, aztec_param);

    // if we have an aggregate type, then converting to the MAggregate 
    // class should never ever fail.
    assert(aggregate != NULL);

    JSObject *jsobject = newAggregateParam(cx, aztec_obj, aggregate);

    return OBJECT_TO_JSVAL(jsobject);

  } else {
		// Not a type that is understood
		return PRIVATE_TO_JSVAL(aztec_param.m_Ptr);
	}
}


jsval convertParam(JSContext *cx, 
                   JSObject *obj,
                   const MNamedObjectPtr &aztec_obj, 
                   const MArrayParameterPtr &aztec_param,
                   int elementIndex)
{
	int dataType = aztec_param->getElementType();

	if (dataType == MParameterObject::TYPE_INTEGER) {
    int val;
    aztec_param->getElement(elementIndex, val);
    return INT_TO_JSVAL(val);
	} else if (dataType == MParameterObject::TYPE_FLOAT) {
		float val;
		aztec_param->getElement(elementIndex, val);
		return DOUBLE_TO_JSVAL(JS_NewDouble(cx, val));

	} else if (dataType == MParameterObject::TYPE_BOOLEAN) {
		bool val;
		aztec_param->getElement(elementIndex, val);
		return BOOLEAN_TO_JSVAL(val ? JS_TRUE : JS_FALSE);

	} else if (dataType == MParameterObject::TYPE_STRING) {
		MStr val;
		aztec_param->getElement(elementIndex, val);
		JSString *str = JS_NewStringCopyZ(cx, val.c_str());
		return STRING_TO_JSVAL(str);

	} else if (dataType == MParameterObject::TYPE_MATRIX) {
		// TBD: create a matrix object and pass it back
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Matrix]");
  		return STRING_TO_JSVAL(str);

	} else if (dataType == MParameterObject::TYPE_IMAGE) {
		// TBD: create some sort of object to hold an image
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Image]");
		return STRING_TO_JSVAL(str);

	} else {

    return convertParam(cx, obj, aztec_obj, aztec_param->getElement(elementIndex));

  } 
}


}
