#include <AztecMainPCH.h>

#include "resource.h"

//#include "MenuItem.h"
//#include "Keyboard.h"
//#include "MdlConst.h"

#include "DlgGlobs.h"   // Global dialog variables.
#include "MdlGlobs.h"
//#include "MdlMsgs.h"

//#include "KeyFuncGeneral.h"
#include "KeyFuncMain.h"

#include <views/RenderView.h>

#include <IniFile.h>
#include <params/MParameterFactory.h>
#include <misc/MSceneHelper.h>

#include <views/FormView.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif



#ifdef WIN32
  
DWORD startTime;

#else
#include <time.h>

time_t startTime;

#endif

/// This tells the render timer to start timing.
void startRenderTimer() {
#ifdef WIN32
  startTime = GetTickCount();
#else
  time(&startTime);
#endif
}

/// This stops the timing, and returns the number of seconds elapsed since the start.
float stopRenderTimer() {
#ifdef WIN32
  DWORD endTime = GetTickCount();
  return (float)(endTime - startTime) / 1000.0;
#else
  time_t endTime;
  time(&endTime);

  return difftime(endTime, startTime);
#endif
}


class RenderListener : public Aztec::MSceneRenderer::UpdateListener {
public:
  void onUpdate(const MSceneRendererPtr &renderer, 
                float amountCompleted, 
                const MImagePtr &image) 
  {
    // find a image view, and set the image.
    MBaseViewWndPtr view = g_ViewList.FindViewOfType("RenderView");

    RenderView *renderView = AZTEC_CAST(RenderView, view);

    if (renderView != NULL) {
      renderView->getImageComponent()->setImage(renderer->getImage());
      renderView->getImageComponent()->Invalidate();
    }

    if (amountCompleted >= 1.0) {
      MSystemManager::getInstance()->logOutput("Rendering Completed in %.1f seconds.", stopRenderTimer());
    }
  }
};

void readInSettings(CIniFile &ini, const std::string &iniGroup, const MParameterObjectListPtr &list) {
  ini.startSectionIteration(iniGroup);

  CIniString value, valueName;

  while (ini.getNextSectionValue(valueName, value)) {
    MParameterObjectPtr param = list->getParameter(valueName.c_str());
    if (param != NULL) {
      param->setValueString(value.c_str());
    }
  }
}

void writeOutSettings(CIniFile &ini, const std::string &iniGroup, const MParameterObjectListPtr &list) {
  for (int i = 0; i < list->getNumParams(); ++i) {
    MStr value;
    if (list->getParameter(i)->getValueString(value)) {
      ini.SetValue(iniGroup, list->getParameter(i)->getLongName().c_str(), value.c_str()); 
    }
  }
}

MParameterObjectListPtr getDefaultGeneralOptions() {
  MParameterObjectListPtr generalOptions = new MParameterObjectList(NULL);

  // Add in our standard parameters.
  Aztec::MChoiceParameterPtr rendererChoice = Aztec::MParameterFactory::createChoice("renderer", "renderer", "Current Renderer");

  MPluginManagerPtr plugMan = MSystemManager::getInstance()->getPluginManager();
  for (int i = 0; i < plugMan->getRendererCount(); ++i) {
    rendererChoice->addChoice(plugMan->getRenderer(i)->getName().c_str());
  }
  rendererChoice->setValueInteger(0);

  generalOptions->addParam(rendererChoice);
  generalOptions->addParam( MParameterFactory::createInteger("outputWidth", "outputWidth", "outputWidth") );
  generalOptions->addParam( MParameterFactory::createInteger("outputHeight", "outputHeight", "outputHeight") );
  generalOptions->addParam( MParameterFactory::createBoolean("antialiasing", "antialiasing", "antialiasing") );

  generalOptions->getParameter("outputWidth")->setValueInteger(320);
  generalOptions->getParameter("outputHeight")->setValueInteger(240);
  generalOptions->getParameter("antialiasing")->setValueBoolean(true);

  generalOptions->addParam( MParameterFactory::createObject("camera", "camera", "Camera to render") );

  return generalOptions;
}

void readInRenderSettings(const char *optionCategory,
                          const MParameterObjectListPtr &options)
{
  if (options != NULL) {
    CIniFile settings(MSystemManager::getInstance()->getUserSettingsPath() + "/LastRenderOptions.ini");
  
    settings.ReadFile();

    readInSettings(settings, optionCategory, options);

    // now write out the settings to make sure we store any defaults that 
    // weren't stored in the file.
    writeOutSettings(settings, optionCategory, options);

    // save the actual file
    settings.WriteFile();
  }
}

void writeOutRenderSettings(const char *optionCategory,
                            const MParameterObjectListPtr &options)
{
  if (options != NULL) {
    CIniFile settings(MSystemManager::getInstance()->getUserSettingsPath() + "/LastRenderOptions.ini");
  
    settings.ReadFile();

    // now write out the settings
    writeOutSettings(settings, optionCategory, options);

    // save the actual file
    settings.WriteFile();
  }
}

int doRender(MSceneObjectPtr camera) {

  MParameterObjectListPtr generalOptions = getDefaultGeneralOptions();

  readInRenderSettings("general", generalOptions);

  // now place our current camera in the settings
  MSceneHelper<MCamera>::SceneShapeVector cameras = MSceneHelper<MCamera>::findObjectsOfType(g_Scene);

  if (camera == NULL) {
    MStr cameraName;
    generalOptions->getParameter("camera")->getValueString(cameraName);
    camera = AZTEC_CAST(MSceneObject, g_Scene->getObjectList()->findObject(cameraName));
  }

  if (camera == NULL && cameras.size() > 0) {
    camera = cameras[0].first;
  }

  if (camera != NULL) {
    generalOptions->getParameter("camera")->setValueString(camera->getName().c_str());
  }

  // save our settings
  writeOutRenderSettings("general", generalOptions);


  MStr currentRendererStr;
  if (!generalOptions->getParameter("renderer")->getValueString(currentRendererStr)) {
    // If we couldn't retrieve the value, try to be a bit more graceful about it.
    currentRendererStr = "PovRay 3.5 Ray Tracer";

    MSystemManager::getInstance()->logOutput(
      "Warning: Could not retrieve renderer from the rendering defaults file, using '%s' instead.", 
      currentRendererStr.c_str());
  }

  // Now try to retrieve the renderer plugin
  MSceneRendererPtr renderer = MSystemManager::getInstance()->getPluginManager()->getRenderer(currentRendererStr.c_str());

  // if we don't have a valid renderer, report an error and bail out.
  if (renderer == NULL) {
    MSystemManager::getInstance()->logOutput("Error: Could not find a renderer matching '%s', aborting render", currentRendererStr.c_str());
    return 0;
  }

  MParameterObjectListPtr pluginOptions = renderer->getOptions();

  // If we do not have a set of options, make them so we can set some values.
  if (pluginOptions != NULL) {
    // Now retrieve the plugin values from the file if we can, and write out any new defaults.
    readInRenderSettings(currentRendererStr, pluginOptions);

    // now combine the two lists together.
    for (int i = 0; i < pluginOptions->getNumParams(); ++i) {
      generalOptions->addParam(pluginOptions->getParameter(i));
    }
  }

  // Set the render window as the current window.
  KWindowRenderView();

  renderer->initialise();
  MSystemManager::getInstance()->logOutput("Rendering Started...");
  startRenderTimer();
  renderer->start(g_Scene, generalOptions, new RenderListener());

  return 1;
}

int doRender() {
  return doRender(NULL);
}

#include <controls/MParameterGroup.h>

int createViewWindow(MBaseViewWndPtr view, const char *className, const char *viewName, bool eraseBackground);

class ParameterGroupView : public MFormView {
public:
  ParameterGroupView() {
  }

  ~ParameterGroupView() {
    for (int i = 0; i < groups.size(); ++i) {
      if (groups[i] != NULL) {
        groups[i]->DestroyWindow();
        delete groups[i];
        groups[i] = NULL;
      }
    }
    groups.clear();
  }

  AztecGUI::MParameterGroup* addGroup(const char *groupName) {
    AztecGUI::MParameterGroup *p = new AztecGUI::MParameterGroup(groupName);
    
    p->Create(NULL, "", WS_CHILD | WS_VISIBLE, CRect(0,0,200,200), getContentPane(), -1);

    groups.push_back(p);
    return p;
  }

  int getGroupCount() {
    return groups.size();
  }

  AztecGUI::MParameterGroup* getGroup(int index) {
    return groups[index];
  }

  // MFormView methods
  void onFormAction(const std::string &action) {
    if (action == "Ok" || action == "Apply") {
      for (GroupList::iterator it = groups.begin(); it != groups.end(); ++it) {
        (*it)->setParameterValues();
      }
    }

    MFormView::onFormAction(action);
  }

  // MBaseViewWnd methods
  void ViewCreate() {
    MFormView::ViewCreate();

    addButton("Do Render");
    addButton("Ok");
    addButton("Cancel");
    addButton("Apply");
  }

	//{{AFX_MSG(ImageViewerView)
  afx_msg void OnClose();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()


private:
  typedef std::vector<AztecGUI::MParameterGroup*> GroupList;
  GroupList groups;

};

void ParameterGroupView::OnClose() {
  g_ViewList.DeleteView(this);
}




BEGIN_MESSAGE_MAP(ParameterGroupView, MFormView)
	//{{AFX_MSG_MAP(ImageViewerView)
	ON_WM_CLOSE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


class OptionListener : public MFormView::ActionListener {
public:
  OptionListener() {
  }

  void addGroup(const char *group, const MParameterObjectListPtr &options) {
    if (options != NULL) {
      items.push_back(Item(group, options));
    }
  }

  // ActionListener methods
  void onAction(MFormView *form, const std::string &action) {
    if (action == "Ok" || action == "Apply") {
      for (int i = 0; i < items.size(); ++i) {
        writeOutRenderSettings(items[i].first.c_str(), items[i].second);        
      }
    }

    if (action == "Do Render") {
      doRender();
    }
  }

private:
  typedef std::pair<std::string, MParameterObjectListPtr> Item;
  typedef std::vector<Item> ItemList;

  ItemList items;
};

int showRenderOptions() {
  ParameterGroupView *groupView = new ParameterGroupView;
  MBaseViewWndPtr view = groupView;
  createViewWindow(view, "FormView", "Oh yeah baby", true);

  MParameterObjectListPtr generalOptions = getDefaultGeneralOptions();
  readInRenderSettings("general", generalOptions);

  OptionListener* optionListener = new OptionListener();
  optionListener->addGroup("general", generalOptions);


  AztecGUI::MParameterGroup *p = groupView->addGroup("Options");
  for (int i = 0; i < generalOptions->getNumParams(); ++i) {
    p->addParameter(generalOptions->getParameter(i));
  }

  MPluginManagerPtr plugMan = MSystemManager::getInstance()->getPluginManager();
  for (int i = 0; i < plugMan->getRendererCount(); ++i) {
    MSceneRendererPtr renderer = plugMan->getRenderer(i);
    MParameterObjectListPtr renderOptions = renderer->getOptions();
    readInRenderSettings(renderer->getName().c_str(), renderOptions);

    optionListener->addGroup(renderer->getName().c_str(), renderOptions);

    AztecGUI::MParameterGroup *p = groupView->addGroup(renderer->getName().c_str());
    for (int i = 0; i < renderOptions->getNumParams(); ++i) {
      p->addParameter(renderOptions->getParameter(i));
    }
  
  }

  groupView->addActionListener(optionListener);
  groupView->Invalidate();
  groupView->doLayout();
  view->ShowWindow(SW_SHOW);

  return 0;
}

void InitRenderActionList(MActionListType *AL) {
  g_SysMan->logOutput("Initialising Render Action List");
  

  AL->Add("doRender", "Does a render", "Rendering", doRender);
  AL->Add("showRenderOptions", "Show the Rendering Options", "Rendering", showRenderOptions, g_MainDlg, ID_RENDERING_RENDERINGOPTIONS);
}




	