//---------------------------------------------------------------------------
#include <AztecMainPCH.h>

#include "MenuItem.h"
#include "Keyboard.h"
#include "MDLGlobs.h"

#include <ctype.h>
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <algorithm>

#include <assert.h>

#include "MTextFileStream.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//----------------------------------------------------------------------------------------
//  MKeystroke
//----------------------------------------------------------------------------------------

using namespace std;

MKeystroke::MKeystroke() {
  m_Ctrl = m_Alt = m_Shift = false;
  m_Key = 0;
  m_Act = NULL;
}

MKeystroke::MKeystroke(const MKeystroke &src) {
  m_Ctrl = src.m_Ctrl;
  m_Alt = src.m_Alt;
  m_Shift = src.m_Shift;
  m_Act = src.m_Act;
  m_Erase = src.m_Erase;
  m_Key = src.m_Key;
}

MKeystroke::MKeystroke(MAction *Ac, WORD K, bool C, bool A, bool S) {
  m_Key = K;
  m_Ctrl = C;
  m_Alt = A;
  m_Shift = S;
  m_Act = Ac;

  if (m_Act) {
    m_Category = m_Act->m_Cat;
  }
}

MKeystroke::MKeystroke(const MStr &category, WORD K, bool C, bool A, bool S) {
  m_Key = K;
  m_Ctrl = C;
  m_Alt = A;
  m_Shift = S;
  m_Act = NULL;
  m_Category = category;
}

MKeystroke::~MKeystroke() {
}

int MKeystroke::ChangeMenuSC() {
  if (m_Erase.Valid()) {
    m_Erase.SetShortcut("");
    m_Erase.SetItem(0,0);
  }
  
  if (!m_Act) {
    return 0;
  }
  
  if (!m_Act->m_Menu.Valid()) {
    return 0 ;
  }
  
  m_Act->m_Menu.SetShortcut((LPCTSTR)GetKeystrokeString());
  
  return 1;
}

MAction* MKeystroke::getAction() {
  return m_Act;
}

void MKeystroke::setAction(MAction *act) {
  m_Act = act;
  if (m_Act) {
    m_Category = m_Act->m_Cat;
  } else {
    m_Category = "";
  }

}

MStr MKeystroke::GetKeystrokeString()
{
  MStr  SC;
  
  if (m_Alt)
    SC += "Alt+";
  if (m_Shift)
    SC += "Shift+";
  if (m_Ctrl)
    SC += "Ctrl+";
  
  SC += VKeyToString(m_Key);
  
  return SC;
}

bool MKeystroke::operator<(const MKeystroke &rhs) {
  if (m_Key < rhs.m_Key) return true;
  if (m_Ctrl == true && rhs.m_Ctrl == false) return true;
  if (m_Alt == true && rhs.m_Alt == false) return true;
  if (m_Shift == true && rhs.m_Shift == false) return true;
  if (m_Category < rhs.m_Category) return true;

  return false;
}

bool MKeystroke::operator==(const MKeystroke &rhs) {
  if (m_Key == rhs.m_Key &&
      m_Ctrl == rhs.m_Ctrl &&
      m_Alt == rhs.m_Alt &&
      m_Shift == rhs.m_Shift) {

    // use a blank string as a don't care match.
    if (m_Category.GetLength() == 0 || rhs.m_Category.GetLength() == 0) {
      return true;
    }
    // otherwise we need an identical match.
    return m_Category.compareNoCase(rhs.m_Category) == 0;

  }

  return false;
}



//----------------------------------------------------------------------------------------
//  MKeyActionMap
//----------------------------------------------------------------------------------------

MKeyActionMap::MKeyActionMap() {
  m_Actions = NULL;
}

MKeyActionMap::~MKeyActionMap() {
  RemoveAll();
}

int MKeyActionMap::Add(const MStr &ActionName, const MStr &Cat, WORD K, bool C, bool A, bool S) {
  MKeystroke   key(FindAction(ActionName,Cat), K, C, A, S);

  m_Keys.push_back( key );
  
  key.ChangeMenuSC();
  
  return 1;
}

int MKeyActionMap::Remove() {
  return Remove(numMappings() - 1);
}

int MKeyActionMap::Remove(int index) {
  vector<MKeystroke>::iterator  i;
  int count;

  count = 0;
  for (i = m_Keys.begin(); i != m_Keys.end(); ++i) {
    if (count == index) {
      m_Keys.erase(i);
      return 1;
    }
   
  }
  
  return 0;
}

int MKeyActionMap::Remove(const MKeystroke &key) {
  vector<MKeystroke>::iterator  i;

  i = find(m_Keys.begin(), m_Keys.end(), key);

  if (i != m_Keys.end()) {
    m_Keys.erase(i);
    return 1;
  }
  return 0;
}

void MKeyActionMap::RemoveAll() {
  m_Keys.clear();
}


int MKeyActionMap::Set(int n, const MStr &ActionName, const MStr &Cat, WORD K, bool C, bool A, bool S) {
  MKeystroke   *Key;
  
  Key = (*this)[n];
  
  if (!Key) {
    return 0;
  }
  
  Key->setAction(FindAction(ActionName,Cat));
  Key->m_Ctrl = C;
  Key->m_Alt = A;
  Key->m_Shift = S;
  
  Key->ChangeMenuSC();
  
  return 1;
}

int MKeyActionMap::Change(int n, const MStr &ActionName, const MStr &Cat, WORD K, bool C, bool A, bool S) {
  MKeystroke   *Key;
  
  Key = (*this)[n];
  
  if (!Key)
    return 0;
  
  Key->setAction(FindAction(ActionName,Cat));
  Key->m_Ctrl = C;
  Key->m_Alt = A;
  Key->m_Shift = S;
  
  //   Key->ChangeMenuSC();
  
  return 1;
}

void MKeyActionMap::UpdateAllMenus() {
  vector<MKeystroke>::iterator  i;

  for (i = m_Keys.begin(); i != m_Keys.end(); ++i) {
    i->ChangeMenuSC();
  }
}

int MKeyActionMap::SortAlpha() {
  
  return 0;
}

MKeystroke* MKeyActionMap::FindKey(WORD K, const MStr &Cat, bool C, bool A, bool S) {
  vector<MKeystroke>::iterator  i;
  MKeystroke  key(Cat, K, C, A, S);

  i = find(m_Keys.begin(), m_Keys.end(), key);

  if (i != m_Keys.end()) {
    return &(*i);
  }

  return NULL;
}

MAction* MKeyActionMap::FindAction(HMENU hMenu, DWORD nID, const MStr &Cat) {
  MAction *Act;
  
  if (!m_Actions->m_First) {
    return NULL;
  }
  
  Act = m_Actions->m_First;
  while (Act)  {
    
    if (Act->m_Menu.GetID() == nID) {
      if ((Act->m_Cat.compareNoCase(Cat) == 0) || (Cat.GetLength() == 0)) {
        return Act;
      }
    }
    Act = Act->m_Next;
  }
  return NULL;
}

MKeystroke* MKeyActionMap::FindKey(const MStr &ActionName, const MStr &Cat) {
  vector<MKeystroke>::iterator  i;

  for (i = m_Keys.begin(); i != m_Keys.end(); ++i) {
    if (i->getAction() != NULL) {
      if (i->getAction()->m_ActionName.compareNoCase(ActionName) == 0) {
        if (Cat.GetLength() == 0 || i->getAction()->m_Cat.compareNoCase(Cat) == 0) {
          return &(*i);
        }
      }
    }
  }
  return NULL;
}

int MKeyActionMap::ExecuteKey(WORD K, bool C, bool A, bool S, const MStr &Cat) {
  MKeystroke *Key;
  
  Key = FindKey(K, Cat, C, A, S);
  if (!Key) {
    return 0;
  }
  
  Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo(Key->getAction()->m_DisplayName.c_str());
  int result = Key->getAction()->m_Exec();
  Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();
  return result;
}

int MKeyActionMap::ExecuteAction(HMENU hMenu, DWORD nID, const MStr &Cat) {
  MAction *Act;
  
  Act = FindAction(hMenu, nID, Cat);
  if (!Act) {
    return 0;
  }
  
  Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo(Act->m_DisplayName.c_str());
  int result = Act->m_Exec();
  Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();
  return result;
}

// Executes the action of the given name.
int MKeyActionMap::ExecuteAction(const MStr &ActionName) {
  Aztec::MSystemManager::getInstance()->getUndoManager()->beginUndo(ActionName.c_str());

  int execResult = 0;
  MAction *Act;
  
  Act = FindAction(ActionName, "");
  if (!Act) {
    // if we can't find the action, we try to execute the function 
    // as a script function.
    std::string scriptFunc = ActionName.c_str();
    
    if (scriptFunc.length() > 0) {
      if (scriptFunc.rfind(")") == scriptFunc.length() - 1) {
        // if we are ending with a bracket, then we just concatenate 
        // a semicolon.
        scriptFunc += ";";
      } else {
        // if we do not end with a bracket, then we cocatenate brackets
        // and a semicolon.
        scriptFunc += "();";
      }

      std::string result;
      if (MScriptInterpreter::getInstance()->ExecuteScript(scriptFunc.c_str(), result)) {
        execResult = 1;
      } 
    }
    execResult = 0;
  } else {
    execResult = Act->m_Exec();
  }

  Aztec::MSystemManager::getInstance()->getUndoManager()->endUndo();

  return execResult;
}


int MKeyActionMap::numMappings() {
  return m_Keys.size();
}

MAction* MKeyActionMap::FindAction(const MStr &ActionName, const MStr &Cat) {
  MAction*  A;
  int       i = 0;
  
  if (!m_Actions) {
    return NULL;
  }
  
  while((A = m_Actions->GetAction(i++))) {
    if ((ActionName.compareNoCase(A->m_ActionName) == 0)) {
      if ((Cat.compareNoCase(A->m_Cat) == 0) || Cat.GetLength() == 0) {
        return A;
      }
    }
  }
      
  return NULL;
}

MAction* MKeyActionMap::FindAction(int n) {
  return m_Actions->GetAction(n);
}

MKeystroke* MKeyActionMap::operator[](int n) {
  return &m_Keys[n];
}

MKeyActionMap& MKeyActionMap::operator=(MKeyActionMap &RHS) {
  RemoveAll();
  
  m_Keys = RHS.m_Keys;
  
  m_Actions = RHS.m_Actions;
  
  return *this;
}

int MKeyActionMap::OutputPlainText(MStr FileName) {
   // TODO: reimplement this.
   assert(0);

   return 1;
}


int MKeyActionMap::SaveToFile(MStr FileName)
{
  MTextFileWriter stream;
  if (stream.open(FileName)) {
    vector<MKeystroke>::iterator  key;

    for (key = m_Keys.begin(); key != m_Keys.end(); ++key) {
      if (key->getAction() != NULL && key->m_Key) {
        MStr  str;
        
        str.Format("\"%s\" = %i", (LPCTSTR)key->getAction()->m_ActionName, key->m_Key);
        
        if (key->m_Alt) {
          str += "+alt";
        }
        if (key->m_Ctrl) {
          str += "+ctrl";
        }
        if (key->m_Shift) {
          str += "+shift";
        }
        
        str += " \"";
        str += key->getAction()->m_Cat;
        str += "\"";
        str += "\n";

        stream.writeString(str);
      }
    }

    stream.close();
  }

  return 1;
}

int MKeyActionMap::LoadFromFile(MStr FileName) {
  MTextFileReader stream;

  if (stream.open(FileName)) {
    while (!stream.isAtEOF()) {
      MStr  actionName, keyCombo, category;
      int   keyStroke;
      bool  shift = false, ctrl = false, alt = false;

      actionName = stream.readString();

      if (stream.isAtEOF()) {
        break;
      }

      // skip pass the equals sign.
      stream.readString();

      keyCombo = stream.readString();
      category = stream.readString();

      if (keyCombo.findSubstring("shift") != -1) {
        shift = true;
      }
      if (keyCombo.findSubstring("ctrl") != -1) {
        ctrl = true;
      }
      if (keyCombo.findSubstring("alt") != -1) {
        alt = true;
      }
      keyStroke = keyCombo.ToInt();

      MAction *action;
      action = FindAction(actionName, category);
      if (action) {
        m_Keys.push_back(MKeystroke(action, keyStroke, ctrl, alt, shift));
      }
    }

    stream.close();
  }

  return 1;
}



//----------------------------------------------------------------------------------------
//  MAction
//----------------------------------------------------------------------------------------

MAction::MAction() {
  m_DisplayName = "";
  m_Cat = "";
  m_Exec = NULL;
  m_Menu.SetItem(0,0);
}

MAction::MAction(const MStr &ActionName, const MStr &DisplayName, ActionFunc *E, CMenuItem M) {
  m_DisplayName = DisplayName;
  m_ActionName = ActionName;
  m_Exec = E;
  m_Menu = M;
  m_Cat = "";
}

MAction::MAction(const MStr &ActionName, const MStr &DisplayName, const MStr &Cat, ActionFunc *E, CMenuItem M) {
  m_DisplayName = DisplayName;
  m_ActionName = ActionName;
  m_Exec = E;
  m_Menu = M;
  m_Cat = Cat;
}

void MAction::Set(const MStr &ActionName, const MStr &DisplayName, ActionFunc *E, CMenuItem M) {
  m_DisplayName = DisplayName;
  m_ActionName = ActionName;
  m_Exec = E;
  m_Menu = M;
  m_Cat = "";
}

void MAction::Set(const MStr &ActionName, const MStr &DisplayName, const MStr &Cat, ActionFunc *E, CMenuItem M) {
  m_DisplayName = DisplayName;
  m_ActionName = ActionName;
  m_Exec = E;
  m_Menu = M;
  m_Cat = Cat;
}

//----------------------------------------------------------------------------------------
//  MActionListType
//----------------------------------------------------------------------------------------
MActionListType::MActionListType() {
  m_First = NULL;
}

MActionListType::~MActionListType() {
  MAction   *F,*N;
  
  F = m_First;
  
  if (F == NULL) {
    return;
  }
  
  while (F) {
    N = F->m_Next;
    delete F;
    F = N;
  }
}

int MActionListType::Add(const MStr &ActionName, const MStr &DisplayName, ActionFunc *E, CMenuItem M) {
  MAction *A;
  
  A = new MAction(ActionName, DisplayName,E,M);
  A->m_Prev = NULL;
  A->m_Next = m_First;
  
  if (m_First == NULL) {
    m_First = A;
    return 1;
  }
  
  if (m_First) {
    m_First->m_Prev = A;
  }
  
  m_First = A;
  
  return 1;
}

int MActionListType::Add(const MStr &ActionName, const MStr &DisplayName, const MStr &Cat, ActionFunc *E, CMenuItem M) {
  MAction *A;
  
  A = new MAction(ActionName, DisplayName,Cat, E,M);
  A->m_Prev = NULL;
  A->m_Next = m_First;
  
  if (m_First == NULL) {
    m_First = A;
    return 1;
  }
  
  if (m_First) {
    m_First->m_Prev = A;
  }
  
  m_First = A;
  
  return 1;
}

int MActionListType::Add(const MStr &ActionName, const MStr &DisplayName, const MStr &Cat, ActionFunc *E, CWnd *Wnd, UINT IDItem) {
  
  return Add(ActionName, DisplayName, Cat, E, CMenuItem(Wnd, IDItem));
}

int MActionListType::Sort() {
  //
  return 0;
}

MAction* MActionListType::GetAction(int i) {
  MAction  *A;
  
  A = m_First;
  
  while (A && i>0) {
    A = A->m_Next;
    i--;
  }
  
  return A;
}


MAction* MActionListType::operator[](int i) {
  MAction*  A,N;
  
  A = m_First;
  
  while (A && i>0) {
    A = A->m_Next;
    i--;
  }
  
  return A;
}

int MActionListType::EnumList(EnumActionFunc *Func, int lParam) {
  MAction*  A;
  
  A = m_First;
  
  while (A) {
    if (Func(A,lParam) == 0) {
      return 0;
    }
    
    A = A->m_Next;
  }
  return 1;
}


MStr VKeyToString(WORD Ch, WORD Old) {
  MStr A;
  
  if (Ch >= VK_F1 && Ch<= VK_F24) {  // F1 -> F12
    int diff = Ch - VK_F1;
    A += "F";
    if (Ch < VK_F10) {
      A += (char)(diff+'1');
    } else {
      if (Ch < VK_F20) {
        diff -= 10;
        A += "1";
        A += (char)(diff+'1');
      } else {
        diff -= 20;
        A += "2";
        A += (char)(diff+'1');
      }
    }
  } else {
    A = (char)Ch;
  }
  
  
  switch(Ch) {
  case VK_DELETE: A = "Delete" ; break ;
  case VK_BACK: A = "Backspace" ; break ;
  case VK_SPACE: A = "Space" ; break ;
  case VK_INSERT: A = "Insert" ; break ;
  case VK_END: A = "End" ; break ;
  case VK_HOME: A = "Home" ; break ;
  case VK_PRIOR: A = "Page Up" ; break ;
  case VK_NEXT: A = "Page Down" ; break ;
  case VK_ESCAPE: A = "Escape" ; break ;
  case VK_LEFT: A = "Left Arrow" ; break ;
  case VK_UP: A = "Up Arrow" ; break ;
  case VK_RIGHT: A = "Right Arrow" ; break ;
  case VK_DOWN: A = "Down Arrow" ; break ;
  case VK_CAPITAL: A = "CapsLock" ; break ;
  case VK_RETURN: A = "Return" ; break ;
    
  case VK_NUMPAD0: A = "Numpad 0" ; break ;
  case VK_NUMPAD1: A = "Numpad 1" ; break ;
  case VK_NUMPAD2: A = "Numpad 2" ; break ;
  case VK_NUMPAD3: A = "Numpad 3" ; break ;
  case VK_NUMPAD4: A = "Numpad 4" ; break ;
  case VK_NUMPAD5: A = "Numpad 5" ; break ;
  case VK_NUMPAD6: A = "Numpad 6" ; break ;
  case VK_NUMPAD7: A = "Numpad 7" ; break ;
  case VK_NUMPAD8: A = "Numpad 8" ; break ;
  case VK_NUMPAD9: A = "Numpad 9" ; break ;
    
  case VK_MULTIPLY: A = "Numpad *" ; break ;
  case VK_ADD: A = "Numpad +" ; break ;
  case VK_SUBTRACT: A = "Numpad -" ; break ;
  case VK_DECIMAL: A = "Numpad ." ; break ;
  case VK_DIVIDE: A = "Numpad /" ; break ;
  case VK_SEPARATOR: A = "Separator"; break ;
    
  case VK_NUMLOCK: A = "Numlock"; break ;
  case VK_SCROLL: A = "Scroll Lock"; break ;
  case VK_TAB: A = "Tab" ; break ;
    
  case 0xBA: A = ';'; break ;
  case 0xBB: A = '='; break ;
  case 0xBC: A = ','; break ;
  case 0xBD: A = '-'; break ;
  case 0xBE: A = '.'; break ;
  case 0xBF: A = '/'; break ;
  case 0xC0: A = '`'; break ;
  case 0xDB: A = '['; break ; 
  case 0xDC: A = '\\'; break ;
  case 0xDD: A = ']'; break ;
  case 0xDE: A = '\''; break ;
    
  }
  
  return A;
}

