//---------------------------------------------------------------------------
#include <AztecMainPCH.h>
#include "Resource.h"

#include "RegCtrl.h"
#include "RegClass.h"

#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <assert.h>

// All extra units
#include "MDLGlobs.h"
#include "Colors.h"

// All dialogs and stiff go here
#include "DlgGlobs.h"

#include "MTextFileStream.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//---------------------------------------------------------------------------

void WriteCFGVersion(MOutputStream &stream) {
  stream.writeString("ConfigVersion = 1.0");
  stream.writeString("\n");
}

void WriteCFGMRU(MOutputStream &stream, int nMRU, MStr *MRUList) {
  // int number of mru, then list of mru
  for (int n = 0; n < nMRU; n++) {
    MStr  str;

    str.Format("MRU%i = ", n);
    stream.writeString(str);
    stream.writeString("\"");
    stream.writeString((MRUList[n] != NULL ? MRUList[n] : ""));
    stream.writeString("\"");
    stream.writeString("\n");
  }
}

void WriteCFGColors(MOutputStream &stream, int nCols, COLORREF **Cols, MStr *ColNames) {
  for (int i=0;i<nCols;i++) {
    MRGBAByte colour;

    colour.r = *Cols[i] & 0x0000FF;
    colour.g = (*Cols[i] & 0x00FF00) >> 8;
    colour.b = (*Cols[i] & 0xFF0000) >> 16;

    stream.writeString("setColor ");
    stream.writeString("\"");
    stream.writeString(ColNames[i]);
    stream.writeString("\" = ");
    stream.writeInt(colour.r);
    stream.writeString(" ");
    stream.writeInt(colour.g);
    stream.writeString(" ");
    stream.writeInt(colour.g);
    stream.writeString("\n");
  }
}




int WriteConfigFile(const char *FileName) {
  MTextFileWriter stream;
  
  if (stream.open(FileName)) {
    WriteCFGVersion(stream);
    WriteCFGMRU(stream, g_MRU_Count, g_MRUList);
    WriteCFGColors(stream, NUM_COLORS, COLORS, COLORNAMES);
    
    stream.writeString("MaxUndoNodes = ");
    stream.writeInt(Aztec::getSystemManager()->getUndoManager()->getMaxUndos());
    stream.writeString("\n");

    stream.writeString("setUserConfig OrbitViewKey = \"");
    g_ProgSet.m_PanOrbitKey.WriteToStream(stream);
    stream.writeString("\"\n");

    stream.writeString("setUserConfig PanViewKey = \"");
    g_ProgSet.m_PanTrackKey.WriteToStream(stream);
    stream.writeString("\"\n");

    stream.writeString("setUserConfig ZoomKey = \"");
    g_ProgSet.m_PanZoomKey.WriteToStream(stream);
    stream.writeString("\"\n");

    stream.writeString("setUserConfig ManipGrabKey = \"");
    g_ProgSet.m_ManipGrabKey.WriteToStream(stream);
    stream.writeString("\"\n");

    stream.close();
  
    return 1;
  }

  return 0;
  
}

void ReadSetColour(MInputStream &stream) {
  MStr  colourName;
  MRGBAByte colour;
  
  // get the color name
  colourName = stream.readString();
  // get the equals sign.
  stream.readString();
  
  colour.r = stream.readInt();
  colour.g = stream.readInt();
  colour.b = stream.readInt();
  
  // TODO: convert the colornames global to
  // a STL map.
  
  // convert the colour to a DWORD for the appropriate
  for (int c=0;c<NUM_COLORS;c++) {
    if (COLORNAMES[c].compareNoCase(colourName) == 0) {
      *COLORS[c] = ((int)colour.b << 16) + ((int)colour.g << 8) + colour.r;
    }
  }
}

void ReadSetUserConfig(MInputStream &stream) {
  MStr  configItem;

  configItem = stream.readString();

  if (configItem == "OrbitViewKey") {
    MShiftState state;

    // read the equal sign
    stream.readString();

    // read the key combo
    state.ReadFromStream(stream);

    g_ProgSet.m_PanOrbitKey = state;
  }
  if (configItem == "PanViewKey") {
    MShiftState state;

    // read the equal sign
    stream.readString();

    // read the key combo
    state.ReadFromStream(stream);

    g_ProgSet.m_PanTrackKey = state;
  }
  if (configItem == "ZoomKey") {
    MShiftState state;

    // read the equal sign
    stream.readString();

    // read the key combo
    state.ReadFromStream(stream);

    g_ProgSet.m_PanZoomKey = state;
  }
  if (configItem == "ManipGrabKey") {
    MShiftState state;

    // read the equal sign
    stream.readString();

    // read the key combo
    state.ReadFromStream(stream);

    g_ProgSet.m_ManipGrabKey = state;
  }
}

int ReadConfigFile(const char *FileName) {
  MTextFileReader stream;

  if (stream.open(FileName)) {


    while (!stream.isAtEOF()) {
      MStr  configItem;
      MStr  str;

      configItem = stream.readString();

	  if (configItem == NULL) {
		  continue;
	  }

      if (configItem == "ConfigVersion") {

        // read the equal sign
        stream.readString();
        // read the version number.
        str = stream.readString();

        // make sure we have a valid version number.
        assert( str == "1.0" );

      } else if (configItem == "setColor") {
        ReadSetColour(stream);

      } else if (configItem == "MaxUndoNodes") {
        // read the equals sign.
        stream.readString();

        // read the number of undo nodes.
        Aztec::getSystemManager()->getUndoManager()->setMaxUndos(stream.readInt());


        // Check to see if the config item is an mru item.
      } else if (configItem.findSubstring("MRU") == 0) {
        int mruIndex;
        MStr  mruFile;

        // get the index from the config item name.
        mruIndex = configItem.c_str()[3] - '0';

        // read the equal sign.
        stream.readString();

        mruFile = stream.readString();
        g_MRUList[mruIndex] = mruFile.c_str();
      } else if (configItem == "setUserConfig") {
        ReadSetUserConfig(stream);
      }
    }


    stream.close();
  }
   return 1;
}



