#include <AztecMainPCH.h>
#include <MGLScaleTool.h>

// AztecMain includes
#include <MDLGlobs.h>

// standard includes
#include <math.h>

// This function takes the current gl matrix, and normalises the three main vectors, so
// scaling does not effect drawing
static void NormaliseGLMatrix(const MVector3 &ScaleVec) {
  float    Mat[16];
  MVector3 XVec, YVec, ZVec;
  
  glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
  
  XVec.set(Mat[0], Mat[1], Mat[2]);
  YVec.set(Mat[4], Mat[5], Mat[6]);
  ZVec.set(Mat[8], Mat[9], Mat[10]);
  
  XVec.normalize();
  YVec.normalize();
  ZVec.normalize();
  
  XVec *= (float)fabs(ScaleVec.x);
  YVec *= (float)fabs(ScaleVec.y);
  ZVec *= (float)fabs(ScaleVec.z);
  
  Mat[0] = XVec.x; Mat[1] = XVec.y; Mat[2] = XVec.z;
  Mat[4] = YVec.x; Mat[5] = YVec.y; Mat[6] = YVec.z;
  Mat[8] = ZVec.x; Mat[9] = ZVec.y; Mat[10] = ZVec.z;
  
  glLoadMatrixf(Mat);
}


//--------------------
//  MGLScaleToolType
//--------------------
MGLScaleToolType::MGLScaleToolType()
{
  m_RequiresSel = true;
  setName("KToolScale");
  m_ScaleVec.set(1,1,1);
}

int MGLScaleToolType::DrawTool(bool Select, MShiftState ShiftState, MBaseViewWndPtr View)
{
  // Get the current viewport and cast it to a gl wnd.
  COpenGLWnd     *GLWnd;
  bool           DrawManips;
  
  
  GLWnd = AZTEC_CAST(COpenGLWnd, View);
  
  if (GLWnd == NULL)
    return 0;
  
  if (!m_Dragging) {
    m_PivotPoint = g_Scene->getSelectionCentre();
    getAxisMatrix(GLWnd);
  }
  
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  
  // Perform the viewport transformation
  
  GLWnd->DoViewportTransform();
  glMatrixMode(GL_MODELVIEW);
  
  float ScaleFact = GLWnd->GetScaleFactor(m_PivotPoint);
  glTranslatef(m_PivotPoint.x, m_PivotPoint.y, m_PivotPoint.z);
  glMultMatrixf((float*)m_AxisTransform.m);
  
  glScalef(ScaleFact, ScaleFact, ScaleFact);
  
  if (RequiresSelection())
    DrawManips = g_Scene->anythingSelected();
  else
    DrawManips = true;
  
  if (DrawManips)
  {
    GLWnd->DrawScaleIcon(g_IconSize,0.5f,1, Select);
  }
  
  glPopMatrix();
  
  glPopAttrib();
  return 1;
}

int MGLScaleToolType::onMouseDown(int X, int Y, const MShiftState &Shift)
{
  
  MVector3    Orig, Dir, ViewNorm;
  MMatrix4    XFormMat;
  
  {
    COpenGLWnd     *GLWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
    
    if (GLWnd != NULL)
    {
      
      GLWnd->m_ViewXFormInv.transform(MVector3(1,0,0),m_RightVec);
      
      m_RightVec.x = m_RightVec.x;
      m_RightVec.y = m_RightVec.y;
      m_RightVec.z = m_RightVec.z;
    }
  }
  GetPlaneParams(m_Constraint, m_AxisTransform);
  
  m_ScaleVec.set(1,1,1);
  
  MXYZToolType::onMouseDown(X, Y, Shift);

  return TOOLRESULT_DRAWALL;
}

int MGLScaleToolType::onMouseUp(int X, int Y, const MShiftState &Shift) {
  MXYZToolType::onMouseUp(X, Y, Shift);
  
  UpdateKeys("Scale", TRANSFORM_CHAN_SCALE);
  
  m_ScaleVec.set(1,1,1);
  
  return TOOLRESULT_DRAWALL;
}

int MGLScaleToolType::onMouseMove(int X, int Y, const MShiftState &Shift)
{
  MXYZToolType::onMouseMove(X, Y, Shift);
  
  if (!m_Dragging) {
    return TOOLRESULT_DRAWNONE;
  }
  
  MVector3 OfsVec, N;
  
  OfsVec = m_CurVec;
  OfsVec -= m_DownVec;
  
  N.set(1,1,1);
  if (m_PickedManip == 1) { // X Axis
    N.y = 0;
    N.z = 0;
  } else if (m_PickedManip == 2) { // Y Axis
    N.x = 0;
    N.z = 0;
  } else if (m_PickedManip == 3) { // Z Axis
    N.x = 0;
    N.y = 0;
  } else if (m_PickedManip == 4) {
    float    Dist;
    Dist = OfsVec * m_RightVec;
    OfsVec.x = Dist;
    OfsVec.y = OfsVec.x;
    OfsVec.z = OfsVec.x;
  }
  
  m_ScaleVec.x = (float)(1 + 0.1 * OfsVec.x * N.x);
  m_ScaleVec.y = (float)(1 + 0.1 * OfsVec.y * N.y);
  m_ScaleVec.z = (float)(1 + 0.1 * OfsVec.z * N.z);

  MMatrix4 localScaleXForm, scaleXForm;

  localScaleXForm.identity();
  localScaleXForm.m[0][0] = m_ScaleVec.x;
  localScaleXForm.m[1][1] = m_ScaleVec.y;
  localScaleXForm.m[2][2] = m_ScaleVec.z;

  scaleXForm = m_AxisTransformInv * localScaleXForm * m_AxisTransform;
//  scaleXForm = localScaleXForm;

  // Go through the currently selected objects, and move their components.
  MBaseObjectPtr BaseObj;
  MSceneObjectPtr Obj;
  g_Scene->getObjectList()->beginIteration();
  while (( BaseObj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    Obj = AZTEC_CAST(MSceneObject, BaseObj);
    if (Obj == NULL) {
      continue;
    }
    
    MTreeObjectNodePtr ObjNode;
    
    ObjNode = g_Scene->getObjectList()->getCurrentNode();
    MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
    if (compObj != NULL && compObj->isInComponentMode()) {
      MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
      if (editableCompObj != NULL && editableCompObj->isInComponentMode()) {
        editableCompObj->restoreComponentPositions(editableCompObj->getComponentMode(), MComponentisedObject::COMPONENT_SELECTED);

        MMatrix4 objToWorld;
        MMatrix4 worldToObj;
        g_Scene->getWorldTransformMatrix(ObjNode, objToWorld);
        worldToObj = objToWorld;
        worldToObj.inverse();

        MMatrix4 transform;
        transform.identity();

        MVector3 localPivot = worldToObj * m_PivotPoint;

        worldToObj.m[3][0] = 0;
        worldToObj.m[3][1] = 0;
        worldToObj.m[3][2] = 0;
        worldToObj.normaliseRotation();
        objToWorld.m[3][0] = 0;
        objToWorld.m[3][1] = 0;
        objToWorld.m[3][2] = 0;
        objToWorld.normaliseRotation();

        transform.m[3][0] -= localPivot.x;
        transform.m[3][1] -= localPivot.y;
        transform.m[3][2] -= localPivot.z;

        transform *= objToWorld  * scaleXForm * worldToObj;// * worldToObj;

        transform.m[3][0] += localPivot.x;
        transform.m[3][1] += localPivot.y;
        transform.m[3][2] += localPivot.z;

        editableCompObj->transformComponents(
          editableCompObj->getComponentMode(),
          transform,
          MComponentisedObject::COMPONENT_SELECTED,
          false);

      }
    } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      MTransformObjectPtr XForm;
      MVector3          Vec;
      
      XForm = g_Scene->getTransform(g_Scene->getObjectList()->getCurrentNode());
      
      if (XForm != NULL) {
        MVector3       Vec;
        
        XForm->fetchValues();
        Vec = XForm->getScaleVector(Obj->getTime());
        Vec.x *= m_ScaleVec.x;
        Vec.y *= m_ScaleVec.y;
        Vec.z *= m_ScaleVec.z;
        XForm->setScaleVector(Vec);
        
        XForm->setParamByName("Scl", XForm->getScaleVector(Obj->getTime()).convertToString(), false);
      }
    }
  }         
  g_Scene->getObjectList()->endIteration();

  UpdateKeys("Scale", TRANSFORM_CHAN_SCALE, false);
  
  return TOOLRESULT_DRAWALL;
}

//------------------
//  MScaleToolType
//------------------
MScaleToolType::MScaleToolType()
{
  m_RequiresSel = true;
}

int MScaleToolType::DrawTool(bool Select, MShiftState ShiftState, MBaseViewWndPtr View)
{
  if (AZTEC_CAST(COpenGLWnd, View))
    return MGLScaleToolType::DrawTool(Select, ShiftState, View);
  
  return 0;
}

int MScaleToolType::onMouseDown(int X, int Y, const MShiftState &Shift)
{
  if (AZTEC_CAST(COpenGLWnd, g_CurView))
    return MGLScaleToolType::onMouseDown(X, Y, Shift);
  
  return TOOLRESULT_DRAWNONE;
}

int MScaleToolType::onMouseUp(int X, int Y, const MShiftState &Shift)
{
  if (AZTEC_CAST(COpenGLWnd, g_CurView))
    return MGLScaleToolType::onMouseUp(X, Y, Shift);
  
  return TOOLRESULT_DRAWNONE;
}

int MScaleToolType::onMouseMove(int X, int Y, const MShiftState &Shift)
{
  if (AZTEC_CAST(COpenGLWnd, g_CurView))
    return MGLScaleToolType::onMouseMove(X, Y, Shift);
  
  return TOOLRESULT_DRAWNONE;
}


