//
// Routines for performing CSG on polygonal objects (b-rep)
//
// Written by Alexander Enzmann, July 2001
//
//
//
#if !defined( __RTPOLY_CSG )
#define __RTPOLY_CSG

#include "MMath.h"
#include "RangTree.h"

using namespace Aztec;

#include <set>
#include <list>

// A single vertex for a triangle.
class TriVertex {
public:
	TriVertex() { }
	TriVertex(MVector3 &tW, MVector3 &tP, MVector3 &tN, MVector3 &tU) {
		W = tW;
		P = tP;
		N = tN;
		U = tU;
	}
	TriVertex(TriVertex *tri) {
		W = tri->W;
		P = tri->P;
		N = tri->N;
		U = tri->U;
	}
	~TriVertex();

	MVector3 W, P, N, U; // u/v, object/world coordinate, normal, and screen coordinate

	TriVertex &operator=(const TriVertex &b) {
		W = b.W;
		P = b.P;
		N = b.N;
		U = b.U;
		return *this;
	}

	// Linearly interpolate between two vertices.  The 
	void lerp(float t, TriVertex *ptA, TriVertex *ptB) {
		MVector3 v;
		v = ptB->W - ptA->W;
		W = ptA->W + t * v;
		v = ptB->P - ptA->P;
		P = ptA->P + t * v;
		v = ptB->U - ptA->U;
		U = ptA->U + t * v;
		v = ptB->N - ptA->N;
		N = ptA->N + t * v;
	}

	// For two vertices to be equal, all of their components must be equal.  This
	// does have some implications due to precision, but for now it will have to do.
	friend bool operator==(const TriVertex &a, const TriVertex &b) {
		if (a.U == b.U && a.N == b.N && a.P == b.P && a.W == b.W)
			return 1;
		else
			return 0;
	}

	// Lexicographic ordering of vertices (primarily for sorting them)
	friend bool operator<(const TriVertex &a, const TriVertex &b) {
		if (a.W.x < b.W.x)
			return 1;
		else if (a.W.x == b.W.x) {
			if (a.W.y < b.W.y)
				return 1;
			else if (a.W.y == b.W.y) {
				if (a.W.z < b.W.z)
					return 1;
				else if (a.W.z == b.W.z) {
					if (a.U.x < b.U.x)
						return 1;
					else if (a.U.x == b.U.x) {
						if (a.U.y < b.U.y)
							return 1;
						else if (a.U.y == b.U.y) {
							if (a.U.z < b.U.z)
								return 1;
							else if (a.U.z == b.U.z) {
								if (a.N.x < b.N.x)
									return 1;
								else if (a.N.x == b.N.x) {
									if (a.N.y < b.N.y)
										return 1;
									else if (a.N.y == b.N.y) {
										if (a.N.z < b.N.z)
											return 1;
									}
								}
							}
						}
					}
					
				}
			}
		}
		return 0;
	}
};

// A container for a triangle vertex.  The indirection is so we can store
// a smaller amount of information in the various STL containers.
// NOTE: these vertices are intended to be managed by a PolyTriList.  If you store/use
// them outside one of them, you will be responsible for freeing the memory
// associated with "ptr".  Normally the destructor for PolyTriList frees all
// the vertices.
class TriVertexPtr {
public:
	TriVertex *ptr;
	TriVertexPtr() : ptr(NULL) {
	}
	TriVertexPtr(TriVertex *p) {
		ptr = p;
	}
	~TriVertexPtr() {
	}
	TriVertexPtr &operator=(const TriVertexPtr &b) {
		ptr = b.ptr;
		return *this;
	}

	// Lexicographic ordering of vertices (primarily for sorting them)
	friend bool operator<(const TriVertexPtr &a, const TriVertexPtr &b) {
		return (*(a.ptr)) < (*(b.ptr));
	}
};

class TriVertexPtrCompare {
public:
	bool operator()(const TriVertexPtr &a, const TriVertexPtr &b) {
		return ((*(a.ptr)) < (*(b.ptr))) ? true : false;
	}
};

class VertexSet : public std::set< TriVertexPtr, TriVertexPtrCompare > {
public:
	~VertexSet();
};
typedef VertexSet::iterator VertexSetPtr;

class PolyTri {
protected:
	//	plane equation for the polygon
	MPlane plane;
public:
	//	the array of points that compose the triangle
	TriVertexPtr vertices[3];
	// MVector3 W[3], P[3], U[3], N[3];

	// Pointer to user-defined data
	void *prim, *data;
	// short i1, i2; // Indices used to project this triangle into 2D
	// Range3D bbox; // Axis aligned bounding box for this triangle

	// Triangles on the other side of each of the three edges.  Edge 0 corresponds
	// to [P[0], P[1]], edge 1 is [P[1], P[2]], and edge 2 is [P[2], P[0]].
	PolyTri *adjacent[3];

	PolyTri(MVector3 *iW, MVector3 *iP, MVector3 *iU, MVector3 *iN, int i0, int i1, int i2,
			void *iprim = NULL, void *idata = NULL);
	PolyTri(MVector3 *iW, MVector3 *iP, MVector3 *iU, MVector3 *iN, void *iprim = NULL, void *idata = NULL);
	PolyTri(TriVertex *v0, TriVertex *v1, TriVertex *v2, void *iprim = NULL, void *idata = NULL);
	PolyTri(PolyTri *tri);

	// Classification of this triangle with respect to some object.  The enumerated
	// type and member variable are provided as a convience for performing CSG.  There
	// is a significant difference between unknown and indeterminate.  Unknown is used
	// to indicate that the triangle hasn't been tested.  Indeterminate is used if the
	// code can't tell if the triangle is inside or outside the other surface.  Inside
	// and outside are used if the code knows definitely on which side the triangle lies.
	enum  { unknown, indeterminate, inside, outside } containment_value;

	// Figure out which two indices represent the largest changes in extent.
	void compute_indices(int &i1, int &i2);

	// Find an axis aligned bounding box for the triangle
	void compute_bbox(Range3D &bbox);

	//	return the ith vertex of the polygon
	const MVector3 &Vertex(int i) const { return vertices[i].ptr->W; }

	//	return a reference to the plane equation of the triangle
	const MPlane &Plane(void) const { return plane; }

	//	compute the normal to the triangle (the plane the triangle lies in)
	MVector3 Normal(void) const;

	// Return the point at the center of the triangle
	MVector3 Center(void) {
		MVector3 P = vertices[0].ptr->W;
		P += vertices[1].ptr->W;
		P += vertices[2].ptr->W;
		P *= 1.0f / 3.0f;
		return P;
	}

	// test a point to see if it is inside the polygon (crossings method)
	bool Contains(const MVector3 &pt) const;

	//	completely reverse the orientation of the triangle
	void Invert(void);

	//	reverse the orientation of the vertex normals
	void InvertNormals(void);

	//	compute the distance at which the ray intersects the triangle
	float intersect(const MRay &r) const;

	// Test a triangle to see if it touches/slices this triangle
	bool intersect(const PolyTri *tri);

	// Deallocate the memory associated with the vertices.  NOTE: make sure
	// this triangle isn't contained in a PolyTriList when this is called.
	void FreeVertices();
};

typedef PolyTri *PolyTriPtr;

class PolyTriList : public std::list< PolyTriPtr > {
public:
	VertexSet vertices;

	PolyTriList(void) {}
	~PolyTriList(void);

	//	add a polygon to the list
	void Push(PolyTriPtr tri);

	// Invert all polygons on the list
	void Invert(void);

	//	pop the head from the list
	PolyTriPtr Pop(void);

	// Compute adjacency information for all of the triangles in the list.  The
	// value of epsilon is used to tell when two vertices are close enough to
	// consider them equal.
	void ComputeAdjacencies();

	// Deallocate vertex information
	void FreeVertices();
};

class tri23tree : public Range_Tree {
protected:
	// Classification type for plane comparisons
	enum split_result { coplanar, inside, outside, splitting };
	// Split a PolyTri by a plane
	split_result Split3D(const PolyTriPtr tri, const MPlane &plane,
						 PolyTriPtr &in0, PolyTriPtr &in1,
						 PolyTriPtr &out0, PolyTriPtr &out1);

	// P is used during CSG/intersection testing.  It holds the starting
	// point of the ray.
	MVector3 P;

	// What operation are we performing?
	enum {
		drawing, searching, finding_slicers, slicing, collecting,
		inserting, deleting, extracting, unlinking, counting_crossings
	} state;

	// List of triangles resulting from an operation on this tree.
	PolyTriList result_list;

	// User supplied tree manipulation - in this case we are collecting all
	// PolyTris that fall within a given range
	void process(Range3D &range, void *value, void *data);

	// User supplied cleanup routine
	void cleanup(void *data);

	// See if a given PolyTri intersects any of the PolyTris in the data list.
	// If so, then pointers to the PolyTris are added to result_list.
	void check_PolyTri_intersections(range_data *data_list, PolyTriPtr tri);

	// Determine if a ray parallel to the y-axis intersects tri.
	bool yintersect(MVector3 &P, PolyTriPtr tri, MVector3 &W);
	// Determine if a ray parallel to the z-axis intersects tri.
	bool zintersect(MVector3 &P, PolyTriPtr tri, MVector3 &W);

	// Chop all the PolyTris in another tree
	void slice_tree(range_data *data_list, tri23tree *tree);

	// Gather any PolyTris at the leaf of a tree that intersect range
	void collect_PolyTris(Range3D &range, range_data *data_list, PolyTriList *list);

	// Draw a triangle using the externally defined routine "void DrawTriangle(PolyTriPtr tri)"
	void draw_triangle(range_data *data_list);

	// Copy the underlying range data (support for public Clone() method)
	void clone_PolyTri_tree(node23 *old_tree, node23 *new_tree);

	// Rip everything out of the tree, collecting triangles as we go
	void extractTris(nodeptr intree, PolyTriList &tris);
public:
	~tri23tree();

	void Insert(PolyTriPtr tri);
	void Delete(PolyTriPtr tri);
	void Insert(PolyTriList &tris);
	void Delete(PolyTriList &tris);

	// Return true if the given point is contained by the PolyTri mesh (odd
	// number of crossings
	bool Inside(const MVector3 &p);
	bool InsideAll(const MVector3 &p);

	// Return the list of PolyTris that intersect the input PolyTri
	// Note: this routine is NOT reentrant!  It relies on the protected
	// state and result_list variables.
	void FindSlicers(PolyTriPtr tri);

	// Slice all PolyTris in the input tree that intersect PolyTris in this tree
	void Slice(tri23tree *tree);

	// Return a list of all PolyTris in the tree
	void Tris(PolyTriList &tris);

	// Pull all triangles out of the tree and deposit in "tris".  The
	// tree is empty after this operation
	void ExtractTris(PolyTriList &tris);

	// Return a list of all PolyTris that touch a bounding box
	void TrisHitting(Range3D &range, PolyTriList &tris);

	// Remove references to a list of PolyTris, without deallocating the
	// PolyTris themselves.
	void Unlink(PolyTriList &tris);

	// Remove references to all contents of the tree, without deallocating
	// the PolyTris themselves.
	void UnlinkAll();

	// Draw all triangles in this tree
	void Draw();

	// Generate an exact clone of this tree (including making full copies
	// of all PolyTris)
	void Clone(tri23tree &tree);

	// Return the number of PolyTris in the tree
	int TriCount();
};

#define CSG_OP_OBJ1_INSIDE  0x0001
#define CSG_OP_OBJ1_OUTSIDE 0x0002
#define CSG_OP_OBJ2_INSIDE  0x0004
#define CSG_OP_OBJ2_OUTSIDE 0x0008

#define CSG_UNION (CSG_OP_OBJ1_OUTSIDE | CSG_OP_OBJ2_OUTSIDE)
#define CSG_INTERSECTION (CSG_OP_OBJ1_INSIDE | CSG_OP_OBJ2_INSIDE)
#define CSG_DIFFERENCE1 (CSG_OP_OBJ1_OUTSIDE | CSG_OP_OBJ2_INSIDE)
#define CSG_DIFFERENCE2 (CSG_OP_OBJ1_INSIDE | CSG_OP_OBJ2_OUTSIDE)
#define CSG_COLLECTION (CSG_UNION | CSG_INTERSECTION)


// Perform the Constructive Solid Geometry (CSG) operation, csg_op, on objects
// obj1 and obj2.  The possible values of csg_op are given above, and are
// simply bitwise unions of the masks CSG_OP_OBJ1_INSIDE ... CSG_OP_OBJ2_OUTSIDE.
void MeshCSG(int csg_op, PolyTriList &obj1, PolyTriList &obj2, PolyTriList &resultobj);

#endif // __RTPOLY_CSG

