#ifndef AztecView_Header
#define AztecView_Header

#include <gui/MContainer.h>
#include <gui/MMenuBar.h>

namespace AztecGUI {

  class MToolType;
  typedef Aztec::MRefCountedPtr<MToolType> MToolTypePtr;

  class AztecView;
  typedef Aztec::MRefCountedPtr<AztecView> AztecViewPtr;

  /**
   * This is the base class for all views in Aztec. A view is just a control 
   * that presents the data in the 3D scene to the user in a special way. Some
   * examples of views are 3D views, graph views, the outline view (seeing the
   * objects in a tree heirarchy), and many others.
   *
   * To implement the view, the drawView() and getViewGroup() group methods 
   * need to be overridden. If mouse input is required, the 
   * MComponent::onMousePressed, MComponent::onMouseReleased, 
   * MComponent::onMouseMoved should be overridden.
   */
  class AztecView : public Aztec::MContainer {
  public:
    /**
     * This gets the AztecView that contains the given componnet. This 
     * function just keeps getting parent containers until we find a view. 
     *
     * @return the View that contains the object, otherwise NULL if we couldn't find one.
     */
    static AztecViewPtr getViewForComponent(const Aztec::MComponentPtr &component);

    AztecView(const std::string &name);
    ~AztecView();

    /**
     * Creates a copy of this view ready to insert into a new container. This 
     * is used to switch views, or to crate new floating views.
     *
     * @return The new view
     */
    virtual AztecViewPtr createCopy() = 0;

    /**
     * Gets the name of the view. 
     */
    std::string getName();

    /**
     * Sets the name of the view.
     */
    void setName(const std::string &name);

    /**
     * This is called when the view needs to be redrawn.
     */
    virtual void drawView() = 0;

    /**
     * This gets the group that the view belongs too. Eeach group of views 
     * has a common set of keyboard shortcuts, tools and options associated 
     * with them. For example, anything that is part of the "3DScene" group, 
     * has common keys for rotating and panning views. Anything part of the 
     * "ObjectList" group, has common was of scrolling around the list of 
     * objects, and performing common tasks.
     */
    virtual std::string getViewGroup() const = 0;

    /**
     * This draws a border around the view. Typically used to draw the 
     * selected border around everything.
     */
    void drawBorder(const Aztec::MRect2D &r, int width);

    /**
     * Gets the default tool to use when we want to select something. 
     *
     * @return NULL if there is no default select tool.
     */
    virtual MToolTypePtr getSelectTool();

    /**
     * This method performs the correct asctions based on the return value 
     * from functinos like MToolType::onMouseMove() and similar functions. 
     * It basically checks to see if it should redraw the current view or 
     * all the views, and does the appropriate thing.
     */
    void handleToolResult(int toolResult);

    Aztec::MMenuBarPtr getMenuBar() const;

    void displayContextMenu(const Aztec::MMouseEvent &event);

    // MComponent methods
    void onCreate();
    bool onPaint();

  protected:
    /**
     * The name of this view.
     */
    std::string viewName;

    Aztec::MMenuBarPtr viewMenuBar;

    virtual void constructContextMenu(const Aztec::MMenuShellPtr &menu, const Aztec::MMouseEvent &event);
  };

}

#include <tools/MToolType.h>

#endif

