#include <Aztec3DPCH.h>

#include "tools/InsetTool.h"

// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include "MSystemManager.h"
#include "MEditableMesh.h"
#include "MAnimMesh.h"
#include "MUIManager.h"
#include <MComponentisedObject.h>

// standard includes
#include <math.h>
#include <GL/gl.h>
#include <set>
#include <assert.h>

namespace AztecGUI {
  using std::set;

  static Aztec::MVector3 m_ExtrudeVec;

  InsetTool::InsetTool()
  {
    m_RequiresSel = true;
    distance = 0.0f;
    m_ExtrudeVec.set(1,0,0);
    dragging = false;

  }

  std::string InsetTool::getName() {
    return "toolInset";
  }

  static std::vector<Aztec::MVector3> rectPoints;

  int InsetTool::drawTool(bool select, const Aztec::MComponentPtr &component) {
    // we want to draw an arrow at the centre of the selection, 
    // and and pointing in the average normal direction.

    // update our normal. If nothing was selected, we don't draw anything
    if (!dragging && !updateExtrudeNormal()) {
      return 1;
    }

    AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(component);

    if (GLWnd == NULL) {
      return 0;
    }
  
    if (rectPoints.size() > 0) {
      glBegin(GL_QUADS);
      for (unsigned i = 0; i < rectPoints.size(); ++i) {
        glVertex3fv((float*)&rectPoints[i]);
      }
      glEnd();
    }

    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    glPushMatrix();

    Aztec::MScene::getGlobalScene()->GLTransform(Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    glPushMatrix();

    float ScaleFact = GLWnd->getScalingFactor(m_CurrentPos);
    glTranslatef(m_CurrentPos.x, m_CurrentPos.y, m_CurrentPos.z);
  
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    glMultMatrixf((float*)m_GLTransform.m);

    // draw the axis icon
    glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, select, DRAWAXIS_Z | DRAWAXIS_ARROWS);

    // if we are extruding the faces, draw an arrow where we started from.
    if (dragging) {
      glPopMatrix();
      glPushMatrix();

      ScaleFact = GLWnd->getScalingFactor(m_StartPos);
      glTranslatef(m_StartPos.x, m_StartPos.y, m_StartPos.z);
      glScalef(ScaleFact, ScaleFact, ScaleFact);
      glMultMatrixf((float*)m_GLTransform.m);

      glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.2f, 2.0f, false, DRAWAXIS_ALLGRAY | DRAWAXIS_Z | DRAWAXIS_ARROWS);
    }
  
    glPopMatrix();
    glPopMatrix();
    glPopMatrix();
  
    glPopAttrib();
  
    return 1;
  }

  bool InsetTool::updateExtrudeNormal() {
    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr meshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj != NULL && sceneObj->getShapeObject() != NULL) {
      meshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    }

    if (meshObj == NULL) {
      return false;
    }

    Aztec::MVector3 Norm, centre;
    int n, count;
  
    count = 0;
    Norm.set(0,0,0);

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      return false;
    }

    int numComps = meshObj->getComponentCount(Aztec::MUIManager::getComponentMode());

    // go through and extract the average normal and centre
    for(n = 0; n < numComps; n++) {
      if (!meshObj->isComponentFlagged(Aztec::MUIManager::getComponentMode(), n)) {
        continue;
      }

      centre += meshObj->getComponentCentre(Aztec::MUIManager::getComponentMode(), n);
      Norm += meshObj->getComponentNormal(Aztec::MUIManager::getComponentMode(), n);
      count++;
    }
    if (count) {
      Norm.normalize();
      centre /= (float)count;
    } else {
      return false;
    }
  
    m_StartPos = centre;
    m_CurrentPos = m_StartPos;
    m_ExtrudeVec = Norm;

    // transform the drawing to out z axis lines up with the normal.
    {
      Aztec::MVector3  x,y,z;

      m_GLTransform.identity();
      z = m_ExtrudeVec;

      // construct two otrhonormal vectors
      y.set(-z.y, z.z, -z.x);
      x = z.crossProduct(y);
      y = z.crossProduct(x);

      m_GLTransform.m[0][0] = x.x; m_GLTransform.m[0][1] = x.y; m_GLTransform.m[0][2] = x.z;
      m_GLTransform.m[1][0] = y.x; m_GLTransform.m[1][1] = y.y; m_GLTransform.m[1][2] = y.z;
      m_GLTransform.m[2][0] = z.x; m_GLTransform.m[2][1] = z.y; m_GLTransform.m[2][2] = z.z;
    }

    return true;
  }

  // 

  int InsetTool::onMouseDown(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseDown(event);

    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr MeshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    if (MeshObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
      return TOOLRESULT_DRAWNONE;
    }

    if (!updateExtrudeNormal()) {
      dragging = false;
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extrude requires some faces selected.");
    
      return TOOLRESULT_DRAWNONE;
    }

    dragging = true;
  
    {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
    }
  

    Aztec::MRay viewRay, normalRay;
    Aztec::MVector3  e1,e2;

    e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeVec);



    normalRay.set(m_StartPos, e2);
    viewRay = getCurrentRay();

    Aztec::MMath::distanceBetween(normalRay, viewRay, &distance, NULL);
    mouseStartedPos = m_StartPos + normalRay.Dir * distance;

    distance = 0.0;

    // if we are in point mode, we get split the dges around the selected points
    // and move the one point.
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {

    // extrude the faces if that is the mode we are in
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      MeshObj->takeSnapshot();

      // For face mode we have to make it extract the border of faces

    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {

      MeshObj->takeSnapshot();

      // We need to build up a list of edges that are selected, and build the 
      // list according to the way the edges are connected. This is important 
      // because we need a consistent way of determining if we are dealing with 
      // the left or right hand side of an edge.
      
      // unset all the edge flags we might be able to use.
      for (int t = 0; t < MeshObj->getNumTris(); ++t) {
        for (int e = 0; e < 3; ++e) {
          MeshObj->unsetEdgeFlag(t, e, EDGE_FLAGFORCHANGE);
        }
      }

    }
  
    return TOOLRESULT_DRAWALL;
  }

  inline void changeVertex(const Aztec::MEditableMeshPtr &mesh, int tri, int oldVert, int newVert) {
    for (int i = 0; i < 3; ++i) {
      if (mesh->getTriangleVertex(tri, i) == oldVert) {
        mesh->setTriangleVertex(tri, i, newVert);
      }
    }
  }

  static void insetPolygonFromTriangle(const Aztec::MEditableMeshPtr &mesh, int tri, float fraction) {
    std::set<int> trisInPoly;
    std::vector<int> pointsInPoly;
    std::vector<int> newPoints;

    mesh->getPolygonFromTriangle(tri, trisInPoly);
    mesh->getPolygonVerticesFromTriangle(tri, pointsInPoly);

    std::vector<Aztec::MVector3> normals;

    const Aztec::MVector3 triNormal = mesh->getTriangleNormal(tri);

    normals.resize(pointsInPoly.size());
    for (unsigned i = 0; i < pointsInPoly.size(); ++i) {
      const Aztec::MVector3 a = mesh->getVertexPosition(pointsInPoly[(i + pointsInPoly.size() - 1)%pointsInPoly.size()]);
      const Aztec::MVector3 b = mesh->getVertexPosition(pointsInPoly[(i + 1)%pointsInPoly.size()]);
      const Aztec::MVector3 o = mesh->getVertexPosition(pointsInPoly[i]);
      Aztec::MVector3 n1 = (a - o);
      n1.normalize();
      n1 /= triNormal;
      Aztec::MVector3 n2 = (o - b);
      n2.normalize();
      n2 /= triNormal;

      Aztec::MVector3 normal = (n1+n2);
      normal.normalize();
      normals[i] = normal;
    }

    newPoints.assign(pointsInPoly.size(), -1);
    // go and move all the vertices.
    for (unsigned i = 0; i < pointsInPoly.size(); ++i) {
      // make a new vertex
      int newVert = newPoints[i] = mesh->getNumVerts();
      Aztec::MVector3 newPos = mesh->getVertexPosition(pointsInPoly[i]) + normals[i] * fraction;
      mesh->addVertex(newPos.x, newPos.y, newPos.z);

      // change all the triangles over.
      for (std::set<int>::iterator p = trisInPoly.begin(); p != trisInPoly.end(); ++p) {
        changeVertex(mesh, *p, pointsInPoly[i], newVert);
      }
    }

    // now make some new polygons.
    for (unsigned i = 0; i < pointsInPoly.size(); ++i) {
      int a = pointsInPoly[i];
      int b = pointsInPoly[(i+1)%pointsInPoly.size()];
      int c = newPoints[(i+1)%newPoints.size()];
      int d = newPoints[i];
      
      std::vector<int> points;
      points.push_back(a);
      points.push_back(b);
      points.push_back(c);
      points.push_back(d);
      
      mesh->triangulatePolygon(points);
    }

    for (std::set<int>::iterator p = trisInPoly.begin(); p != trisInPoly.end(); ++p) {
      mesh->setTriangleFlag(*p, TRIANGLE_FLAGFORCHANGE);
    }

  }

  static void inset(const Aztec::MEditableMeshPtr &mesh, Aztec::AztecFlags flags, float fraction) {
    mesh->unsetTriangleFlags(TRIANGLE_FLAGFORCHANGE);

    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      // if we find a flagged triangle, do an inset on the polygon
      if (mesh->isTriangleFlagged(tri, flags) && !mesh->isTriangleFlagged(tri, TRIANGLE_FLAGFORCHANGE)) {
        insetPolygonFromTriangle(mesh, tri, fraction);
      }
    }
  }

  int InsetTool::onMouseMove(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseMove(event);
    // if we are extruding, keep track of how the mouse has moved etc.
  
    if (dragging) {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
      AztecGLCanvasPtr glWnd = AztecGLView::getGLCanvasFor(event.getComponent());
    
      if (glWnd == NULL) {
        distance = (float)(m_DownPos.x - m_CurPos.x) / 25.0f;
      } else {
        Aztec::MRay viewRay, normalRay;
        Aztec::MVector3  e1,e2;

        e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeVec);



        normalRay.set(mouseStartedPos, e2);
        viewRay = glWnd->getRay(event.getX(), event.getY());

        Aztec::MMath::distanceBetween(normalRay, viewRay, &distance, NULL);
        distance = -distance;
      }

      m_CurrentPos = m_StartPos - distance * m_ExtrudeVec;

      if (distance < 0) distance = 0;


      MeshObj->retreiveSnapshot();
    
      if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
        inset(MeshObj, TRIANGLE_SELECTED, distance);
      }

      MeshObj->calculateNormals();
    
      return TOOLRESULT_DRAWALL;
    }
  
    return TOOLRESULT_DRAWNONE;
  }

  int InsetTool::onMouseUp(const Aztec::MMouseEvent &event) {
    MXYZToolType::onMouseUp(event);

    if (dragging) {
/*      // If the extrusion didn't aceheive anything, just undo it.
      if (fabs(distance) < 0.01) {
        Aztec::getSystemManager()->getUndoManager()->undo();
      } else {
        Aztec::MBaseObjectPtr BaseObj;
        Aztec::MSceneObjectPtr Obj;
  
        Aztec::MScene::getGlobalScene()->getObjectList()->beginIteration();
        while (( BaseObj = Aztec::MScene::getGlobalScene()->getObjectList()->getNext() ) != NULL ) {
          Obj = AZTEC_CAST(Aztec::MSceneObject, BaseObj);
          if (Obj == NULL) {
            continue;
          }
    
          Aztec::MTreeObjectNodePtr ObjNode;
          Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
          ObjNode = Aztec::MScene::getGlobalScene()->getObjectList()->getCurrentNode();
    
          if (compObj != NULL && compObj->isInComponentMode()) {
            Aztec::MAnimMeshPtr AnimMesh = AZTEC_CAST(Aztec::MAnimMesh, compObj);
            if (AnimMesh != NULL) {
              AnimMesh->updateKey(Aztec::MScene::getGlobalScene()->getTime(), Aztec::MSystemManager::getInstance()->getSettings()->m_Animate);
            }
          }
        }
        Aztec::MScene::getGlobalScene()->getObjectList()->endIteration();
      }
    */
      dragging = false;
//      updateExtrudeNormal();
    }
  
    return TOOLRESULT_DRAWALL;
  }

}
