#include <Aztec3DPCH.h>
#include <tools/MIKBoneTool.h>

// Aztec2 includes
#include <views/AztecViewManager.h>
#include <views/AztecGLView.h>

// AztecLib includes
#include <animation/MIKControllerCCD.h>
#include <MUIManager.h>
#include <MBone.h>
#include <MSystemManager.h>
#include <MUIManager.h>

namespace AztecGUI {

  MIKBoneTool::MIKBoneTool() {
    currentState = PickingFirstBone;
  }

  std::string MIKBoneTool::getName() {
    return "toolIKBone";
  }

  void MIKBoneTool::initialise() {
    firstBone = NULL;
    secondBone = NULL;
    pickedBone = NULL;
    currentState = PickingFirstBone;

    Aztec::MSystemManager::getInstance()->logOutput("Click on the bone that will be the start of the IK chain, then press ENTER");
  }

  bool MIKBoneTool::finish() {
    if (currentState == PickingFirstBone) {
      firstBone = pickedBone;
      currentState = PickingSecondBone;
      Aztec::MScene::getGlobalScene()->selectObject(pickedBone, false);

      // we need to update the display because our selection has changed.
      AztecViewManager::redrawAllViews();

      Aztec::MSystemManager::getInstance()->logOutput("Click on the bone that will be the last bone of the IK chain, then press ENTER");

      // return false because we have other things we need to pick.
      return false;
    }

    if (currentState == PickingSecondBone) {
      secondBone = pickedBone;
      currentState = DonePicking;
      Aztec::MScene::getGlobalScene()->selectObject(pickedBone, false);

      createIKObject();

      Aztec::MSystemManager::getInstance()->logOutput("");

      // return true because we are finished with this tool.
      return true;
    }

    // if we are in any other state, we just finish the tool as per normal
    return true;
  }

  bool MIKBoneTool::cancel() {
    // if we are cancelling, then just use the default behaviour.
    return M3DToolType::cancel();
  }

  int MIKBoneTool::drawTool(bool select, const Aztec::MComponentPtr &component)
  {
    return TOOLRESULT_DRAWALL;
  }

  int MIKBoneTool::onMouseDown(const Aztec::MMouseEvent &event)
  {
    M3DToolType::onMouseDown(event);
  
    // Check to see if the mouse buttons are down.
    Aztec::MUIManager::ComponentType OldMode;
  
    AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(event.getComponent());
  
    if (GLWnd == NULL) {
      return TOOLRESULT_DRAWNONE;
    }
  
    // check to see if all the mouse buttons are up
    if (!event.getShiftState().leftMouseDown) {
      return TOOLRESULT_DRAWNONE;
    }
  
    // select all the objects in the selection
  
    OldMode = Aztec::MUIManager::getComponentMode();
    Aztec::MUIManager::setComponentMode(Aztec::MComponentisedObject::OBJECT_TYPE);

    GLWnd->getSelection(event.getX(), event.getY(), m_InputList);
  
    Aztec::MUIManager::setComponentMode(OldMode);
  
    // This is the bone that we have selected.
    Aztec::MSceneObjectPtr closestBone;
    float minDistance = 10000000000;
  
    // iterate over the list, and get out all the selection items, and find 
    // the closest one.
    Aztec::MBaseObjectPtr Obj;
  
    for (int i = 0; i < m_InputList.size(); ++i) {
      Aztec::MSelectionItem *selItem = &m_InputList[i];
    
      // make sure our selection didn't have any components,
      // since we are just after whole.
      if (selItem->getNumComponents() == 0) {
        Aztec::MSceneObjectPtr selectedObject = AZTEC_CAST(Aztec::MSceneObject, selItem->getObject());
      
        if (selectedObject == NULL) {
          continue;
        }

        Aztec::MBoneObjectPtr bone = AZTEC_CAST(Aztec::MBoneObject, selectedObject->getShapeObject());

        if (bone == NULL) {
          continue;
        }
      
        // now that we definately have a bone, find the closest one that 
        // we clicked on
        Aztec::MVector3 boneCentre = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(selectedObject), Aztec::MVector3(0,0,0));

        if ((boneCentre - getCurrentRay().Org).length2() < minDistance) {
          minDistance = (boneCentre - getCurrentRay().Org).length2();
          closestBone = selectedObject;
        }
  
      }
    }

    // deselect the old bone that we had.
    Aztec::MScene::getGlobalScene()->selectObject(pickedBone, false);
    pickedBone = closestBone;
    // selec the new bone
    Aztec::MScene::getGlobalScene()->selectObject(pickedBone);

    return true;
  }

  void MIKBoneTool::createIKObject() {
    // if we dont have two bones, then give up
    if (firstBone == NULL || secondBone == NULL) {
      return;
    }

    // create the end effector for the ik chain.
    Aztec::MVector3 effectorLocation = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(secondBone), Aztec::MVector3(0,0,0));

    Aztec::MSceneObjectPtr endEffector = new Aztec::MSceneObject();
    endEffector->setName("Effector");
    endEffector->findParameter("Translate")->setValueVector(effectorLocation);
    endEffector->findParameter("drawAxis")->setValueBoolean(true);

    Aztec::MScene::getGlobalScene()->addObject(endEffector);

    // now create the actual ik solver.
    Aztec::MIKControllerCCD *ik = new Aztec::MIKControllerCCD();
    Aztec::MIKControllerPtr ikPtr = ik;

    ik->setName("IKController");

    ik->getEndEffectorParameter()->setValue(endEffector);
    ik->getStartBoneParameter()->setValue(firstBone);
    ik->getEndBoneParameter()->setValue(secondBone);

    Aztec::MScene::getGlobalScene()->addObject(ikPtr);

    Aztec::MScene::getGlobalScene()->selectNone();
    Aztec::MScene::getGlobalScene()->selectObject(endEffector);
  }

}