#ifndef MComponent_Header
#define MComponent_Header

#include "MRefCountedPtr.h"


namespace Aztec {

  class IMComponent;
  typedef MRefCountedPtr<IMComponent> IMComponentPtr;

  class IMContainer;
  typedef MRefCountedPtr<IMContainer> IMContainerPtr;

  class MContainer;
  class MComponent;
  typedef MRefCountedPtr<MContainer> MContainerPtr;
  typedef MRefCountedPtr<MComponent> MComponentPtr;

}

#include <gui/MGuiBase.h>
#include <gui/MKeyEvent.h>
#include <gui/MMouseEvent.h>
#include <gui/MMouseCursor.h>

#include <gui/MPoint2D.h>
#include <gui/MSize2D.h>
#include <gui/MRect2D.h>
#include <gui/MColour.h>

#include <string>


namespace Aztec {


  /**
   * This MComponent class is a window of some sort. It does not necessarily
   * represent an actual Operating system window, but is really a rectangular 
   * window that exists somewhere.
   */
  class AZTECGUI_EXPORT IMComponent : public virtual IMGuiBase {
  protected:
    IMComponent() { }
    ~IMComponent() { }

  public:
    typedef IMComponentPtr Ptr;

    /**
     * Sets the name of the window.
     */
    virtual void setName(const std::string name) = 0;

    /**
     * Gets the name of the window.
     */
    virtual std::string getName() = 0;

    /**
     * Sets the parent of this window.
     */
    virtual void setParent(MContainerPtr parent) = 0;

    /**
     * Gets the parent of this window.
     */
    virtual MContainerPtr getParent() = 0;

    /**
     * Sets if the component is visible or not.
     */
    virtual void setVisible(bool visible) = 0;

    /**
     * sets the position of the window
     */
    virtual void setPosition(const MPoint2D &pos) = 0;

    /**
     * This sets the Position of the component's top left corner.
     */
    virtual void setPosition(int X, int Y) = 0;

    /**
     * This sets the position of the windoe. ie The top left coordiante
     * and the width and height of the component.
     */
    virtual void setPosition(int X, int Y, int width, int height) = 0;

    /**
     * This gets the top left coordinate of the component.
     */
    virtual MPoint2D getPosition() = 0;

    /**
     * This sets the size (width and height) of the component.
     */
    virtual void setSize(const MSize2D &size) = 0;

    /**
     * This sets the size (width and height) of the component.
     */
    virtual void setSize(int width, int height) = 0;

    /**
     * Gets the size (width and height) of the component.
     */
    virtual MSize2D getSize() = 0;

    /**
     * Gets the minimum size of this component.
     */
    virtual MSize2D getMinimumSize() = 0;

    /**
     * Sets the minimum size of this component.
     */
    virtual void setMinimumSize(const MSize2D &size) = 0;

    /**
     * Finds out if the minimum size has been set or not.
     */
    virtual bool isMinimumSizeSet() = 0;

    /**
     * Gets the preferred size of this component.
     */
    virtual MSize2D getPreferredSize() = 0;

    /**
     * Sets the preferred size of this component.
     */
    virtual void setPreferredSize(const MSize2D &size) = 0;

    /**
     * Finds out if the preferred size has been set or not.
     */
    virtual bool isPreferredSizeSet() = 0;

    /**
     * Gets the window Rectangle bounding this component.
     */
    virtual MRect2D getWindowRect() = 0;

    /**
     * Sets the window rectangle of the comonent.
     */
    virtual void setWindowRect(const MRect2D &rect) = 0;

    /**
     * Tells the control to repaint itself on the screen.
     */
    virtual void refresh() = 0;

    /**
     * Sets wether or not the component can receive mouse or keyboard input.
     */
    virtual void setEnabled(bool enabled) = 0;

    /**
     * This sets up this control to always receive mouse events or not.
     *
     * @param capturing True if this contrl will receive all the mouse events 
     *                  from now on, false to set the mouse events to go to 
     *                  the proper control under the cursor.
     */
    virtual void setMouseCapture(bool capturing) = 0;

    /**
     * Sets the mouse cursor that is displayed when the mouse is inside this component.
     */
    virtual void setMouseCursor(MMouseCursor cursor) = 0;

    /**
     * Sets the background colour for the widget. This may effect different 
     * parts of the component depending on what compont it actually is.
     */
    virtual void setBackgroundColour(const MColour &colour) = 0;

    /**
     * Sets the background colour for the widget. This may effect different 
     * parts of the component depending on what compont it actually is.
     */
    virtual void setForegroundColour(const MColour &colour) = 0;

    /**
     * This happens when the mouse pointer moves
     */
    virtual bool onMouseMove(const MMouseEvent &event) = 0;

    /**
     * This happens when a mouse button was depressed.
     *
     * @param event the Mouse event 
     */
    virtual bool onMousePressed(const MMouseEvent &event) = 0;

    /**
     * This happens when a mouse button was released.
     *
     * @param event the Mouse event 
     */
    virtual bool onMouseReleased(const MMouseEvent &event) = 0;

    /**
     * This happens when a mouse button was double clicked.
     *
     * @param event the Mouse event 
     */
    virtual bool onMouseDoubleClick(const MMouseEvent &event) = 0;

    /**
     * This happens when a mouse wheel was used.
     *
     * @param event the Mouse event 
     */
    virtual bool onMouseWheel(const MMouseEvent &event) = 0;

    /**
     * This happens when the window is in the process of being resized
     */
    virtual bool onResize(int newWidth, int newHeight) = 0;

    /**
     * The is called when a component or window is shown.
     */
    virtual bool onShow() = 0;

    /**
     * This is called when a paint event occurs
     */
    virtual bool onPaint() = 0;

    /**
     * This is called when the underlying framework has created a physical 
     * window. The calling of this method basically indicates that this 
     * component is ready to have things done to it. For example, if it is a 
     * container, controls may be added to it, things like that.
     */
    virtual void onCreate() = 0;

    /**
     * This is called when a key has been pressed in a component, or if a key 
     * has been held down, and the keystroke is being repeated.
     */
    virtual bool onKeyPressed(const MKeyEvent &event) = 0;

    /**
     * This is called when a key has been released on the keyboard.
     */
    virtual bool onKeyReleased(const MKeyEvent &event) = 0;

    virtual void addMouseEventListener(const MMouseEventListenerPtr &listener) = 0;
    virtual void addKeyEventListener(const MKeyEventListenerPtr &listener) = 0;
  };

}


/**
 * The following header is included via the AZTEC_GUI
 * variable that should be defined for each platform
 * and used to specify an include directory
 * (ie. ../AztecGUICommon/include/gui/$(AZTEC_GUI)
 * when compiling this lib
 */
#include <MComponentImpl.h>


#include <gui/MContainer.h>


#endif

