#include <AztecGUICommonPCH.h>

#include <gui/scripting/AztecGUIScripting.h>
#include <gui/MComponent.h>

namespace Aztec {

  static JSBool js_MComponent_constructor(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_setVisible(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_setMinimumSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_getMinimumSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_setBackgroundColour(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_addMouseListener(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_addKeyListener(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_setPosition(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_setSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_getPosition(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
  static JSBool js_getSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);

  static void js_MComponent_finalize(JSContext *cx, JSObject *obj);
  
  
  JSClass js_MComponent_class = {
    "MComponent", JSCLASS_HAS_PRIVATE,
      JS_PropertyStub, JS_PropertyStub,
      JS_PropertyStub, JS_PropertyStub,
      JS_EnumerateStub, JS_ResolveStub,
      JS_ConvertStub,   js_MComponent_finalize
  };
  
  JSFunctionSpec js_MComponent_methods[] = {
    // Object/mesh creation
    {"setVisible",            js_setVisible,            1},
    {"getMinimumSize",        js_getMinimumSize,        0},
    {"setMinimumSize",        js_setMinimumSize,        2},
    {"setBackgroundColour",   js_setBackgroundColour,   3},
    {"setPosition",           js_setPosition,           2},
    {"setSize",               js_setSize,               2},
    {"getPosition",           js_getPosition,           0},
    {"getSize",               js_getSize,               0},
    {"addMouseListener",      js_addMouseListener,      3},
    {"addKeyListener",        js_addKeyListener,        3},
    {NULL,                    NULL,                     0} 
  };
  
  JSObject* initMComponentClass(JSContext *cx) {
    JSObject *cl = JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
      &js_MComponent_class, js_MComponent_constructor, 0,
      NULL, js_MComponent_methods, 0, 0);

    return cl;
    
  }

  static JSBool js_MComponent_constructor(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    return js_generic_constructor(cx, obj, new MComponent());
  }
 
  static JSBool js_setVisible(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
  {
    if (argc != 1) {
      return JS_FALSE;
    }
    
    if (!JSVAL_IS_BOOLEAN(argv[0])) {
      return JS_FALSE;
    }
    
    JSBool visible = JSVAL_TO_BOOLEAN(argv[0]);
    
    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      component->setVisible(visible != JS_FALSE);
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }
  
  static JSBool js_setMinimumSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    // Check to see if we only have one arg. If we do, try and convert it to an MSize2D object.
    int32 width = 0, height = 0;

    if (argc == 1) {
      jsval width_value;
      jsval height_value;

      JSObject *obj = JSVAL_TO_OBJECT(argv[0]);
      if (!JS_GetProperty(cx, obj, "width", &width_value)) return JS_FALSE;
      if (!JS_GetProperty(cx, obj, "height", &height_value)) return JS_FALSE;

      if (!JS_ValueToInt32(cx, width_value, &width)) return JS_FALSE;
      if (!JS_ValueToInt32(cx, height_value, &height)) return JS_FALSE;
    } else if (argc == 2) {
      if (!JS_ValueToInt32(cx, argv[0], &width)) return JS_FALSE;
      if (!JS_ValueToInt32(cx, argv[1], &height)) return JS_FALSE;
    } else {
      return JS_FALSE;
    }
    
    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      component->setMinimumSize(MSize2D(width, height));
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_getMinimumSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc != 0) {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      *rval = OBJECT_TO_JSVAL(createMSize2D(cx, NULL, component->getMinimumSize()));
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_setBackgroundColour(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc != 3) {
      return JS_FALSE;
    }
    
    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      jsdouble r, g, b;
      
      if (!JS_ValueToNumber(cx, argv[0], &r)) return JS_FALSE;
      if (!JS_ValueToNumber(cx, argv[1], &g)) return JS_FALSE;
      if (!JS_ValueToNumber(cx, argv[2], &b)) return JS_FALSE;
      
      component->setBackgroundColour(MColour(r, g, b));

      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_setPosition(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc < 2 || argc > 4) {
      return JS_FALSE;
    }
    
    int32 x, y, width, height;

    if (!JS_ValueToInt32(cx, argv[0], &x)) return JS_FALSE;
    if (!JS_ValueToInt32(cx, argv[1], &y)) return JS_FALSE;

    if (argc >= 3) {
      if (!JS_ValueToInt32(cx, argv[2], &width)) return JS_FALSE;
    }
    if (argc >= 4) {
      if (!JS_ValueToInt32(cx, argv[3], &height)) return JS_FALSE;
    }
    
    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      if (width > -1 && height > -1) {
        component->setPosition(x, y, width, height);
      } else {
        component->setPosition(x, y);
      }
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_setSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    int32 width, height;

    // if we only have one argument, it wil be an MSize2D object.
    if (argc == 1 && JSVAL_IS_OBJECT(argv[0])) {
      jsval width_value;
      jsval height_value;

      JSObject *obj = JSVAL_TO_OBJECT(argv[0]);
      if (!JS_GetProperty(cx, obj, "width", &width_value)) return JS_FALSE;
      if (!JS_GetProperty(cx, obj, "height", &height_value)) return JS_FALSE;

      if (!JS_ValueToInt32(cx, width_value, &width)) return JS_FALSE;
      if (!JS_ValueToInt32(cx, height_value, &height)) return JS_FALSE;

    // otherwise we will have 2 integers
    } else if (argc == 2) {
      if (!JS_ValueToInt32(cx, argv[0], &width)) return JS_FALSE;
      if (!JS_ValueToInt32(cx, argv[1], &height)) return JS_FALSE;
    } else {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      component->setSize(width, height);
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_getPosition(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc != 0) {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      *rval = OBJECT_TO_JSVAL(createMSize2D(cx, NULL, component->getSize()));
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  static JSBool js_getSize(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc != 0) {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    
    if (component != NULL) {
      *rval = OBJECT_TO_JSVAL(createMSize2D(cx, NULL, component->getSize()));
      return JS_TRUE;
    } else {
      return JS_FALSE;
    }
  }

  class JSMouseListener : public MMouseEventListener {
  public:
    JSMouseListener(JSContext *cx, JSObject *obj, JSFunction *function, uintN argc, jsval *argv) {
      this->cx = cx;
      this->func = function;
      this->object = obj;
      for (uintN i = 0; i < argc; ++i) {
        args.push_back(argv[i]);
      }
    }

    void onMouseEvent(const MComponentPtr &component, const MMouseEvent &event) {

      jsval rval;

      jsval *argv = new jsval[args.size() + 1];

      argv[0] = OBJECT_TO_JSVAL(createMouseEvent(cx, NULL, event));
      for (int i = 0; i < args.size(); ++i) {
        argv[i+1] = args[i];
      }

      JS_CallFunction(cx, object, func, args.size() + 1, argv, &rval);

      delete[] argv;
    }

    JSContext *cx;
    JSFunction *func;
    JSObject *object;
    std::vector<jsval> args;
  };

  static JSBool js_addMouseListener(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc < 1) {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    JSFunction *func = JS_ValueToFunction(cx, argv[0]);
    
    if (func != NULL && component != NULL) {

      component->addMouseEventListener(new JSMouseListener(cx, obj, func, argc - 1, argv + 1));

      return JS_TRUE;
    } else {
      return JS_FALSE;
    }


    return JS_FALSE;
  }

  class JSKeyListener : public MKeyEventListener {
  public:
    JSKeyListener(JSContext *cx, JSObject *obj, JSFunction *function, uintN argc, jsval *argv) {
      this->cx = cx;
      this->func = function;
      this->object = obj;
      for (uintN i = 0; i < argc; ++i) {
        args.push_back(argv[i]);
      }
    }

    void onKeyEvent(const MComponentPtr &component, const MKeyEvent &event) {

      jsval rval;

      jsval *argv = new jsval[args.size() + 1];

      argv[0] = OBJECT_TO_JSVAL(createKeyEvent(cx, NULL, event));
      for (int i = 0; i < args.size(); ++i) {
        argv[i+1] = args[i];
      }

      JS_CallFunction(cx, object, func, args.size() + 1, argv, &rval);

      delete[] argv;
    }

    JSContext *cx;
    JSFunction *func;
    JSObject *object;
    std::vector<jsval> args;
  };


  static JSBool js_addKeyListener(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval) {
    if (argc < 1) {
      return JS_FALSE;
    }

    MComponentPtr component = getGuiObject<MComponent>(cx, obj);
    JSFunction *func = JS_ValueToFunction(cx, argv[0]);
    
    if (func != NULL && component != NULL) {

      component->addKeyEventListener(new JSKeyListener(cx, obj, func, argc - 1, argv + 1));

      return JS_TRUE;
    } else {
      return JS_FALSE;
    }


    return JS_FALSE;
  }

  static void js_MComponent_finalize(JSContext *cx, JSObject *obj) {
    JS_GUIObject *component = (JS_GUIObject*)JS_GetPrivate(cx, obj);

    if (component != NULL) {
      delete component;
    }

    JS_SetPrivate(cx, obj, 0);
  }

}
