#ifndef MSceneRendererH
#define MSceneRendererH

#include "ModelGeneric.h"

namespace Aztec {

  class MSceneRenderer;
  typedef MRefCountedPtr<MSceneRenderer> MSceneRendererPtr;

}

#include <MBaseObject.h>
#include <MScene.h>
#include <MImages.h>

namespace Aztec {

  class MGENEXPORT MSceneRenderer : public MShapeObject {
  public:
    class MGENEXPORT UpdateListener : public MRefCountedObject {
    public:
      /**
       * This is called from the renderers rendering thread. This should
       * notify the user interface that things have changed, and update any
       * current image and current progress.
       *
       * @param renderer The renderer that is informing about the update.
       * @param amountCompleted the amount of the image completed in a 
       *                        range of 0 to 1.
       * @param image The most recent image from the renderer. This may be 
       *              NULL, and if so, the user interface should retrieve the 
       *              most recent image from the MSceneRenderer::getImage()
       *              method.
       */
      virtual void onUpdate(const MSceneRendererPtr &renderer, 
                            float amountCompleted, 
                            const MImagePtr &image) = 0;

    protected:
      UpdateListener();
      virtual ~UpdateListener();
    };

    typedef MRefCountedPtr<UpdateListener> UpdateListenerPtr;

    MSceneRenderer();
    ~MSceneRenderer();

    /**
     * Gets the name of the renderer.
     */
    virtual std::string getName() = 0;

    /**
     * This tells the renderer to initialise itself, and prepare itself to 
     * start rendering. This should make sure any support files are ready, and
     * report any errors or warnings gracefully.
     */
    virtual void initialise() = 0;

    /**
     * Tells the renderer to commence rendering. The render must do the 
     * following things in this order:
     *   1. Make a copy of the scene contents (Either an exact copy, or convert
     *      it to a scene language the renderer understands. It doesn't matter).
     *      This is so that the user can continue using aztec, and it will not 
     *      effect the output of the renderer.
     *   2. Start a new thread which commences rendering.
     *   3. return from this function.
     *
     * While the rendering thread is active, it must periodically call the 
     * listener's onUpdate method so the user can recieve some feedback as to 
     * what is happening.
     *
     * If the renderer fails at any point during the startup process, it should
     * cancel the rendering thread, and return false.
     *
     * @param scene The scene to render
     * @param options The options to render the scene with. This contains 
     *                both the generic options, such as outputWidth, 
     *                antialiasing, and other things, as well as plugin 
     *                sepcific options.
     * @param listener The listener that recieves update messages from the 
     *                 renderer when updates in the result image happen.
     * @return true if the render started okay, false if it did not.
     */
    virtual bool start(const MScenePtr &scene, 
                       const MParameterObjectListPtr &options, 
                       const UpdateListenerPtr &listener) = 0;

    /**
     * This tells the renderer to stop where it is, and terminate rendering. 
     * This must :
     *    1. Stop rendering.
     *    2. All the listeners onUpdate() method one last time to supply it 
     *       with the most recent image.
     *    3. Exit the rendering thread.
     */
    virtual void stop() = 0;

    /**
     * This retrieves the most recent image that the renderer has generated.
     * If a render is halfway through, this must return a valid image, that 
     * can be returned in a thread safe manner. 
     *
     * @return The last rendered image.
     */
    virtual MImagePtr getImage() = 0;

    /** 
     * This returns a list which contains all the possible options for a 
     * renderer, including default values. Each renderer need not worry about 
     * permanent storage of these values, as the server (Aztec) will handle 
     * this aspect of it by itself.
     */
    virtual MParameterObjectListPtr getOptions() = 0;
  };

}
#endif
