#ifndef MTextFileStream_Header
#define MTextFileStream_Header

#include "ModelGeneric.h"

namespace Aztec {
  class MTextFileReader;
  class MTextFileWriter;

  typedef MRefCountedPtr<MTextFileReader> MTextFileReaderPtr;
  typedef MRefCountedPtr<MTextFileWriter> MTextFileWriterPtr;

}

#include "MStreams.h"

#include <stdio.h>

namespace Aztec {

  class MGENEXPORT MTextFileReader : public MInputStream {
  public:
    /**
     * Flag that means any quotes read from the file
     * indicate that it is a start of a single string.
     */
    static const int QUOTED_STRINGS;

    /**
     * Default constructor, with no stream being opened.
     */
    MTextFileReader();
    /**
     * Constructor with a given filename, and attempts to open
     * the given file for reading.
     */
    MTextFileReader(const MStr &filename, int flags = QUOTED_STRINGS);

    /**
     * destructor
     */
    virtual ~MTextFileReader();

    /**
     * opens the stream to read from the given filename
     */
    bool open(const MStr &filename);
    /**
     * Closes the current stream.
     */
    void close();
    /**
     * Returns true if the stream is currently open
     */
    bool isOpen();
    /**
     * Returns true if we are currently at the end of the stream.
     */
    bool isAtEOF();

    /**
     * reads in from the current position until
     * it hits an end of line character
     */
    MStr readLine();

    // methods from MInputStream
    char readChar();
    unsigned char readUnsignedChar();
    short readShort();
    unsigned short readUnsignedShort();
    int readInt();
    unsigned int readUnsignedInt();
    float readFloat();
    double readDouble();
    bool readBoolean();
    MStr readString();
    int readBuffer(void *buffer, int bufSize);

  protected:
    /**
     * Internal file handle to use
     */
    FILE *m_FileHandle;

    /**
     * internal variable to keep track of reading options.
     * It is a binary or-ing of the flags listed at the top of
     * this class.
     */
    unsigned int m_Flags;
  };

  /**
   * This is a class that writes text files. 
   */
  class MGENEXPORT MTextFileWriter : public MOutputStream {
  public:
    /**
     * Default constructor, with no stream being opened.
     */
    MTextFileWriter();
    /**
     * Constructor with a given filename, and attempts to open
     * the given file for writing.
     */
    MTextFileWriter(const MStr &filename, bool append = false);
    /**
     * destructor
     */
    virtual ~MTextFileWriter();

    /**
     * opens the stream to write to from the given filename.
     * It will append the file if required, otherwise it
     * will create/overwrite it.
     *
     * @param filename The filename of th file to open
     * @param append This appends the file if this is 
     *               true, false otherwise
     * @return true if the file was sucessfully opened.
     */
    bool open(const MStr &filename, bool append = false);
    /**
     * Clsoes the current stream.
     */
    void close();
    /**
     * Returns true if the stream is currently open
     */
    bool isOpen();
    /**
     * Returns true if we are currently at the end of the stream.
     */
    bool isAtEOF();

    // methods from MOutputStream
    void writeChar(char value);
    void writeUnsignedChar(unsigned char value);
    void writeShort(short value);
    void writeUnsignedShort(unsigned short value);
    void writeInt(int value);
    void writeUnsignedInt(unsigned int value);
    void writeFloat(float value);
    void writeDouble(double value);
    void writeBoolean(bool value);
    void writeString(const MStr &value);
    void writeString(const char *value);
    void writeBuffer(const void *buffer, int bufSize);


  protected:
    /**
     * Internal file handle
     */
    FILE *m_FileHandle;

  };

}

#endif
