#ifndef MTimeSegment_Header
#define MTimeSegment_Header

#include <ModelGeneric.h>

namespace Aztec {
  class MTimeSegment;
  typedef MRefCountedPtr<MTimeSegment> MTimeSegmentPtr;
}

#include <MNamedObject.h>
#include <params/MIntParameter.h>
#include <params/MObjectListParameter.h>

#include <vector>

namespace Aztec {

  /**
   * An MTimeSegment is responsible for keeping track of
   * a single sgment of time. That is, a bit of time from 
   * one point int time to another. These objects are used
   * to logically group shunks of time together in meaninful 
   * ways. Each time segment can have an associated group
   * of objects. This allows groups of animations of objects
   * to be named so to speak.
   */
  class MGENEXPORT MTimeSegment : public MNamedObject {
  public:
    typedef MTimeSegmentPtr Ptr;

    // Construction/Destruction
    MTimeSegment();
    ~MTimeSegment();

    /**
     * This sets the start and end time
     *
     * @param tickStart The new starting time for the segment
     * @param tickEnd The new ending time for the segment
     */
    void setRange(int tickStart, int tickEnd);

    /**
     * Gets the beginning time of the segment
     */
    int getStart();

    /**
     * Gets the end time of the segment.
     */
    int getEnd();

    /**
     * Gets the number of objects in this group
     *
     * @return the number of objects this segment represents
     */
    int getObjectCount();

    /**
     * Gets the object at the nth position
     *
     * @param index The index of the object to get
     * @return the object at the given index
     */
    MNamedObjectPtr getObject(int index);

    /**
     * This gets all the objects this segment contains
     *
     * @param objects The vector to receive the objects
     */
    void getObjects(std::vector<MNamedObjectPtr> &objects);

    /**
     * Adds the given object to the segment. It will not
     * add the object if it is already there.
     *
     * @param object The object to add
     */
    void addObject(MNamedObjectPtr object);

    /**
     * This removes the given object from the segment.
     *
     * @param object the object to remove
     */
    void removeObject(MNamedObjectPtr object);

    /**
     * This removes the object at the given index
     *
     * @param index The index of the object to remove.
     */
    void removeObject(int index);

    /**
     * Finds out if the given object is part of the 
     * segment or not.
     *
     * @param object The object to test
     * @return true if the object is in this segment, false
     *         if it is not.
     */
    bool isObjectInSegment(MNamedObjectPtr object);

    // MBaseObject methods
    MStr getClassName() {return MStr("MTimeSegment");};
    MStr getParentClassName() {return MStr("MNamedObject");};
    MBaseObjectPtr createNew();
    void setFrom(MBaseObjectPtr SrcObj);

  protected:
    MIntParameterPtr m_StartParam;
    MIntParameterPtr m_EndParam;
    MObjectListParameterPtr m_ObjectsParam;
    MStringParameterPtr m_SegmentName;

  };
  

}
#endif
