#include "StdAfx.h"
#include "MCompleteTranslator.h"
#include "MSystemManager.h"
#include "MBinaryFileStream.h"
#include "MTextFileStream.h"
#include <MVector3KeyList.h>

#include <params/MArrayParameter.h>
#include <params/MVector3KeyParameter.h>
#include <ctype.h>
#include <assert.h>

#define  MCOMPLETETRANSLATOR_VERSION      ((WORD)0x0104)

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// We need to define a special assert statement that executes the code even
// if we are not in debug mode.
#ifdef _DEBUG 
#define CheckAssert(x) assert((x))
#else
#define CheckAssert(x) (x)
#endif

namespace Aztec {
  
  MCompleteTranslator::MCompleteTranslator() {
  }
  
  MCompleteTranslator::~MCompleteTranslator() {
  }

  MStr MCompleteTranslator::getClassName() {
    return MStr("MCompleteTranslator");
  }

  MStr MCompleteTranslator::getParentClassName() {
    return MStr("MSceneTranslator");
  }

  MTranslatorPtr MCompleteTranslator::createNew() const {
    return NULL;
  }
  
  std::string MCompleteTranslator::getFilter() {
    return "*.AztecAscii";
  }

  std::string MCompleteTranslator::getFilterDescription() {
    return "Aztec Scene Files";
  }

  bool MCompleteTranslator::canImport() {
    return true;
  }

  bool MCompleteTranslator::canExport() {
    return true;
  }

  class SceneWriter {
  public:
    SceneWriter(MOutputStream &out) 
      : stream(out)
    {
    }
    ~SceneWriter() { }

    typedef enum { EQUALS, LINKED_FROM } OperatorType;

    virtual void beginGroup(const std::string &type, const std::string &name) = 0;
    virtual void endGroup() = 0;

    virtual void writeValue(int value, bool writeTypeName = true) = 0;
    virtual void writeValue(float value, bool writeTypeName = true) = 0;
    virtual void writeValue(const std::string &value, bool writeTypeName = true) = 0;
    virtual void writeValue(const char *value, bool writeTypeName = true) = 0;
    virtual void writeValue(bool value, bool writeTypeName = true) = 0;
    virtual void writeValue(const MVector3 &value, bool writeTypeName = true) = 0;
    virtual void writeValue(const MMatrix4 &value, bool writeTypeName = true) = 0;

    virtual void writeName(const char* value, OperatorType oper = EQUALS) = 0;
    virtual void writeName(const std::string &value, OperatorType oper = EQUALS) = 0;

    virtual void beginValue() = 0;
    virtual void endValue() = 0;
  protected:
    MOutputStream &stream;
  };

  class AsciiSceneWriter : public SceneWriter {
  public:
    AsciiSceneWriter(MOutputStream &out) 
      : SceneWriter(out), 
        currentTabs(0),
        valueNameJustWritten(false),
        newLineStarted(true),
        spaceJustWritten(true)

    {
    }

    // SceneWriter methods
    void beginGroup(const std::string &type, const std::string &name) {
      if (name.length() > 0) {
        beginLine(type + " " + name + " {");
        endLine();
      } else {
        beginLine(type + " {");
        endLine();
      }
      ++currentTabs;
    }

    void endGroup() {
      assert(currentTabs > 0);
      --currentTabs;
      beginLine("}");
      endLine();
    }

    virtual void writeValue(int value, bool writeTypeName) {
      beginLine();
      if (writeTypeName) {
        writeString("int ");
      }
      char buf[32];
      sprintf(buf, "%i", value);
      writeString(buf);
    }

    virtual void writeValue(float value, bool writeTypeName) {
      beginLine();
      if (writeTypeName) {
        writeString("float ");
      }
      char buf[32];
      sprintf(buf, "%.5f", value);
      writeString(buf);
    }

    virtual void writeValue(const std::string &value, bool writeTypeName) {
      beginLine();

      if (writeTypeName) {
        writeString("string ");
      }

      if (value.length() == 0) {
        writeString("\"\"");
      } else {
        bool doQuotes = false;
        if (value.find(" ") != std::string::npos) {
          doQuotes = true;
        }

        if (doQuotes) {
          writeString("\"");
        }
        writeString(value);
        if (doQuotes) {
          writeString("\"");
        }
      }
    }

    virtual void writeValue(const char *value, bool writeTypeName) {
      beginLine();

      if (writeTypeName) {
        writeString("string ");
      }

      if (value == NULL || strlen(value) == 0) {
        writeString("\"\"");
      } else {
        bool doQuotes = false;
        if (strstr(value, " ") != NULL) {
          doQuotes = true;
        }

        if (doQuotes) {
          writeString("\"");
        }
        writeString(value);
        if (doQuotes) {
          writeString("\"");
        }
      }
    }

    virtual void writeValue(bool value, bool writeTypeName) {
      beginLine();
      if (writeTypeName) {
        writeString("bool ");
      }
      if (value) {
        writeString("true");
      } else {
        writeString("false");
      }
    }

    virtual void writeValue(const MVector3 &value, bool writeTypeName) {
      beginLine();
      if (writeTypeName) {
        writeString("vector3 ");
      }
      char buf[256];
      sprintf(buf, "%.5f %.5f %.5f", value.x, value.y, value.z);
      writeString(buf);
    }

    virtual void writeValue(const MMatrix4 &value, bool writeTypeName) {
      beginLine();
      if (writeTypeName) {
        writeString("matrix4 ");
      }
      char buf[256];

      for (int i = 0; i < 4; ++i) {
        sprintf(buf, "%.5f %.5f %.5f %.5f ", value.m[i][0], value.m[i][1], value.m[i][2], value.m[i][3]); 
        writeString(buf);
      }
    }

    virtual void writeName(const char* value, OperatorType oper) {
      if (strlen(value) > 0) {
        writeString(value);
        writeSpace();
        writeString(getSymbol(oper));
        writeSpace();
      }
      valueNameJustWritten = true;
    }

    virtual void writeName(const std::string &value, OperatorType oper) {
      if (value.length() > 0) {
        writeString(value);
        writeSpace();
        writeString(getSymbol(oper));
        writeSpace();
      }
      valueNameJustWritten = true;
    }

    virtual void beginValue() {
      beginLine();
    }

    virtual void endValue() {
      // only bother startnig anew line if we have't alrady done so.
      if (!newLineStarted) {
        endLine();
      }
    }

    void beginLine(const std::string &line) {
      doIndent();
      stream.writeString(line.c_str());
    }

    void beginLine(const char *line) {
      doIndent();
      stream.writeString(line);
    }

    void beginLine() {
      doIndent();
    }

    void endLine() {
      writeString("\n");
      newLineStarted = true;
    }

  private:
    void doIndent() {
      // if we have just written a value name or if we aren't at the start of 
      // a line, then don't worry about tabs
      if (!valueNameJustWritten && newLineStarted) {
        for (int i = 0; i < currentTabs; ++i) {
          stream.writeString("\t");
        }
        newLineStarted = false;
        spaceJustWritten = true;
      } else {
        if (!newLineStarted) {
          writeSpace();
        }

        valueNameJustWritten = false;
        newLineStarted = false;
      }
    }

    void writeString(const std::string &value) {
      stream.writeString(value.c_str());
      newLineStarted = false;
    }

    void writeString(const char *value) {
      if (value != NULL) {
        stream.writeString(value);
      }
      newLineStarted = false;
      spaceJustWritten = false;
    }

    void writeSpace() {
      if (!spaceJustWritten) {
        stream.writeString(" ");
        spaceJustWritten = true;
      }
    }

    const char *getSymbol(OperatorType oper) {
      switch (oper) {
      case EQUALS:
        return "=";
      case LINKED_FROM:
        return "<-";
      }
      assert(0);
      return NULL;
    }

    int currentTabs;
    bool valueNameJustWritten;
    bool newLineStarted;
    bool spaceJustWritten;
  };

  bool writeValueFloat(SceneWriter &writer, const MFloatValuePtr &value, const MScenePtr &scene) {
    if (value == NULL) {
      return false;
    }

    MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, value);
    
    // if we have a keylist, and we have keys, then write them out.
    // otherwise just write out a single value
    if (keyList != NULL && keyList->getKeyCount() > 0) {
      writer.beginGroup("keys", "");

      writer.beginValue();
      writer.writeName("count");
      writer.writeValue(keyList->getKeyCount(), false);
      writer.endValue();

      for (int i = 0; i < keyList->getKeyCount(); ++i) {
        MFloatKeyPtr floatKey = AZTEC_CAST(MFloatKey, keyList->getKeyAtIndex(i));

        assert(floatKey != NULL);

        writer.beginValue();
        writer.writeValue((float)floatKey->getKeyTime(), false);
        writer.writeValue(floatKey->getValue(), false);
        writer.writeValue(floatKey->getInTangent(), false);
        writer.writeValue(floatKey->getOutTangent(), false);
        writer.endValue();
      }

      writer.endGroup();
    } else {
      // just write out the current value. The time shouldn't matter, as
      // the value of this float shouldn't change.
      writer.writeValue(value->getValueAtTime(scene->getTime()));
    }

    return true;
  }

  bool writeVector3KeyList(SceneWriter &writer, const MKeyableValuePtr &keyValue, const MScenePtr &scene) {
    MVector3KeyListPtr keyList = AZTEC_CAST(MVector3KeyList, keyValue);

    if (keyList == NULL) {
      return false;
    }

    writer.beginGroup("vector3keylist", "");

    writer.beginValue();
    writer.writeName("x");
    writeValueFloat(writer, keyList->getXValue(), scene);
    writer.endValue();

    writer.beginValue();
    writer.writeName("y");
    writeValueFloat(writer, keyList->getYValue(), scene);
    writer.endValue();

    writer.beginValue();
    writer.writeName("z");
    writeValueFloat(writer, keyList->getZValue(), scene);
    writer.endValue();

    writer.endGroup();

    return true;
  }

  bool writeKeyList(SceneWriter &writer, const MParameterObjectPtr &param, const MScenePtr &scene) {
    // Check to see if we have a keyable parameter, and if we do, and we 
    // have keys set, then write them out.
    MKeyParameterPtr keyParam = AZTEC_CAST(MKeyParameter, param);
    MKeyableValuePtr keyValue;
    std::vector<int> keyTimes;
    
    if (keyParam != NULL) {
      keyValue = keyParam->getKeyableValue();
    }

    if (keyValue != NULL) {
      keyValue->getKeyTimes(keyTimes);
    }

    // do nothing if we have no keys set
    if (keyTimes.size() == 0) {
      return false;
    }
    
    // attempt to write it out as a vector3 key list.
    if (writeVector3KeyList(writer, keyValue, scene)) {
      return true;
    }

    // attempt to write it out as an animatable float value.
    if (writeValueFloat(writer, AZTEC_CAST(MFloatValue, keyValue), scene)) {
      return true;
    }

    return false;
  }

  void writeOutParam(SceneWriter &writer, const MParameterObjectPtr &srcParam, const MScenePtr &scene, bool writeName = true) {
    // if the parameter isn't storable, then just do nothing.
    if (!srcParam->isStorable()) {
      return;
    }

    // if our value is directed by another value, then we write that
    // instead of a simply assignemnt one.
    writer.beginValue();

    if (srcParam->isConnected() && srcParam->getInputParameter()->getOwner() != NULL) {

      writer.writeName(srcParam->getLongName(), SceneWriter::LINKED_FROM);

      std::string value;

      MNamedObject *namedObj = AZTEC_CAST(MNamedObject, srcParam->getInputParameter()->getOwner());

      assert(namedObj != NULL);

      value = namedObj->getName();
      value += ".";
      value += srcParam->getInputParameter()->getLongName();

      writer.writeValue(value, false);

    } else {
      MParameterObjectPtr param = srcParam;

      if (writeName) {
        writer.writeName(param->getLongName());
      } else {
        writer.writeName("");
      }


      // if we are here, but the parameter is connected, it means the
      // connected parameter had no owner, which means we cannot reference it.
      // So isntead of inserting a bogus reference to it, we just write out
      // the values of the input parameter instead.
      if (srcParam->isConnected()) {
        param = srcParam->getInputParameter();
      }

      int paramType = param->getDataType();

      // first attempt to write the parameter as an animated one.

      if (writeKeyList(writer, param, scene)) {

      } else if (paramType == MParameterObject::TYPE_INTEGER) {
        int value;
        param->getValueInteger(value);
        writer.writeValue(value);
      } else if (paramType == MParameterObject::TYPE_FLOAT) {
        float value;
        param->getValueFloat(value);
        writer.writeValue(value);
      } else if (paramType == MParameterObject::TYPE_BOOLEAN) {
        bool value;
        param->getValueBoolean(value);
        writer.writeValue(value);
      } else if (paramType == MParameterObject::TYPE_STRING ||
                 paramType == MParameterObject::TYPE_OBJECT) {
        MStr value;
        param->getValueString(value);
        writer.writeValue(value.c_str());
      } else if (paramType == MParameterObject::TYPE_VECTOR) {
        MVector3 value;
        param->getValueVector(value);
        writer.writeValue(value);
      } else if (paramType == MParameterObject::TYPE_ARRAY) {
        MArrayParameterPtr array = AZTEC_CAST(MArrayParameter, param);

        assert(array != NULL);
        writer.beginGroup("array", "");

        writer.beginValue();
        writer.writeName("count");
        writer.writeValue(array->getElementCount(), false);
        writer.endValue();

        // write out each parameter of the array
        for (int index = 0; index < array->getElementCount(); ++index) {
          writeOutParam(writer, array->getElement(index), scene, false);
        }

        writer.endGroup();
      } else if (paramType == MParameterObject::TYPE_AGGREGATE) {
        MAggregateParameterPtr aggregate = AZTEC_CAST(MAggregateParameter, param);

        assert(aggregate != NULL);
        writer.beginGroup("aggregate", "");

        // do use the parameter instead.
        for (int index = 0; index < aggregate->getMemberCount(); ++index) {

          MParameterObjectPtr element = aggregate->getMember(index);

          writeOutParam(writer, element, scene);
        }

        writer.endGroup();
      } else if (paramType == MParameterObject::TYPE_MATRIX) {
        MMatrix4ParameterPtr matrixParam = AZTEC_CAST(MMatrix4Parameter, param);
        MMatrix4 value;
        value.identity();
        matrixParam->getValueMatrix(value);

        writer.writeValue(value);
      } else {
        writer.writeValue("[Unknown]");
      }

    }

    writer.endValue();
  }

  static void writeOutObject(SceneWriter &writer, const MNamedObjectPtr &object, const MScenePtr &scene) {
    // iterate over all the parameters of the object, and store them.
    MParameterObjectListPtr paramList = object->getParamList();

    for (int index = 0; index < paramList->getNumParams(); ++index) {
      writeOutParam(writer, paramList->getParameter(index), scene);
    }
  }

  typedef enum { WriteBegun, WriteFinished } WriteState;
  typedef std::map<MBaseObjectPtr, WriteState> ObjectSet;

  void setObjectWriteFinished(ObjectSet &objects, const MBaseObjectPtr &object) {
    std::pair<ObjectSet::iterator, bool> result = objects.insert(ObjectSet::value_type(object, WriteFinished));

    if (result.second == false) {
      result.first->second = WriteFinished;
    }
  }
  void setObjectWriteBegun(ObjectSet &objects, const MBaseObjectPtr &object) {
    std::pair<ObjectSet::iterator, bool> result = objects.insert(ObjectSet::value_type(object, WriteBegun));

    if (result.second == false) {
      result.first->second = WriteBegun;
    }
  }

  static void doWriteObject(SceneWriter &writer, 
                            const MNamedObjectPtr &object, 
                            const MScenePtr &scene,
                            ObjectSet &objectsWritten);

  static void writeInputs(SceneWriter &writer, 
                          MDAGNode *object, 
                          const MScenePtr &scene,
                          ObjectSet &objectsWritten) {

    ObjectSet::iterator objectIt = objectsWritten.find(object);
    if (objectIt != objectsWritten.end()) {
      return;
    }

    setObjectWriteBegun(objectsWritten, object);

    MDAGNode **inputs = new MDAGNode*[object->getInputCount()];
    object->getInputs(inputs);
    std::set<MNamedObject*> namedObjsToWrite;

    MDAGNode *input;
    for (int index = 0; index < object->getInputCount(); ++index) {
      input = inputs[index];

      assert(input != NULL);

      // convert this object to a named object to see if we can 
      // write some text about it.
      MNamedObject *namedObj = AZTEC_CAST(MNamedObject, input);

      // if we haven't yet written out this object, then do so
      if (objectsWritten.find(input) == objectsWritten.end()) {
        writeInputs(writer, input, scene, objectsWritten);
        
        setObjectWriteFinished( objectsWritten, input );
        
        // if the input was a named object, then write that out as well.
        if (namedObj != NULL) {
          doWriteObject(writer, namedObj, scene, objectsWritten);
        }
      }

      // if we do not have a named object, we may have a parameter. If we do 
      // have a parameter, then we have to try to write out its owner.
      MParameterObject *param = AZTEC_CAST(MParameterObject, input);

      if (param != NULL) {
        namedObj = AZTEC_CAST(MNamedObject, param->getOwner());
      }
     
      // if it turned out that the input was a parameter, and we need to
      // write out the named object, then do that as well.
      if (namedObj != NULL &&
          input != namedObj && 
          objectsWritten.find(namedObj) == objectsWritten.end()) 
      {
        namedObjsToWrite.insert(namedObj);
      }
    }

    // now write out all the named objects that parameters we have written the inputs to belong to.
    for (std::set<MNamedObject*>::iterator it = namedObjsToWrite.begin(); it != namedObjsToWrite.end(); ++it) {
      writeInputs(writer, *it, scene, objectsWritten);
      setObjectWriteFinished( objectsWritten, *it );
      doWriteObject(writer, *it, scene, objectsWritten);
    }


    // convert this object to a named object to see if we can 
    // write some text about it.
    MNamedObject *namedObj = AZTEC_CAST(MNamedObject, object);

    if (namedObj != NULL) {
      doWriteObject(writer, namedObj, scene, objectsWritten);
    }

    delete[] inputs;
  }

  static void doWriteObject(SceneWriter &writer, 
                            const MNamedObjectPtr &object, 
                            const MScenePtr &scene,
                            ObjectSet &objectsWritten) {

    std::string type, name;

    type = object->getClassName().c_str();
    name = object->getName();

    ObjectSet::iterator objectIt = objectsWritten.find(object);

    if (objectIt == objectsWritten.end() || objectIt->second != WriteFinished) {

      writer.beginGroup(type, name);

      if (object != NULL) {
        writeOutObject(writer, object, scene);
        setObjectWriteFinished( objectsWritten, object );
      }

      writer.endGroup();
    }
  }
  static void writeOutScene(SceneWriter &writer, const MScenePtr &scene) {
    writer.beginValue();
    writer.writeName("Aztec_Scene_File_Version");
    writer.writeValue("1.0");
    writer.endValue();

    writer.beginGroup("Scene", "myScene");

    writer.beginGroup("Objects", "");
    // loop over all the objects in the scene and write them all out.
    MBaseObjectTreePtr objects = scene->getObjectList();
    MBaseObjectPtr object;

    ObjectSet objectsWritten;

    objects->beginIteration();
    while ((object = objects->getNext()) != NULL) {
      MTreeObjectNodePtr node = objects->getCurrentNode();

      MNamedObjectPtr namedObject = AZTEC_CAST(MNamedObject, object);

      if (namedObject != NULL) {
        writeInputs(writer, &*namedObject, scene, objectsWritten);
      }
    }
    objects->endIteration();

    writer.endGroup(); // "Objects"

    writer.endGroup(); // "Scene" "myScene"
  }

  bool MCompleteTranslator::exportFile(MStr Filename, MScenePtr Scene) {
//    MBinaryFileWriter outStream;
    MTextFileWriter outStream;

    outStream.open(Filename);

    if (!outStream.isOpen()) {
      getSystemManager()->logOutput("Error: Could not open '%s' for writing.", Filename.c_str());
      return false;
    }
    
    if (Scene == NULL) {
      return 0;
    }
    
    AsciiSceneWriter writer(outStream);
    writeOutScene(writer, Scene);

    outStream.close();
    
    return true;
  }


  class SceneReader {
  public:
    SceneReader(MInputStream &in) 
      : stream(in)
    {
    }
    ~SceneReader() { }

    typedef enum { EQUALS, LINKED_FROM } OperatorType;

    virtual int readInt() = 0;
    virtual float readFloat() = 0;
    virtual bool readBool() = 0;
    virtual std::string readString() = 0;
    virtual MVector3 readVector3() = 0;
    virtual MMatrix4 readMatrix4() = 0;
    virtual OperatorType readOperator() = 0;

    virtual void beginGroup() = 0;
    virtual void beginGroup(std::string &type, std::string &name) = 0;
    virtual void endGroup() = 0;
    virtual bool atEndOfGroup() = 0;

  protected:
    MInputStream &stream;
  };

  void trimWhitespace(MStr &str) {
    for (int i = 0; i < str.GetLength(); ++i) {
      if (isspace(str[i])) {
        str[i] = ' ';
      }
    }

    str.RemoveTrailing(' ');
    str.RemoveLeading(' ');
  }

  class AsciiSceneReader : public SceneReader {
  public:
    AsciiSceneReader(MInputStream &in) 
      : SceneReader(in)
    {
      isAtEndOfGroup = false;
    }

    // MSceneReader methods
    int readInt() {
      return strToInt(readString().c_str());
    }

    float readFloat() {
      return strToFloat(readString().c_str());
    }

    bool readBool() {
      return strToBool(readString().c_str());
    }

    std::string readString() {
      // if we are at the end of the group, do nothing until we
      // are finished.
      if (atEndOfGroup()) {
        return "";
      }

      MStr str = stream.readString();
      trimWhitespace(str);

      // check to see if we have hit a curly brace. If we do
      // mark ourselves being at the end of a group.
      if (str == "}") {
        isAtEndOfGroup = true;
        str = "";
      }

      return str.c_str();
    }

    MVector3 readVector3() {
      bool hasBracket = false;
      std::string values[3];

      values[0] = readString();

      // check to see if we have an opening bracket.
      if (values[0] == "[") {
        hasBracket = true;
        values[0] = readString();
      }

      values[1] = readString();
      values[2] = readString();

      if (hasBracket) {
        readString(); // skip past the ]
      }

      return MVector3(strToFloat(values[0].c_str()), 
                      strToFloat(values[1].c_str()), 
                      strToFloat(values[2].c_str()));
    }

    MMatrix4 readMatrix4() {
      assert(0);
      return MMatrix4();
    }

    OperatorType readOperator() {
      std::string str = readString();
      if (str == "=") {
        return EQUALS;
      } else if (str == "<-") {
        return LINKED_FROM;
      } else {
        assert(0);
        return EQUALS;
      }
    }

    void beginGroup() {
      std::string tempType, tempName;
      beginGroup(tempType, tempName);
    }

    void beginGroup(std::string &type, std::string &name) {
      std::string str;

      // here we have a group that is preceded by 0, 1 or 2 text names, 
      // followed by a curly opening brace. If we encounter a brace at 
      // any time, we assume the names are blank, and move on.
      str = readString();

      if (atEndOfGroup()) {
        return;
      }

      if (str == "{") {
        return;
      }

      type = str;

      str = readString();

      if (str == "{") {
        return;
      }

      name = str;

      str = readString();
      
      // now make sure we have a properly formed group.
      assert(str == "{");
    }

    void endGroup() {
      // do a read string here to make sure that we read the end of group
      // marker. since this does nothing if we are already at the end of
      // the group, it is safe.
      readString();

      // make sure that we are ending a group when we are actually
      // at the end.
      CheckAssert(atEndOfGroup());
      isAtEndOfGroup = false;
    }

    bool atEndOfGroup() {
      if (stream.isAtEOF()) {
        return true;
      }
      return isAtEndOfGroup;
    }

  private:
    bool isAtEndOfGroup;
  };

  class SceneInfo {
  public:
    MScenePtr scene;

    void addObject(const std::string &name, const MNamedObjectPtr &namedObj) {
      namedObjects.insert(NameObjectMap::value_type(name, namedObj));
      scene->addObject(namedObj);
    }

    MNamedObjectPtr getObject(const std::string &name) {
      // if we have an empty name, then we have no object.
      if (name.length() == 0) {
        return NULL;
      }

      NameObjectMap::iterator it = namedObjects.find(name);

      // we should ALWAYS find an object, since the file we are reading
      // in must have been written out in a correct order.
      assert(it != namedObjects.end());

      return it->second;

    }

    typedef std::map<std::string, MNamedObjectPtr> NameObjectMap;

    NameObjectMap namedObjects;

  };

  bool isDataType(const std::string &type) {
    static std::set<std::string> knownTypes;

    if (knownTypes.size() == 0) {
      knownTypes.insert("string");
      knownTypes.insert("bool");
      knownTypes.insert("float");
      knownTypes.insert("int");
      knownTypes.insert("vector3");
      knownTypes.insert("vector3keylist");
      knownTypes.insert("matrix4");
      knownTypes.insert("array");
      knownTypes.insert("aggregate");
      knownTypes.insert("[unknown]");
    }

    return knownTypes.find(type) != knownTypes.end();
  }

  static bool readFloatValue(SceneReader &reader,
                      const MFloatValuePtr &floatValue) {
    std::string type = reader.readString();
    
    // if the type is a key, then read in the key information.
    if (type == "keys") {
      MFloatKeyableValuePtr keyableValue = AZTEC_CAST(MFloatKeyableValue, floatValue);
      MKeyListPtr keyList = AZTEC_CAST(MKeyList, floatValue);

      reader.beginGroup();
      
      CheckAssert(reader.readString() == "count");
      CheckAssert(reader.readOperator() == SceneReader::EQUALS);

      int count = reader.readInt();

      for (int i = 0; i < count; ++i) {
        long time = reader.readInt();
        float value = reader.readFloat();
        float inTangent = reader.readFloat();
        float outTangent = reader.readFloat();

        if (keyableValue != NULL) {
          keyableValue->setKey(value, time);
  
          // if we have a proper key list, then we can use our tangent 
          // information and put that on the key,
          if (keyList != NULL) {
            MFloatKeyPtr floatKey = AZTEC_CAST(MFloatKey, keyList->getKeyAtTime(time));

            floatKey->setInTangent(inTangent);
            floatKey->setOutTangent(outTangent);
          }

        }
      }

      reader.endGroup();


      // otherwise check to see if the data is a float.
    } else if (type == "float") {
      float value = reader.readFloat();

      MFloatKeyableValuePtr keyableValue = AZTEC_CAST(MFloatKeyableValue, floatValue);
      if (keyableValue != NULL) {
        keyableValue->setInitialValue(value);
      }
    } else {
      assert(0);
    }

    return true;
  }

  static bool readVector3KeyParam(SceneReader &reader, 
                       const MParameterObjectPtr &destParam) 
  {
    MVector3KeyParameterPtr keyParam = AZTEC_CAST(MVector3KeyParameter, destParam);
    MVector3KeyListPtr keyList;
    MFloatValuePtr xValue, yValue, zValue;

    if (keyParam != NULL) {
      keyList = AZTEC_CAST(MVector3KeyList, keyParam->getKeyableValue());
    }

    if (keyList != NULL) {
      xValue = keyList->getXValue();
      yValue = keyList->getYValue();
      zValue = keyList->getZValue();
    }

    reader.beginGroup();

    while (!reader.atEndOfGroup()) {
      std::string name = reader.readString();

      if (reader.atEndOfGroup()) {
        break;
      }

      // make sure that we have a valid name.
      assert(name == "x" || name == "y" || name == "z");

      // read in the equals sign, and it must be an equals sign.
      SceneReader::OperatorType oper = reader.readOperator();
      assert(oper == SceneReader::EQUALS);

      if (name == "x") {
        readFloatValue(reader, xValue);
      } else if (name == "y") {
        readFloatValue(reader, yValue);
      } else if (name == "z") {
        readFloatValue(reader, zValue);
      } else {
        assert(0);
      }

    }

    reader.endGroup();

    return true;
  }

  static bool readSpecialData(SceneReader &reader, 
                       const MParameterObjectPtr &destParam,
                       const std::string &type,
                       SceneInfo &info);

  static bool readArray(SceneReader &reader, 
                        const MParameterObjectPtr &destParam,
                        SceneInfo &info) 
  {
    MArrayParameter *array = AZTEC_CAST(MArrayParameter, destParam);

    reader.beginGroup();

    CheckAssert(reader.readString() == "count");
    CheckAssert(reader.readOperator() == SceneReader::EQUALS);

    int count = reader.readInt();

    if (array != NULL) {
      array->resize(count);
    }

    for (int i = 0; i < count; ++i) {
      std::string type = reader.readString();

      // attempt to read in the element, and ail out if we can't.
      MParameterObjectPtr destParam;

      if (array != NULL) {
        destParam = array->getElement(i);
      }

      if (!readSpecialData(reader, destParam, type, info)) {
        return false;
      }
    }

    reader.endGroup();

    return true;
  }

  static bool readAggregate(SceneReader &reader, 
                            const MParameterObjectPtr &destParam,
                            SceneInfo &info) 
  {
    MAggregateParameter *aggregate = AZTEC_CAST(MAggregateParameter, destParam);

    reader.beginGroup();

    while (!reader.atEndOfGroup()) {
      // read in the parameter name
      std::string paramName = reader.readString();

      if (reader.atEndOfGroup()) {
        break;
      }

      // make sure we have an equals operator
      CheckAssert(reader.readOperator() == SceneReader::EQUALS);
      
      // get the type of the element in the aggregate we are about to read.
      std::string type = reader.readString();

      MParameterObjectPtr destParam;
      
      if (aggregate != NULL) {
        destParam = aggregate->getMember(aggregate->getMemberIndex(paramName.c_str()));
      }

      // attempt to read in the data, and bail out if we can't.
      if (!readSpecialData(reader, destParam, type, info)) {
        return false;
      }

    }

    reader.endGroup();
    return true;
  }

  static bool readSpecialData(SceneReader &reader, 
                       const MParameterObjectPtr &destParam,
                       const std::string &type,
                       SceneInfo &info) 
  {
    if (type == "string" || 
        type == "bool" ||
        type == "int" ||
        type == "float") 
    {
      std::string value = reader.readString();
      if (destParam != NULL) {
        // if our destination parameter is an object type, then
        // we need to translate the name into an object.
        MObjectParameter *objParam = AZTEC_CAST(MObjectParameter, destParam);

        // if we have an actual MObjectParameter, use that instead
        if (objParam != NULL) {
          MNamedObjectPtr objectValue = info.getObject(value.c_str());
          objParam->setValue(objectValue);
        } else {
          // if our destination type is an object type, make sure that
          // we check the stored name with the new name so that we
          // assign it the right value.
          if (destParam->getDataType() == MParameterObject::TYPE_OBJECT) {
            MNamedObjectPtr objectValue = info.getObject(value.c_str());
            if (objectValue != NULL) {
              value = objectValue->getName();              
            }
          }
          destParam->setValueString(value.c_str());
        }
      }
      return true;
    }

    if (type == "vector3") {
      MVector3 vector;
      vector = reader.readVector3();
      if (destParam != NULL) {
        destParam->setValueVector(vector);
      }
      return true;
    }

    if (type == "vector3keylist") {
      return readVector3KeyParam(reader, destParam);
    }

    if (type == "matrix4") {
      MMatrix4 value;

      for (int i = 0; i < 4; ++i) {
        for (int j = 0; j < 4; ++j) {
          value.m[i][j] = reader.readFloat();
        }
      }
      
      MMatrix4ParameterPtr matrixParam = AZTEC_CAST(MMatrix4Parameter, destParam);

      if (matrixParam != NULL) {
        matrixParam->setValueMatrix(value);
      }

    }

    if (type == "array") {
      return readArray(reader, destParam, info);
    }

    if (type == "aggregate") {
      return readAggregate(reader, destParam, info);
    }

    if (type == "[unknown]") {
      return true;
    }

    return false;
  }

  bool readInObject(SceneReader &reader, SceneInfo &sceneInfo) {
    std::string objectType, objectName;
    reader.beginGroup(objectType, objectName);

    if (reader.atEndOfGroup()) {
      return true;
    }

    // make sure that we have read in a valid name
    assert(objectName.length() > 0);

    MNamedObjectPtr namedObject = AZTEC_CAST(MNamedObject, MSystemManager::getInstance()->getPluginManager()->createObjectFromDefault(objectType.c_str()));

    // if we did not create the type successfully, bail out.
    if (namedObject == NULL) {
      assert(0);
      return false;
    }

    namedObject->setName(objectName.c_str());
    sceneInfo.addObject(objectName, namedObject);

    while (!reader.atEndOfGroup()) {
      std::string paramName;
      SceneReader::OperatorType oper;

      paramName = reader.readString();

      if (reader.atEndOfGroup()) {
        break;
      }

      oper = reader.readOperator();

      if (oper == SceneReader::LINKED_FROM) {
        // If we have a linking operation, we have to find the other object
        // and its parameter so we can link to it.

        // get the parameter name.
        std::string srcParameterName = reader.readString();

        // separate the parameter from the object name.
        int dotIndex = srcParameterName.find('.');

        assert(dotIndex != std::string::npos);

        std::string objectName = srcParameterName.substr(0, dotIndex);
        srcParameterName = srcParameterName.substr(dotIndex+1);

        // Get the objects
        MNamedObjectPtr srcObject = sceneInfo.getObject(objectName);
        MParameterObjectPtr sourceParam = srcObject->findParameter(srcParameterName.c_str());
        MParameterObjectPtr destParam = namedObject->findParameter(paramName.c_str());

        // Perform the actual linking of parameters.
        if (destParam != NULL) {
          destParam->setInputParameter(sourceParam);
        }

      } else if (oper == SceneReader::EQUALS) {
        // Here we are assigning a value. First part is to read the first 
        // string. then we compare that string to a list of known types. 
        // If we have a match then we must have a special type, and we read 
        // it in appropriately.
        // If we don't know the type, we have an error.
        std::string value = reader.readString();
        MParameterObjectPtr destParam = namedObject->findParameter(paramName.c_str());

        if (isDataType(value)) {
          readSpecialData(reader, destParam, value, sceneInfo);
        } else if (destParam != NULL) {
          assert(0);
          destParam->setValueString(value.c_str());
        }
      }
    }

    reader.endGroup();

    return true;
  }

  bool readInFile(SceneReader &reader, const MScenePtr &scene) {
    std::string versionName;
    float versionValue;
    
    versionName = reader.readString();
    reader.readString(); // skip the equal sign.
    reader.readString(); // skip the type ID.
    versionValue = reader.readFloat();

    assert(versionName == "Aztec_Scene_File_Version");
    assert(versionValue <= 1.0);

    std::string sceneType, sceneName;

    SceneInfo sceneInfo;

    sceneInfo.scene = scene;

    reader.beginGroup(sceneType, sceneName);
    while (!reader.atEndOfGroup()) {
      std::string objectType, objectName;
      reader.beginGroup(objectType, objectName);
      while (!reader.atEndOfGroup()) {
        if (!readInObject(reader, sceneInfo)) {
          return false;
        }
        
      }

      reader.endGroup();
    }

    reader.endGroup();

    SceneInfo::NameObjectMap::iterator it;

    for (it = sceneInfo.namedObjects.begin(); it != sceneInfo.namedObjects.end(); ++it) {
      it->second->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);

    }

    return true;
  }

  bool MCompleteTranslator::importFile(MStr Filename, MScenePtr scene) {
    MTextFileReader inStream;

    inStream.open(Filename);

    if (!inStream.isOpen()) {
      getSystemManager()->logOutput("Error: Could not open '%s' for reading.", Filename.c_str());
      return false;
    }

    scene->clearScene();

	AsciiSceneReader reader(inStream);
    bool okay = readInFile(reader, scene);    

    inStream.close();
    
    return okay;
  }


  bool MCompleteTranslator::canImportFile(MStr filename) {
    MTextFileReader inStream;

    inStream.open(filename);

    if (!inStream.isOpen()) {
      return false;
    }

    AsciiSceneReader reader(inStream);

    bool canRead = false;

    if (reader.readString() == "Aztec_Scene_File_Version") {
      canRead = true;
    }

    inStream.close();

    return canRead;
  }

}
