#include "StdAfx.h"

#include <MSystemManager.h>
#include <mesh/MVertexArrayParam.h>

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
  MeshVertexPosParam::MeshVertexPosParam(MMesh *mesh, int index, bool storable) {
    this->mesh = mesh;
    this->index = index;
    this->editMeshCheckDone = false;
    this->animMeshCheckDone = false;
    this->editMesh = NULL;
    this->animMesh = NULL;
    setStorable(storable);
  }

  // MVector3KeyParameter methods
  MVector3ValuePtr MeshVertexPosParam::getVector3Value() {
    if (!isAnimMesh()) {
      return NULL;
    }

    return animMesh->getAnimVert(index)->m_Pos;
  }

  void MeshVertexPosParam::createXKeyAt(long time) {
    if (!isAnimMesh()) {
      return;
    }

    animMesh->getAnimVert(index)->m_Pos->setXKey(getValue().x, time);
  }

  void MeshVertexPosParam::createYKeyAt(long time) {
    if (!isAnimMesh()) {
      return;
    }
    animMesh->getAnimVert(index)->m_Pos->setYKey(getValue().y, time);
  }

  void MeshVertexPosParam::createZKeyAt(long time) {
    if (!isAnimMesh()) {
      return;
    }
    animMesh->getAnimVert(index)->m_Pos->setZKey(getValue().z, time);
  }

  MFloatKeyParameterPtr MeshVertexPosParam::getXParameter() {
    if (!isAnimMesh()) {
      return NULL;
    }

    return new MFloatKeyParameterImpl(getShortName() + "x", 
                                      getLongName() + "x",
                                      getFriendlyName() + " X",
                                      animMesh->getAnimVert(index)->m_Pos->getXValue());
  }
  MFloatKeyParameterPtr MeshVertexPosParam::getYParameter() {
    if (!isAnimMesh()) {
      return NULL;
    }
    return new MFloatKeyParameterImpl(getShortName() + "y", 
                                      getLongName() + "y",
                                      getFriendlyName() + " Y",
                                      animMesh->getAnimVert(index)->m_Pos->getYValue());
  }

  MFloatKeyParameterPtr MeshVertexPosParam::getZParameter() {
    if (!isAnimMesh()) {
      return NULL;
    }
    return new MFloatKeyParameterImpl(getShortName() + "z", 
                                      getLongName() + "z",
                                      getFriendlyName() + " Z",
                                      animMesh->getAnimVert(index)->m_Pos->getZValue());
  }


  // MKeyParameter methods
  MKeyableValuePtr MeshVertexPosParam::getKeyableValue() {
    if (!isAnimMesh()) {
      return NULL;
    }

    return animMesh->getAnimVert(index)->m_Pos;
  }

  // MVector3Parameter methods
  MVector3 MeshVertexPosParam::getValue() {
    return mesh->getVertexPosition(index);
  }

  MVector3 MeshVertexPosParam::getValue(long time) {
    return mesh->getVertexPosition(index);
  }

  // MParameterObject methods
  int MeshVertexPosParam::getDataType() const {
    return TYPE_VECTOR;
  }

  int MeshVertexPosParam::getDataMeaning() const {
    return MEANING_POSITION;
  }

  MStr MeshVertexPosParam::getShortName() const {
    char buf[64];
    sprintf(buf, "pnts[%i]", index);
    return buf;
  }

  MStr MeshVertexPosParam::getLongName() const {
    char buf[64];
    sprintf(buf, "points[%i]", index);
    return buf;
  }

  MStr MeshVertexPosParam::getFriendlyName() const {
    char buf[64];
    sprintf(buf, "points[%i]", index);
    return buf;
  }

  MBaseObject* MeshVertexPosParam::getOwner() const {
    return mesh;
  }

  void MeshVertexPosParam::setOwner(MBaseObject *newOwner) {
    assert(0);
  }

  bool MeshVertexPosParam::isInputParameter() {
    return true;
  }

  bool MeshVertexPosParam::getValueString(MStr &value) {
    if (index >= (unsigned int)mesh->getNumVerts()) {
      return false;
    }

    value = mesh->getVertexPosition(index).convertToString();
    return true;
  }

  bool MeshVertexPosParam::getValueInteger(int &value) { return false; }
  bool MeshVertexPosParam::getValueFloat(float &value) { return false; }
  bool MeshVertexPosParam::getValueBoolean(bool &value) { return false; }

  bool MeshVertexPosParam::getValueVector(MVector3 &value) {
    if (index >= (unsigned int)mesh->getNumVerts()) {
      return false;
    }

    value = mesh->getVertexPosition(index);
    return true;
  }

  bool MeshVertexPosParam::setValueString(const MStr &value) { 
    MVector3 vec;
    if (vec.convertFromString(value)) {
      return setValueVector(vec);
    }
    return false; 
  }

  bool MeshVertexPosParam::setValueInteger(int value) { return false; }
  bool MeshVertexPosParam::setValueFloat(float value) { return false; }
  bool MeshVertexPosParam::setValueBoolean(bool value) { return false; }

  bool MeshVertexPosParam::setValueVector(const MVector3 &value) { 
    if (!isEditableMesh()) {
      return false;
    }

    if (index >= (unsigned int)editMesh->getNumVerts()) {
      return false;
    }

    editMesh->setVertexPosition(index, value);

    // if we have an animating mesh, then we need to setup the value for 
    // that as well
    if (isAnimMesh()) {
      // if we are animating, then set the key.
      if (MSystemManager::getInstance()->getSettings()->m_Animate) {
        long time = MSystemManager::getInstance()->getScene()->getTime();
        animMesh->getAnimVert(index)->set(time, value);
      } else {
        // if we are not, then set the initial value instead.
        animMesh->getAnimVert(index)->m_Pos->setInitialValue(value);
      }
    }

    return true; 
  }

  bool MeshVertexPosParam::createKeyAt(long time) { 
    if (!isEditableMesh()) {
      return false;
    }

    editMesh->updateSingleComponentAnimation(MComponentisedObject::POINT_TYPE, index, time, true);
    return true; 
  }

  bool MeshVertexPosParam::setValueParameter(const MParameterObjectPtr &src) {
    MVector3 vec;
    if (src->getValueVector(vec)) {
      return setValueVector(vec);
    }
    
    return false;
  }

  bool MeshVertexPosParam::setFromParameter(MParameterObjectPtr src) {
    return setValueParameter(src);
  }

  bool MeshVertexPosParam::isConnectable() { return false; }
  bool MeshVertexPosParam::isConnected()  { return false; }
  bool MeshVertexPosParam::setInputParameter(const MParameterObjectPtr &input)  { return false; }
  MParameterObjectPtr MeshVertexPosParam::getInputParameter() { return NULL; }

  MParameterObjectPtr MeshVertexPosParam::createCopy() { 
    return MParameterFactory::createVector3(getShortName(), getLongName(), getFriendlyName());
  }

  bool MeshVertexPosParam::isEditableMesh() {
    if (!editMeshCheckDone) {
      editMesh = AZTEC_CAST(MEditableMesh, mesh);
      editMeshCheckDone = true;
    }

    return editMesh != NULL;
  }

  bool MeshVertexPosParam::isAnimMesh() {
    if (!animMeshCheckDone) {
      animMesh = AZTEC_CAST(MAnimMesh, mesh);
      animMeshCheckDone = true;
    }

    return animMesh != NULL;
  }


  MeshVertexArrayParam::MeshVertexArrayParam( MMesh *mesh ) 
    : MParameterObjectImpl("pnts", "points", "points", NULL),
      mesh(mesh),
      editMesh(NULL),
      editMeshCheckDone(false)
  {
  }

  MeshVertexArrayParam::~MeshVertexArrayParam() {
  }

  MBaseObject* MeshVertexArrayParam::getOwner() const {
    return &*mesh->getParamList();
  }

  void MeshVertexArrayParam::setOwner(MBaseObject *newOwner) {
    assert(newOwner == mesh);
  }

  bool MeshVertexArrayParam::isInputParameter() {
    return true;
  }

  int MeshVertexArrayParam::getElementCount() {
    return mesh->getNumVerts();
  }

  int MeshVertexArrayParam::addElement() {
    if (isEditableMesh()) {
      editMesh->addVertex(0,0,0);
      return editMesh->getNumVerts() - 1;
    }
    return -1;
  }

  void MeshVertexArrayParam::removeElement(int index) {
    if (isEditableMesh()) {
      editMesh->deleteVertex(index);
    }
  }

  void MeshVertexArrayParam::resize(int size) {
    if (isEditableMesh()) {
      while (editMesh->getNumVerts() > size) {
        editMesh->deleteVertex(editMesh->getNumVerts()-1);
      }

      // add any vertices that we need to.
      while (editMesh->getNumVerts() < size) {
        editMesh->addVertex(0,0,0);
      }
    }
    
  }

  MParameterObjectPtr MeshVertexArrayParam::getElement(int index) {
    return new MeshVertexPosParam(mesh, index);
  }

  bool MeshVertexArrayParam::getElement(int index, MStr &value) {
    value = mesh->getVertexPosition(index).convertToString();
    return true;
  }

  bool MeshVertexArrayParam::getElement(int index, MVector3 &value) {
    value = mesh->getVertexPosition(index);
    return true;
  }

  bool MeshVertexArrayParam::setElement(int index, const MStr &value) { 
    MVector3 pos;
    if (pos.convertFromString(value)) {
      return setElement(index, pos);
    }
    return false; 
  }

  bool MeshVertexArrayParam::setElement(int index, const MVector3 &value) { 
    if (isEditableMesh()) {
      editMesh->setVertexPosition(index, value);
    }
    return false; 
  }

  int MeshVertexArrayParam::getElementType() const { 
    return TYPE_VECTOR; 
  }

  // MParameterObject methodsfs
  int MeshVertexArrayParam::getDataMeaning() const { 
    return MEANING_ANY; 
  }

  bool MeshVertexArrayParam::createKeyAt(long time) { 
    if (!isEditableMesh()) {
      return false;
    }

    editMesh->updateComponentAnimation(MComponentisedObject::POINT_TYPE, 0, time, true);

    return true; 
  }

  MParameterObjectPtr MeshVertexArrayParam::createCopy() {
    // TODO: return something that can contain a copy
    return NULL;
  }

  bool MeshVertexArrayParam::setValueParameter(const MParameterObjectPtr &src) {
    // this actually has no meaning, so we sort of leave this out for now.
    return false;
  }

  bool MeshVertexArrayParam::isEditableMesh() {
    if (!editMeshCheckDone) {
      editMesh = AZTEC_CAST(MEditableMesh, mesh);
      editMeshCheckDone = true;
    }

    return editMesh != NULL;
  }

}


