#include <MilkshapeAsciiPlugin.h>

#include <stdio.h>
#include <string>
#include <map>

#include <MAnimMesh.h>
#include <MMeshShape.h>
#include <MSceneObject.h>
#include <MStr.h>
#include <MMaterial.h>

// fix up our for loop scoping
#define for if(0);else for


MilkshapeAsciiPlugin::MilkshapeAsciiPlugin() {
}

MilkshapeAsciiPlugin::~MilkshapeAsciiPlugin() {
}

// MSceneTranslator methods
std::string MilkshapeAsciiPlugin::getFilter() {
  return "*.txt";
}

std::string MilkshapeAsciiPlugin::getFilterDescription() {
  return "Milkshape Ascii Files";
}

bool MilkshapeAsciiPlugin::canImport() {
  return true;
}

bool MilkshapeAsciiPlugin::canExport() {
  return true;
}

// this removes any leading and any traling whitespace
void trimWhitespace(std::string &s) {
  unsigned int start = 0, end = 0;

  for (start = 0; start < s.length(); ++start) {
    if (!isspace(s[start]) && s[start] != '"') {
      break;
    }
  }

  for (end = s.length() - 1; end > start; --end) {
    if (!isspace(s[end]) && s[end] != '"') {
      break;
    }
  }

  if (start >= s.length()) {
    s = "";
  } else {
    s = s.substr(start, end - start + 1);
  }
}

bool readLine(FILE *in, std::string &s) {
  char buf[1024];

  if (fgets(buf, 1023, in) == 0) {
    return false;
  }

  s = buf;
  trimWhitespace(s);
  return true;
}

bool MilkshapeAsciiPlugin::canImportFile(MStr Filename) {
  // read the first line of the file, and see if we have a milkshape file
  FILE *f = fopen(Filename.c_str(), "rt");

  if (f == NULL) {
    return false;
  }

  bool okay = false;
  std::string line;

  if (readLine(f, line)) {
    if (line == "// MilkShape 3D ASCII") {
      okay = true;
    }
  }

  fclose(f);

  return okay;
}

bool parseVertexLine(const std::string &line, Aztec::MVector3 *vert, Aztec::MVector3 *uvVert) {
  float x = 0,y = 0,z = 0,u = 0,v = 0;
  if (sscanf(line.c_str(), "%*i %f %f %f %f %f %*i", &x, &y, &z, &u, &v) != 5) {
    return false;
  }

  vert->set(x,-z,y);
  uvVert->set(u,1.0f-v,0);

  return true;
}

bool parseTriangleLine(const std::string &line, Aztec::MTriangle *tri) {
  int vert[3];
  if (sscanf(line.c_str(),"%*i %i %i %i %*i %*i %*i %*i", &vert[0], &vert[1], &vert[2]) != 3) {
    return false;
  }

  tri->setVertex(0, vert[0]);
  tri->setVertex(1, vert[1]);
  tri->setVertex(2, vert[2]);
  return true;
}

class MilkshapeReader {
public:
  bool readMesh(FILE *in, Aztec::MScenePtr scene) {
    std::string line;
    MStr name;
    char buf[1024];
    int matId;

    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MMeshShapePtr meshShape;
    Aztec::MAnimMeshPtr animMesh;
    Aztec::MEditableMeshPtr uvMesh;

    sceneObj = new Aztec::MSceneObject();

    // read in the mesh name and material index
    readLine(in, line);

    sscanf(line.c_str(), "%s %*i %i", buf, &matId);

    line = buf;
    trimWhitespace(line);
    name = line.c_str();

    sceneObj->setName(name);
    objMaterialMap.insert(std::make_pair(sceneObj, matId));

    int numVerts, numTris, numNormals;
    Aztec::MVector3 *verts, *uvwVerts;
    Aztec::MTriangle *tris;

    // get number of vertices
    readLine(in, line);
    sscanf(line.c_str(), "%i", &numVerts);
    
    verts = new Aztec::MVector3[numVerts];
    uvwVerts = new Aztec::MVector3[numVerts];
    
    // read the vertices in
    for(int index = 0; index < numVerts; ++index) {
      readLine(in, line);
      if (!parseVertexLine(line, &verts[index], &uvwVerts[index])) {
        delete[] verts;
        return false;
      }
    }
  
    // read in the number of normals
    readLine(in, line);
    sscanf(line.c_str(), "%i", &numNormals);

    // skip over all the normals.
    for (int index = 0; index < numNormals; ++index) {
      readLine(in, line);
    }

    // read in the number of triangles
    readLine(in, line);
    sscanf(line.c_str(), "%i", &numTris);

    tris = new Aztec::MTriangle[numTris];

    // read the triangles in
    for(int index = 0; index < numTris; ++index) {
      readLine(in, line);
      parseTriangleLine(line, &tris[index]);
    }

    meshShape = new Aztec::MMeshShape();
    animMesh = new Aztec::MAnimMesh();
    uvMesh = new Aztec::MEditableMesh();

    animMesh->addVertsAndTriangles(verts, tris, numVerts, numTris);
    uvMesh->addVertsAndTriangles(uvwVerts, tris, numVerts, numTris);

    animMesh->setName(name + "Mesh");
    meshShape->setName(name + "MeshShape");
    uvMesh->setName(name + "uvMesh");

    meshShape->setMeshObject(animMesh);
    sceneObj->setShapeObject(meshShape);
    animMesh->setTextureMesh(uvMesh);

    scene->addObject(sceneObj);
    scene->addObject(animMesh);
    scene->addObject(meshShape);

    delete[] verts;
    delete[] tris;
    delete[] uvwVerts;

    return true;
  }

  bool readMaterial(FILE *in, Aztec::MScenePtr scene, int materialIndex) {
    std::string line;
    MStr name, filename;
    Aztec::MMaterialPtr material;

    // read the name
    readLine(in, line);
    name = line.c_str();

    material = new Aztec::MMaterial();
    material->setName(name);

    Aztec::MVector3 vec;
    float temp;

    // read in the ambient information
    readLine(in, line);
    sscanf(line.c_str(), "%f %f %f %f", &vec.x, &vec.y, &vec.z, &temp);
    material->findParameter("AmbientColor")->setValueVector(vec);

    // read in the diffuse information
    readLine(in, line);
    sscanf(line.c_str(), "%f %f %f %f", &vec.x, &vec.y, &vec.z, &temp);
    material->findParameter("DiffuseColor")->setValueVector(vec);

    // read in the specular info
    readLine(in, line);
    sscanf(line.c_str(), "%f %f %f %f", &vec.x, &vec.y, &vec.z, &temp);
    material->findParameter("SpecularColor")->setValueVector(vec);
    
    // read in the emmission info
    readLine(in, line);
    sscanf(line.c_str(), "%f %f %f %f", &vec.x, &vec.y, &vec.z, &temp);
    material->findParameter("EmissionColor")->setValueVector(vec);

    // then shininess
    readLine(in, line);
    sscanf(line.c_str(), "%f", &temp);
    material->findParameter("Shininess")->setValueFloat(temp / 3276.8f);
    
    // then opacity
    readLine(in, line);
    sscanf(line.c_str(), "%f", &temp);
    material->findParameter("Opacity")->setValueFloat(temp);
    
    // get line with texture name
    readLine(in, line);
    material->setParamByName("DiffuseMapName", line.c_str());

    // skip alpha texture name
    // want alpha then use 32 bit texture
    readLine(in, line);
    
    // go throguh our object->material id map, and set any materials up.

    std::map<Aztec::MSceneObjectPtr, int>::iterator it;

    for (it = objMaterialMap.begin(); it != objMaterialMap.end(); ++it) {
      if (it->second == materialIndex) {
        Aztec::MSceneObjectPtr sceneObj = it->first;
        sceneObj->setTextureMaterial(material);
        scene->addObject(material);
      }
    }
    
    return true;

  }
  
  // this is the mapping from object to material index.
  std::map<Aztec::MSceneObjectPtr, int> objMaterialMap;
};

bool MilkshapeAsciiPlugin::importFile(MStr Filename, Aztec::MScenePtr Scene) {
  // read the first line of the file, and see if we have a milkshape file
  FILE *f = fopen(Filename.c_str(), "rt");

  if (f == NULL) {
    return false;
  }

  bool okay = true;
  std::string line;
  int numMeshes = 0;

  if (readLine(f, line)) {
    if (line != "// MilkShape 3D ASCII") {
      okay = false;
    }
  }

  MilkshapeReader reader;

  if (okay) {
    // the next 4 lines we do not care about
    for (int i = 0; i < 4; ++i) {
      readLine(f, line);
    }

    // this is the number of meshes line
    readLine(f, line);

    sscanf(line.c_str(), "%*s %i", &numMeshes);

    // read in the meshes
    for (int i = 0; i < numMeshes; ++i) {
      if (!reader.readMesh(f, Scene)) {
        fclose(f);
        return false;
      }
    }

    int numMaterials;

    // skip the empty line
    readLine(f, line);

    // get the number of materials
    readLine(f, line);
    sscanf(line.c_str(), "%*s %i", &numMaterials);
    
    // start reading materials
    for(int index = 0; index < numMaterials; ++index) {
      reader.readMaterial(f, Scene, index);
    }
    
    
  }

  fclose(f);

  return okay;
}

bool MilkshapeAsciiPlugin::exportFile(MStr filename, Aztec::MScenePtr scene) {
  FILE *out = fopen(filename.c_str(), "wt");

  // if we couldn't open the file for writing, just bail out.
  if (out == NULL) {
    return false;
  }

  // otherwise lets write out our file.

  // this is our array of objects we'll be writing
  typedef std::vector< std::pair<Aztec::MSceneObjectPtr, Aztec::MMeshPtr> > ObjectList;
  ObjectList objects;
  std::map<Aztec::MMaterialPtr, int> materials;
  int materialCount = 0;

  // iterate over all our scene objects, and collect all our objects
  // that we'll be writing.

  scene->getObjectList()->beginIteration();
  {
    Aztec::MBaseObjectPtr obj;

    while ( (obj = scene->getObjectList()->getNext()) != NULL) {
      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, obj);

      if (sceneObj == NULL) {
        // see if we have a material instead
        Aztec::MMaterialPtr mat = AZTEC_CAST(Aztec::MMaterial, obj);

        if (mat != NULL) {
          materials.insert(std::make_pair(mat, materialCount));
          ++materialCount;
        }

        continue;
      }

      // make sure we have a shape object to play with
      if (sceneObj->getShapeObject() == NULL) {
        continue;
      }

      // make sure that this object will convert to a mesh okay
      Aztec::MMeshPtr mesh = sceneObj->getShapeObject()->convertToMesh();
      if (mesh == NULL) {
        continue;
      }

      // if we have gotten here, everything is aok.
      objects.push_back( std::make_pair(sceneObj, mesh) );
    }
  }
  scene->getObjectList()->endIteration();

  // now that we have our objects, write out the file.

  fprintf(out, "// MilkShape 3D ASCII\n");
  fprintf(out, "\n");
  fprintf(out, "Frames: 30\n");
  fprintf(out, "Frame: 1\n");
  fprintf(out, "\n");
  fprintf(out, "Meshes: %i\n", objects.size());

  for (unsigned meshNum = 0; meshNum < objects.size(); ++meshNum) {
    Aztec::MSceneObjectPtr object = objects[meshNum].first;
    Aztec::MMeshPtr mesh = objects[meshNum].second;
    Aztec::MMeshPtr uvMesh = mesh->getTextureMesh();
    int materialID = 0;

    // try and find what material we are using
    {
      std::map<Aztec::MMaterialPtr, int>::iterator it;
      it = materials.find(object->getTextureMaterial());
      if (it != materials.end()) {
        materialID = it->second;
      }
    }

    fprintf(out, "\"%s\" 0 %i\n", object->getName().c_str(), materialID);

    // because the format only supports one uv coordinate per vertex, we'll
    // have to use the skin mesh to separate all the vertices
    if (uvMesh == NULL) {
      uvMesh = mesh;
    }

    // set up the mapping from tex vertices to 3d vertices
    // iterate over all the triangles and do them. The mapping
    // should always be n to m where n >= m.
    // TODO: This will not work for two triangles that
    //       share a uv vertex, but are separate on the 3d mesh.
    std::vector<int> vertexMapping;
    vertexMapping.resize(uvMesh->getNumVerts(), -1);
    for (int triIndex = 0; triIndex < uvMesh->getNumTris(); ++triIndex) {
      vertexMapping[uvMesh->getTriangleVertex(triIndex, 0)] = mesh->getTriangleVertex(triIndex, 0);
      vertexMapping[uvMesh->getTriangleVertex(triIndex, 1)] = mesh->getTriangleVertex(triIndex, 1);
      vertexMapping[uvMesh->getTriangleVertex(triIndex, 2)] = mesh->getTriangleVertex(triIndex, 2);
    }
    

    // write out the number of vertices, followed by the vertices
    fprintf(out, "%i\n", uvMesh->getNumVerts());
    for (int n = 0; n < uvMesh->getNumVerts(); ++n) {
      Aztec::MVector3 vert = mesh->getVertexPosition( vertexMapping[n] );
      Aztec::MVector3 uvVert = uvMesh->getVertexPosition( vertexMapping[n] );
      Aztec::MVector3 pos;

      pos = scene->objectToWorldSpace(Aztec::MBaseObjectPtr(object), vert);
      fprintf(out, "1 %.6f %.6f %.6f %.6f %.6f -1\n", 
        pos.x, pos.z, -pos.y, uvVert.x, 1.0 - uvVert.y);
    }

    // write out the normals
    fprintf(out, "%i\n", uvMesh->getNumVerts());
    for (int n = 0; n < uvMesh->getNumVerts(); ++n) {
      Aztec::MVector3 normal = mesh->getVertexNormal( vertexMapping[n] );
      fprintf(out, "%.6f %.6f %.6f\n", 
        normal.x, normal.z, -normal.y);
    }

    // write out the triangles
    // for some reason, we have to write the triangle twice?
    fprintf(out, "%i\n", uvMesh->getNumTris());
    for (int n = 0; n < uvMesh->getNumTris(); ++n) {
      fprintf(out, "1 %i %i %i %i %i %i 1\n",
        uvMesh->getTriangleVertex(n, 0), uvMesh->getTriangleVertex(n, 1), uvMesh->getTriangleVertex(n, 2), 
        uvMesh->getTriangleVertex(n, 0), uvMesh->getTriangleVertex(n, 1), uvMesh->getTriangleVertex(n, 2) );
    }
  }

  // write an empty line
  fprintf(out, "\n");

  // write out our materials
  if (materials.size() == 0) {
    // just write out a dummy material if we have nothing
    fprintf(out, "Materials: 1\n");
    fprintf(out, "\"Material #1\"\n");
    fprintf(out, "0.200000 0.200000 0.200000 1.000000\n"); // ambient
    fprintf(out, "0.800000 0.800000 0.800000 1.000000\n"); // diffuse
    fprintf(out, "0.200000 0.200000 0.200000 1.000000\n"); // specular
    fprintf(out, "0.000000 0.000000 0.000000 1.000000\n"); // emission

    fprintf(out, "0.000000\n"); // shininess
    fprintf(out, "1.000000\n"); // opacity
    fprintf(out, "\"\"\n"); // diffuse filename
    fprintf(out, "\"\"\n"); // alpha filename
  } else {
    fprintf(out, "Materials: %i\n", materials.size());

    std::map<Aztec::MMaterialPtr, int>::iterator it;
    for (it = materials.begin(); it != materials.end(); ++it) {
      Aztec::MMaterialPtr mat = it->first;
      Aztec::MVector3 amb, diff, spec, emm;
      float shin, opac;

      amb = mat->getParamVec("AmbientColor");
      diff = mat->getParamVec("DiffuseColor");
      spec = mat->getParamVec("SpecularColor");
      emm = mat->getParamVec("EmissionColor");
      shin = mat->getParamFloat("Shininess");
      opac = mat->getParamFloat("Opacity");

      fprintf(out, "\"%s\"\n", mat->getName().c_str());
      fprintf(out, "%.6f %.6f %.6f 1.000000\n", amb.x, amb.y, amb.z);
      fprintf(out, "%.6f %.6f %.6f 1.000000\n", diff.x, diff.y, diff.z);
      fprintf(out, "%.6f %.6f %.6f 1.000000\n", spec.x, spec.y, spec.z);
      fprintf(out, "%.6f %.6f %.6f 1.000000\n", emm.x, emm.y, emm.z);
      fprintf(out, "%.6f\n", shin);
      fprintf(out, "%.6f\n", opac);
      fprintf(out, "\"%s\"\n", mat->getParamByName("DiffuseMapName").c_str());
      fprintf(out, "\"\"\n");

    }
  }

  return true;
}

// MBaseObject Methods
MStr MilkshapeAsciiPlugin::getClassName() {
  return MStr("MilkshapeAsciiPlugin");
}

MStr MilkshapeAsciiPlugin::getParentClassName() {
  return MStr("MSceneTranslator");
}

Aztec::MTranslatorPtr MilkshapeAsciiPlugin::createNew() const {
  return new MilkshapeAsciiPlugin();
}

