// MedDLeMath.h
#include "MedDLeTypes.h"
#include "MedDLeMath.h"

CMedDLePoint3D Normalize(CMedDLePoint3D &a)
{
	CMedDLePoint3D p;
	p.x[0]=a.x[0]*a.x[0];
	p.x[1]=a.x[1]*a.x[1];
	p.x[2]=a.x[2]*a.x[2];
	float n;
	n=p.x[0]+p.x[1]+p.x[2];
	n=sqrt(n);

	p=a;
	if(n!=0.0)
	{
		n=1.0f/n;
		p.x[0]*=n;
		p.x[1]*=n;
		p.x[2]*=n;
	}
	else
	{
		p.x[0]=	p.x[1]=	p.x[2]=0.f;
	}

	return p;

}

float DotProduct (const CMedDLePoint3D& v1, const CMedDLePoint3D& v2)
{
	float a;
	a=v1.x[0]*v2.x[0] + v1.x[1] * v2.x[1] + v1.x[2]*v2.x[2];
   return a;
	
}

CMedDLePoint3D CrossProduct (const CMedDLePoint3D& v1, const CMedDLePoint3D& v2)
{
	CMedDLePoint3D result;

	result.x[0] = v1.x[1] * v2.x[2] - v1.x[2] * v2.x[1];
	result.x[1] = v1.x[2] * v2.x[0] - v1.x[0] * v2.x[2];
	result.x[2] = v1.x[0] * v2.x[1] - v1.x[1] * v2.x[0];

	return result;
}

CMedDLePoint3D operator + (const CMedDLePoint3D &p1, const CMedDLePoint3D &p2)
{
	return CMedDLePoint3D(p1.x[0] +p2.x[0], p1.x[1] +p2.x[1],p1.x[2] +p2.x[2] );
}

CMedDLePoint3D operator - (const CMedDLePoint3D &p1, const CMedDLePoint3D &p2)
{
	return CMedDLePoint3D(p1.x[0] - p2.x[0], p1.x[1] - p2.x[1],p1.x[2] - p2.x[2] );
}


CMedDLeMatrix MatrixMultiply(const CMedDLeMatrix &m0,const CMedDLeMatrix &m1)
{
	CMedDLeMatrix c;

	int i, j, k;

	for (i=0; i<4; i++)
	{
		for (j=0; j<4; j++)
		{
            c.m[i][j]=0;
			for (k=0; k<4; k++)
            {
                c.m[i][j]+=m0.m[i][k]*m1.m[k][j];
            }
        }
    }
    return c;
}


CMedDLeMatrix Translate(const float dx, const float dy, const float dz)
{
	CMedDLeMatrix ret = 1.f;
	ret(0,3) = dx;
	ret(1,3) = dy;
	ret(2,3) = dz;
	return ret;
} 

CMedDLeMatrix RotateZ(const float rads)
{
	float cosine = (float)cos(rads);
	float sine = (float)sin(rads);
	CMedDLeMatrix ret = 1.f;
	ret(0,0) = cosine;
	ret(1,1) = cosine;
	ret(0,1) = sine;
	ret(1,0) = -sine;
	return ret;
} // end of RotateX()

CMedDLeMatrix RotateY(const float rads)
{
	float cosine = (float)cos(rads);
	float sine = (float)sin(rads);
	CMedDLeMatrix ret = 1.f;
	ret(0,0) = cosine;
	ret(2,2) = cosine;
	ret(0,2) = -sine;
	ret(2,0) = sine;
	return ret;
} // end of RotateY()

CMedDLeMatrix RotateX(const float rads)
{
	float cosine = (float)cos(rads);
	float sine = (float)sin(rads);
	CMedDLeMatrix ret = 1.f;
	ret(1,1) = cosine;
	ret(2,2) = cosine;
	ret(1,2) = sine;
	ret(2,1) = -sine;
	return ret;
}	// end of RotateZ()

CMedDLeMatrix Scale(const float size)
{
	CMedDLeMatrix ret = 1.f;
	ret(0, 0) = size;
	ret(1, 1) = size;
	ret(2, 2) = size;
	return ret;
}   // end of Scale()


CMedDLeMatrix ViewMatrix(const CMedDLePoint3D from,      // camera location
                            const CMedDLePoint3D at,        // camera look-at target
                            const CMedDLePoint3D world_up,  // worlds up, usually 0, 1, 0
                            const float roll)          // clockwise roll around
                                                       //    viewing direction, 
                                                       //    in radians
{
	CMedDLeMatrix  view = 1.f;
	CMedDLePoint3D  up, right, view_dir;

	view_dir = Normalize(at - from);
	right = CrossProduct(world_up, view_dir);
	up = CrossProduct(view_dir, right);
	right = Normalize(right);
	up = Normalize(up);
	view(0,0) = right.x[0];
	view(0,1) = right.x[1];
	view(0,2) = right.x[2];
	view(1,0) = up.x[0];
	view(1,1) = up.x[1];
	view(1,2) = up.x[2];
	view(2,0) = view_dir.x[0];
	view(2,1) = view_dir.x[1];
	view(2,2) = view_dir.x[2];
	
	view(0,3) = -DotProduct(right, from);
	view(1,3) = -DotProduct(up, from);
	view(2,3) = -DotProduct(view_dir, from);

	if (roll != 0.0f)
	{
		// MatrixMult function shown below
		view = MatrixMultiply(RotateZ(-roll), view); 
	}

	return view;
} // end of ViewMatrix()

CMedDLeMatrix ProjectionMatrix(const float near_plane,     // distance to near clipping plane
                                  const float far_plane,      // distance to far clipping plane
                                  const float fov)            // field of view angle, in radians
{
	float c = (float)cos(fov*0.5);
	float s = (float)sin(fov*0.5);
	float Q = s/(1.0f - near_plane/far_plane);
	CMedDLeMatrix ret = 0.f;
	ret(0, 0) = c;
	ret(1, 1) = c;
	ret(2, 2) = Q;
	ret(3, 2) = -Q*near_plane;
	ret(2, 3) = s;
	return ret;
} // end of ProjectionMatrix()

