//
// this file by brian martin 1996
// brian@phyast.pitt.edu
//
// this is part of the source for the MedDLe quake model viewer/editor
//
//

#include "mdl.h"

MDL_model *model;

int mdl_open=0;
char cur_mdl_name[128];
char cur_path[128];
#define A(n) ((void (*)(args))(n))

typedef struct args{
	int argc;
	char **argv;
};

struct script_lang{

	char *keyword;
	int argc;
	void (*action)(args);

};
/*
ROCKET
GRENADE
GIB
ROTATE
TRACER
ZOMGIB
TRACER2
TRACER3
	*/
void med_load_mdl(args);
void med_save_mdl(args);
void med_get_info(args);
void med_set(args);
void med_add(args);
void med_insert(args);
void med_extract(args);
void med_decompose(args);
void med_new_mdl(args);
void med_path(args);
void med_fname(args);
void med_get2dmap(args);
void med_2dmap(args);
void med_gf(args);
void med_sizeskin(args);
void med_gs(args);
void med_rocket(args);
void med_grenade(args);
void med_gib(args);
void med_rotate(args);
void med_tracer(args);
void med_zombgib(args);
void med_tracer2(args);
void med_tracer3(args);
void med_fliptri(args);
//void med_deletef(args);
//void med_deletes(args);

script_lang sl[] =
{
{"load_mdl",		1,		A(med_load_mdl)},
{"save_mdl", 		1,		A(med_save_mdl)},
{"new_mdl", 		1,		A(med_new_mdl)},
{"get_info", 		1,		A(med_get_info)},
{"add", 				1,		A(med_add)},
{"extract", 		2,		A(med_extract)},
{"decompose",   3,		A(med_decompose)},
{"set", 				2,		A(med_set)},
{"insert", 			2,		A(med_insert)},
{"frame_name", 	2,		A(med_fname)},
{"path",	 			1,		A(med_path)},
{"get_2dmap",		1,		A(med_get2dmap)},
{"2dmap", 	1,		A(med_2dmap)},
{"group_frames", 	3,		A(med_gf)},
{"group_skins", 	3,		A(med_gs)},
{"resize_skin", 	2,		A(med_sizeskin)},
{"rocket_flag", 	0,		A(med_rocket)},
{"grenade_flag", 	0,		A(med_grenade)},
{"gib_flag", 	0,		A(med_gib)},
{"rotate_flag", 	0,		A(med_rotate)},
{"tracer1_flag", 	0,		A(med_tracer)},
{"zombgib_flag", 	0,		A(med_zombgib)},
{"tracer2_flag", 	0,		A(med_tracer2)},
{"tracer3_flag", 	0,		A(med_tracer3)},
{"flip_tri",      1,    A(med_fliptri)},
{NULL, 					0,		NULL }
};

void model_script(char *filename)
{

	FILE *in;
	model=new(MDL_model);
	printf("Running script %s\n",filename);
	in=fopen(filename,"rt");
	if(in==NULL)
	{
		printf("ERROR: can't open script file..\n"); exit(1);
	}

	char str[128];
	cur_path[0]=0;
	// parse med file
	do
	{
		fscanf(in,"%s", str);
		if(feof(in)) goto end_sl;
		if(str[0]=='#')
		{
			char ch;
			while((ch=fgetc(in))!=0x0A) if(ch==EOF) goto end_sl;
		}
		else
		{
			int cur_keyword=-1;
			// match keyword
			for(int i=0; sl[i].keyword!=NULL; i++)
			{
				int j=0;
				while(*(sl[i].keyword+j)!=NULL)
				{
					cur_keyword=i;
					if(str[j]!=*(sl[i].keyword+j)) {cur_keyword=-1;break;}
					j++;
				}
				if(cur_keyword!=-1) break;
			}

			if(cur_keyword==-1) printf("ERROR: confused by \"%s\"\n",str);
			// take action
			else
			{
			// get args
				args a;
				a.argv=new(char *)[sl[cur_keyword].argc];
				for(int i=0; i<sl[cur_keyword].argc; i++)
				{
					fscanf(in,"%s", str);
					int l=1+strlen(str);
					a.argv[i]=new(char)[l];
					strcpy(a.argv[i],str);
				}
				sl[cur_keyword].action(a);
				for(int i=0; i<sl[cur_keyword].argc; i++)
					delete [] a.argv[i];
				delete [] a.argv;
			}
		}
	}while(!feof(in));

	end_sl:
	fclose(in);
	if(mdl_open==1)	clean_mdl(model);

	delete model;
	printf("<end>\n");
}
void med_path(args a)
{
	printf("set path to %s\n",a.argv[0]);
	strcpy(cur_path,a.argv[0]);
}

void med_load_mdl(args a)
{
	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);

	printf("loading %s\n",filename);
	FILE *in=NULL;
	in=fopen(filename,"rb");
	if(in==NULL)
	{
		printf("ERROR: can't open %s\n",filename);
		if(mdl_open==0) exit(1);
	}
	// see if model already open
	if(mdl_open==1) clean_mdl(model);
	strcpy(cur_mdl_name,filename);
	read_mdl(in, model);
	fclose(in);
	mdl_open=1;

}

void med_save_mdl(args a)
{

	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);
	printf("writing mdl file %s\n",filename);
	if(mdl_open!=1){ printf("ERROR: no file open\n"); return;}
	FILE *out;
	out=fopen(filename,"wb");
	if(out==NULL) printf("ERROR: can't open %s\n",filename);
	write_mdl(out, model);
	fclose(out);

}

void med_get_info(args a)
{

	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);

	printf("writing info to %s\n",filename);
	if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}

	FILE *out;
	out=fopen(filename,"wt");
	if(out==NULL) printf("ERROR can't open %s\n",filename);
	fprintf(out,"file name: %s\n\n",cur_mdl_name);
	char *ch;
	ch=model->filetype;
	fprintf(out,"filetype: %c%c%c%c\n",ch[0],ch[1],ch[2],ch[3]);
	fprintf(out,"version: %u\n",model->version);
	fprintf(out,"x scale factor: %f\n",model->xscale);
	fprintf(out,"y scale factor: %f\n",model->yscale);
	fprintf(out,"z scale factor: %f\n",model->zscale);        // scale model
	fprintf(out,"x offset: %f\n",model->xoffset);
	fprintf(out,"y offset: %f\n",model->yoffset);
	fprintf(out,"z offset: %f\n",model->zoffset);     // scale origin
	fprintf(out,"model radius: %f\n",model->radius);                      // bounding radius
	fprintf(out,"eye x position: %f\n",model->eyex);
	fprintf(out,"eye y position: %f\n",model->eyey);
	fprintf(out,"eye z position: %f\n",model->eyez);              // eye position
	fprintf(out,"skins (not including sub skins): %u\n",model->num_skins);                // num skins
	fprintf(out,"skin height: %u\n",model->skinh);                    // skin width
	fprintf(out,"skin width: %u\n",model->skinw);                    // skin height
	fprintf(out,"vertices: %u\n",model->num_vertices);
	fprintf(out,"triangle faces: %u\n",model->num_triangles);
	fprintf(out,"frames (not including sub frames): %u\n",model->num_frames);
	fprintf(out,"sych_type %X\n",model->sync_type);
	fprintf(out,"flags: %X\n",model->flags);
	fprintf(out,"average pixels per triangle: %f\n",model->ave_size);
	fprintf(out,"\nframe names:\n");
	for(int k=0; k<model->num_frames;  k++)
	{
		fprintf(out,"frame %d: ",k+1);

//		fwrite(&model->frame[k].type,4,1,out);
		// if single frames
		if(model->frame[k].type==0)
		{
			fprintf(out,"%s\n",model->frame[k].sframe.name);
			fprintf(out,"bounds: ( %f %f %f ) ( %f %f %f )\n",
				model->xscale*model->frame[k].sframe.bboxmin.v[0]+model->xoffset,
				model->yscale*model->frame[k].sframe.bboxmin.v[1]+model->yoffset,
				model->zscale*model->frame[k].sframe.bboxmin.v[2]+model->zoffset,
				model->xscale*model->frame[k].sframe.bboxmax.v[0]+model->xoffset,
				model->yscale*model->frame[k].sframe.bboxmax.v[1]+model->yoffset,
				model->zscale*model->frame[k].sframe.bboxmax.v[2]+model->zoffset);
		}
		else
		{

			fprintf(out,"grouped frame \n");
//			fwrite(&model->frame[k].gframe.num_gsframes,1,4,out);
//			fwrite(&model->frame[k].gframe.bboxmin,1,4,out);
//			fwrite(&model->frame[k].gframe.bboxmax,1,4,out);
//			for(j=0; j<model->frame[k].gframe.num_gsframes; j++)
//			{
 //				fwrite(&model->frame[k].gframe.gsframe_interval[j],4,1,out);
 //			}

			for(int j=0; j<model->frame[k].gframe.num_gsframes; j++)
			{
				fprintf(out,"  sub frame %d: %s\n",j,model->frame[k].gframe.gsframe[j].name);
				fprintf(out,"  sub frame interval: %g\n",model->frame[k].gframe.gsframe_interval[j]);
				fprintf(out,"subframe bounds: ( %f %f %f ) ( %f %f %f )\n",
				model->xscale*model->frame[k].gframe.gsframe[j].bboxmin.v[0]+model->xoffset,
				model->yscale*model->frame[k].gframe.gsframe[j].bboxmin.v[1]+model->yoffset,
				model->zscale*model->frame[k].gframe.gsframe[j].bboxmin.v[2]+model->zoffset,
				model->xscale*model->frame[k].gframe.gsframe[j].bboxmax.v[0]+model->xoffset,
				model->yscale*model->frame[k].gframe.gsframe[j].bboxmax.v[1]+model->yoffset,
				model->zscale*model->frame[k].gframe.gsframe[j].bboxmax.v[2]+model->zoffset);


			}


		}

	}

	fclose(out);
}

void med_gf(args a)
{
	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}

	int startframe=atoi(a.argv[0]);
	int endframe=atoi(a.argv[1]);
	float frameinterval=atof(a.argv[2]);

	if(endframe > model->num_frames) return;
	// so i have to only change the mdl, not the 3d data

	int k=startframe-1;

	model->frame[k].type=1;
//	printf("%d %d %g\n", startframe,endframe,frameinterval);
	model->frame[k].gframe.num_gsframes=endframe-startframe+1;
	printf("grouping %u frames\n",model->frame[k].gframe.num_gsframes);
	model->frame[k].gframe.gsframe_interval = new (float)[model->frame[k].gframe.num_gsframes];
	model->frame[k].gframe.gsframe = new (MDL_model_frame)[model->frame[k].gframe.num_gsframes];
	printf("setting intervals\n");
	for(int j=0; j<model->frame[k].gframe.num_gsframes; j++)
	{
		model->frame[k].gframe.gsframe_interval[j]=frameinterval*(double)(j+1);
	}
	printf("grouping\n");
	for(int j=0; j<model->frame[k].gframe.num_gsframes; j++)
	{
		memcpy(&model->frame[k].gframe.gsframe[j].bboxmin,&model->frame[k+j].sframe.bboxmin,3);
		memcpy(&model->frame[k].gframe.gsframe[j].bboxmax,&model->frame[k+j].sframe.bboxmax,3);
		memcpy(model->frame[k].gframe.gsframe[j].name,model->frame[k+j].sframe.name,16);

		model->frame[k].gframe.gsframe[j].tv = new (trivertex)[model->num_vertices];
		memcpy(model->frame[k].gframe.gsframe[j].tv,model->frame[k+j].sframe.tv,sizeof(trivertex)*model->num_vertices);
	}

	//find bounding box for group
	memcpy(&model->frame[k].gframe.bboxmin,&model->frame[k].sframe.bboxmin,4);
	memcpy(&model->frame[k].gframe.bboxmax,&model->frame[k].sframe.bboxmax,4);
	for(int j=0; j<model->frame[k].gframe.num_gsframes; j++)
	{
		 for (int i=0; i<3; i++)
		 {
				if( model->frame[k].sframe.bboxmin.v[i] > model->frame[k].gframe.gsframe[j].bboxmin.v[i] )
					model->frame[k].sframe.bboxmin.v[i] = model->frame[k].gframe.gsframe[j].bboxmin.v[i];
				if( model->frame[k].sframe.bboxmax.v[i] < model->frame[k].gframe.gsframe[j].bboxmax.v[i] )
					model->frame[k].sframe.bboxmax.v[i] = model->frame[k].gframe.gsframe[j].bboxmax.v[i];

		 }
	}

	// fix rest of frame numbers
	// copy all frame data
	int kk=startframe;
	for(k=endframe; k<model->num_frames;  k++)
	{
		model->frame[kk].type=model->frame[k].type;
		if(model->frame[k].type==0)
		{
			memcpy(model->frame[kk].sframe.bboxmin.v,model->frame[k].sframe.bboxmin.v,3);
			memcpy(model->frame[kk].sframe.bboxmax.v,model->frame[k].sframe.bboxmax.v,3);
			for(int ii=0; ii<16; ii++) model->frame[kk].sframe.name[ii]=model->frame[k].sframe.name[ii];
//			frame[k].sframe.tv = new (trivertex)[model->num_vertices];
			memcpy(model->frame[kk].sframe.tv,model->frame[k].sframe.tv,sizeof(trivertex)*model->num_vertices);
		}

		else
		{
			 printf("must only group after a group.. know what i mean?\n");
		}
		kk++;
	}

	model->num_frames -= (endframe-startframe);
}


void med_sizeskin(args a)
{
	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}

	int w=atoi(a.argv[0]);
	int h=atoi(a.argv[1]);

		// resize pic if needed and add to model
	printf("resizing skin\n");
	double scalew, scaleh;
	int ow=model->skinw, oh=model->skinh;
	model->skinw=w; model->skinh=h;

	scalew=(double)(ow)/(double)(model->skinw);
	scaleh=(double)(oh)/(double)(model->skinh);

	for (int k=0; k<model->num_vertices; k++)
	{
			double di, dj;
			di=(double)model->vertex[k*3+1];
			dj=(double)model->vertex[k*3+2];
			model->vertex[k*3+1]=(unsigned)(di/scalew);
			model->vertex[k*3+2]=(unsigned)(dj/scaleh);

	}
	unsigned char *oldskin=new (unsigned char)[ow*oh];

	for(int skin=1; skin<=model->num_skins; skin++)
	{
		memcpy(oldskin,model->skin[skin-1].bitmap,ow*oh);
		delete [] model->skin[skin-1].bitmap;
		model->skin[skin-1].bitmap = new (unsigned char)[w*h];

		for(int j=0; j<model->skinh;j++)
		{
			for(int i=0;i<model->skinw;i++)
			{
				unsigned pic_index;
				double di, dj;
				di=(double)i;
				dj=(double)j;
				pic_index=(unsigned)(scalew*di)+(unsigned)(scaleh*dj)*(ow);
				model->skin[skin-1].bitmap[i+j*model->skinw]=oldskin[pic_index];
			}
		}
	}
	delete [] oldskin;

}
void med_gs(args a)
{

	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}

	int startskin=atoi(a.argv[0]);
	int endskin=atoi(a.argv[1]);
	float skininterval=atof(a.argv[2]);

	if(endskin > model->num_skins) return;

	int k=startskin-1;

/*
	{
			model->skin[k].bitmap = new (unsigned char)[model->skinw*model->skinh];
			fread(model->skin[k].bitmap,1,model->skinw*model->skinh,in);

		}
*/

	model->skin[k].type=1;
	model->skin[k].num_gskins=endskin-startskin+1;
	printf("grouping %u skins\n",model->skin[k].num_gskins);

	model->skin[k].gskin_interval = new (float)[model->skin[k].num_gskins];
	model->skin[k].gskin = new (MDL_group_skin)[model->skin[k].num_gskins];
	printf("setting intervals\n");
	for(int j=0; j<model->skin[k].num_gskins; j++)
	{
		model->skin[k].gskin_interval[j]=skininterval*(float)(j+1);

	}
	printf("grouping\n");

	for(int j=0; j<model->skin[k].num_gskins; j++)
	{
				model->skin[k].gskin[j].bitmap = new (unsigned char)[model->skinw*model->skinh];
				memcpy(model->skin[k].gskin[j].bitmap,model->skin[k+j].bitmap,model->skinw*model->skinh);
	}

	int kk=startskin;
	for(k=endskin; k<model->num_skins;  k++)
	{
		model->skin[kk].type=model->skin[k].type;
		if(model->skin[k].type==0)
		{
			memcpy(model->skin[kk].bitmap,model->skin[k].bitmap,model->skinw*model->skinh);
		}

		else
		{
			 printf("must only group after a group.. know what i mean?\n");
		}
		kk++;
	}

	model->num_skins -= (endskin-startskin);
}


void med_get2dmap(args a)
{

	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);

	printf("writing 3d map to %s\n",filename);
	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}

	FILE *out;
	out=fopen(filename,"wt");
	if(out==NULL) printf("ERROR can't open %s\n",filename);

	for(int i=0;i<model->num_vertices;i++)
	{
		fprintf(out,"%d %d %d \n",model->vertex[i*3],model->vertex[i*3+1],model->vertex[i*3+2]);
	}


	fclose(out);
}

void med_2dmap(args a)
{

	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);

	printf("reading 3d map %s\n",filename);
	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}

	FILE *in;
	in=fopen(filename,"rt");
	if(in==NULL) printf("ERROR can't open %s\n",filename);

	char str1[32],str2[32],str3[32];
	for(int i=0;i<model->num_vertices;i++)
	{
		fscanf(in,"%s",str1);
		model->vertex[i*3]=(int)atoi(str1);
		fscanf(in,"%s",str1);
		model->vertex[i*3+1]=(int)atoi(str1);
		fscanf(in,"%s",str1);
		model->vertex[i*3+2]=(int)atoi(str1);
	}

	fclose(in);
}

void med_fname(args a)
{

	int frame=atoi(a.argv[0]);

	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
	if((frame<=0)||(frame>model->num_frames))
	{
		printf("ERROR: can't rename. invalid frame (%d)\n",frame); return;
	}
	printf("renaming frame %d to %s\n",frame,a.argv[1]);

	if(model->frame[frame-1].type==0)
	{
			strncpy(model->frame[frame-1].sframe.name,a.argv[1],16);
			char *ptrname=strpbrk(model->frame[frame-1].sframe.name,"\0");
      ptrname++;
			int iil=strlen(model->frame[frame-1].sframe.name);
      iil++;
			strncpy(ptrname,"MedDLe",16-iil);
	}
	else
	{
		 printf("sorry, can only rename ungrouped frames for now.\n");
	}

}
void med_new_mdl(args a)
{
	if(mdl_open==1) clean_mdl(model);
	mdl_open=1;
	strcpy(cur_mdl_name,cur_path);
	strcat(cur_mdl_name,"untitled.mdl");
	// now make all data!
	// insert first frame, generate triangles, and a blank skin
	// as well as set the data
	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);
	generate_mdl(filename, model);

}
void med_decompose(args a)
{
	args b;
	printf("decomposing %s!\n",cur_mdl_name);
	if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}
	// first argv: med file, skin, frame
	// write med file
	char filename[128];
	strcpy(filename,cur_path);
	strcat(filename,a.argv[0]);
	FILE *out;
	out=fopen(filename,"wt");
	if(out==NULL) printf("ERROR can't open %s\n",filename);
	fprintf(out,"# MedDLe script to make %s\n",cur_mdl_name);

	// extract frames
	fprintf(out,"# add all frames\n");
	setup_3d_data(model);
	for(int i=0; i<model->num_frames; i++)
	{
		char name[128];
		if(model->frame[i].type==0)
		{
			sprintf(name,"%s%s.%s",cur_path,model->frame[i].sframe.name,a.argv[2]);
			printf("extracting %s %d\n",name,i+1);
			export_mdl_frame(name, i+1, model );
			if(i==0) fprintf(out,"new_mdl %s\n",name);
			else fprintf(out,"insert %s %d\n",name,i+1);
		}
		else{printf("ERROR: can only decompose non-grouped frames\n"); }
	}
	delete_3d_data(model);
	// extract skins
	fprintf(out,"# add all skins\n");
	for(int i=0; i<model->num_skins; i++)
	{
		char name[128];
		sprintf(name,"%sskin%d.%s",cur_path,i+1,a.argv[1]);
		printf("extracting %s\n",name);
		export_mdl_skin(name, i+1, model );
		fprintf(out,"insert %s %d\n",name,i+1);
	}
	// 2d point mapping
	fprintf(out,"2dmap ptmap.txt\n");
	b.argc=1;
	b.argv=new(char *)[b.argc];
	b.argv[0]=new(char)[10];
	strcpy(b.argv[0],"ptmap.txt");
	med_get2dmap(b);
	delete [] b.argv[0];
	delete [] b.argv;

	// set some strange things

	fprintf(out,"set flags %u\n",model->flags);
	fprintf(out,"set eyex %f\n",model->eyex);
	fprintf(out,"set eyey %f\n",model->eyey);
	fprintf(out,"set eyez %f\n",model->eyez);
	fprintf(out,"set sync_type %u\n",model->sync_type);


	fprintf(out,"# uncomment out this line to save.\n");
	fprintf(out,"# save_mdl %s\n",cur_mdl_name);


	fprintf(out,"# end MedDLe script \n");
	fclose(out);
	printf("constructed script %s\n",a.argv[0]);
}
void med_set(args a)
{
	// data types
	enum {F,U,S};
	struct mdldata{
		char *name;
		int type;
		void *ptr;
	};
	#define P(n) &model->n

	mdldata dat[]=
	{
	{"xscale",F,P(xscale)},{"yscale",F,P(yscale)},{"zscale",F,P(zscale)},
	{"xoffset",F,P(xoffset)},{"yoffset",F,P(yoffset)},{"zoffset",F,P(zoffset)},
	{"eyex",F,P(eyex)},{"eyey",F,P(eyey)},{"eyez",F,P(eyez)},
	{"version",U,P(version)},{"num_skins",U,P(num_skins)},
	{"num_frames",U,P(num_frames)},{"num_vertices",U,P(num_vertices)},
	{"skin_width",U,P(skinw)},{"skin_height",U,P(skinh)},
	{"sync_type",U,P(sync_type)},{"flags",U,P(flags)},
	{"average_size",F,P(ave_size)},{"radius",F,P(radius)},
	{NULL,0,NULL}
	};

	printf("setting %s to %f\n",a.argv[0],atof(a.argv[1]));
	if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}

	int cur_keyword=-1;
 // match keyword
	for(int i=0; dat[i].name!=NULL; i++)
	{
		int j=0;
		while(dat[i].name[j] !=NULL)
		{
			cur_keyword=i;
			if( *(a.argv[0]+j) != dat[i].name[j]) {cur_keyword=-1;break;}
			j++;
		}
		if(cur_keyword!=-1) break;
	}

	if(cur_keyword==-1) printf("ERROR: confused by \"%s\"\n",a.argv[0]);
	// take action
	else
	{
		// get args
		if(dat[cur_keyword].type==F)
		{
			float value=atof(a.argv[1]);
			memcpy((float *)dat[cur_keyword].ptr,&value,4);
		}
		else if(dat[cur_keyword].type==U)
		{
			int value=atoi(a.argv[1]);
			memcpy((unsigned *)dat[cur_keyword].ptr,&value,4);
		}

	}

}
void med_add(args a)
{

			printf("adding %s\n",a.argv[0]);
			if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}
			quiet=0;
			char *extn;
			extn=strstr(a.argv[0],".");
			char filename[128];
			strcpy(filename,cur_path);
			strcat(filename,a.argv[0]);

			if(!strncmp(extn,".gif",4) || !strncmp(extn,".bmp",4))
			{
				import_mdl_skin(filename, model->num_skins+1, model );
			}

			else if(!strncmp(extn,".raw",4)||!strncmp(extn,".asc",4)||
				!strncmp(extn,".dxf",4)|| !strncmp(extn,".3ds",4)||
				!strncmp(extn,".pov",4)|| !strncmp(extn,".wrl",4)||
				!strncmp(extn,".map",4))
			{
					setup_3d_data(model);
					import_mdl_frame(filename,model->blocks+1, model);
					delete_3d_data(model);

			}
			else
			{
				printf("ERROR: file type not (yet) supported.");
			}
			quiet=1;

}
void med_insert(args a)
{

			int num=atoi(a.argv[1]);
			printf("inserting %s to %d\n",a.argv[0],num);
			if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}
			if(num<=0) {printf("ERROR: can't insert to %d\n",num); return;}
			quiet=0;
			char *extn;
			extn=strstr(a.argv[0],".");
			char filename[128];
			strcpy(filename,cur_path);
			strcat(filename,a.argv[0]);

			if(!strncmp(extn,".gif",4) || !strncmp(extn,".bmp",4))
			{
				if(num>model->num_skins) {printf("WARNING: can't insert, adding \n");}
				import_mdl_skin(filename, num, model );
			}

			else if(!strncmp(extn,".raw",4)||!strncmp(extn,".asc",4)||
				!strncmp(extn,".dxf",4)|| !strncmp(extn,".3ds",4)||
				!strncmp(extn,".pov",4)|| !strncmp(extn,".wrl",4)||
				!strncmp(extn,".map",4))
			{
				if(num>model->blocks) {printf("WARNING: can't insert, adding \n");}
				setup_3d_data(model);
				import_mdl_frame(filename,num, model);
				delete_3d_data(model);
			}
			else
			{
				printf("ERROR: file type not (yet) supported.");
			}
			quiet=1;

}
void med_extract(args a)
{

			int num=atoi(a.argv[1]);
			printf("extracting %s from %d\n",a.argv[0],num);
			if(mdl_open!=1){ printf("ERROR: no mdl file open for info\n"); return;}
			if(num<=0) {printf("ERROR: can't extract %d, too small\n",num); return;}
			char *extn;
			extn=strstr(a.argv[0],".");
			char filename[128];
			strcpy(filename,cur_path);
			strcat(filename,a.argv[0]);

			if(!strncmp(extn,".gif",4) || !strncmp(extn,".bmp",4))
			{
				if(num>model->num_skins) {printf("ERROR: can't extract %d, too big \n",num); return;}
				export_mdl_skin(filename, num, model );
			}

			else if(!strncmp(extn,".raw",4)||!strncmp(extn,".asc",4)||
				!strncmp(extn,".dxf",4)|| !strncmp(extn,".3ds",4)||
				!strncmp(extn,".pov",4)|| !strncmp(extn,".wrl",4)||
				!strncmp(extn,".map",4))
			{
				if(num>model->blocks) {printf("ERROR: can't extract %d, too big\n",num); return;}
				setup_3d_data(model);
				export_mdl_frame(filename,num,model);
				delete_3d_data(model);

			}
			else
			{
				printf("ERROR: file type not (yet) supported.");
			}

}

void med_rotate(args a)  // client side rotation
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)8^model->flags;
			printf("setting rotate flag\n");
}

void med_rocket(args a)      // EF_ROCKET
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)1^model->flags;
			printf("setting rocket flag\n");

}
void med_grenade(args a)  // EF_GRENADE
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)2^model->flags;
			printf("setting grenade flag\n");

}
void med_gib(args a)  //EF_GIB
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			//if(val == 0)
			model->flags=0x00004^model->flags;
			printf("setting gib flag\n");
}
void med_tracer(args a)    // EF_TRACER. w_spike
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)16^model->flags;
			printf("setting tracer1 flag\n");

}
void med_zombgib(args a)    //EF_ZOMGIB
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)32^model->flags;
			printf("setting zombgib flag\n");

}
void med_tracer2(args a)        //EF_TRACER2 ..k_spike
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)64^model->flags;
			printf("setting tracer2 flag\n");

}
void med_tracer3(args a)        // EF_TRACER3.. v_spike
{
			if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}
			model->flags=(unsigned)128^model->flags;
			printf("setting tracer3 flag\n");
}
void med_fliptri(args a)
{
	int val=atoi(a.argv[1]);
	if(mdl_open!=1){ printf("ERROR: no mdl file open\n"); return;}


}
